package com.mapr.db.mapreduce.tools.impl;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.SequenceFile;
import org.apache.hadoop.io.SequenceFile.Reader;
import org.slf4j.LoggerFactory;

import com.mapr.db.impl.IdCodec;
import com.mapr.db.mapreduce.impl.ByteBufWritableComparable;
import com.mapr.db.rowcol.DBDocumentImpl;
import com.mapr.db.rowcol.SequenceFileRowColCodec;

public class FormatResultNonMR  {
  private int numThreads;
  private Path outDir;
  private Path inDir;
  private FileSystem fs;
  private Configuration conf;
  private static final org.slf4j.Logger logger = LoggerFactory.getLogger(FormatResultNonMR.class);


  public FormatResultNonMR (
      Configuration conf,
      Path inDir,
      Path outDir,
      int numThreads) throws IOException {

    this.numThreads = numThreads;
    this.outDir = outDir;
    this.inDir = inDir;
    this.conf = conf;
    this.fs = FileSystem.get(conf);
  }

  private int validateAndCreateDirs() throws Exception {
    /* validate that the outDir doesn't exist and create a new one */
    try {
      if (fs.exists(outDir)) {
        System.err.println("Output dir " + outDir + " already exists");
        return -1;
      }
    } catch (Exception e) {
      System.err.println("Failed to check the status of dir " + outDir);
      throw e;
    }

    try {
      /* create the new dir */
      if (!fs.mkdirs(outDir)) {
        System.err.println("Failed to create dir " + outDir);
        return -1;
      }
    } catch (Exception e) {
      System.err.println("Failed to create dir " + outDir);
      throw e;
    }

    return 0;
  }

  public static String printDiffRow(ByteBuffer row,
      DBDocumentImpl doc) {

    StringBuilder sb = new StringBuilder();
    if (doc != null) {

      String rowKeyString = IdCodec.decode(row).toString();
      String docStringWithTs = doc.getStringWithTs(doc.getJsonPathMap());

      sb.append(
            "\"row\":{ \"_id\":" + rowKeyString
                +", \"value\":{" + docStringWithTs +"}}");
    }
    sb.append("\n");
    return sb.toString();
  }

  public int runWithoutMapReduce() throws Exception {
    int err = 0;

    err = validateAndCreateDirs();
    if (err != 0) {
      return err;
    }

    FileStatus[] files = fs.listStatus(inDir);

    ExecutorService executor = Executors.newFixedThreadPool(numThreads);
    List<FormatResultThread> threads = new ArrayList<FormatResultThread>();

    for (int i = 0; i < files.length; ++i) {
      if (files[i].getLen() == 0) {
        continue;
      }

      FormatResultThread t = new FormatResultThread(files[i]);
      threads.add(t);
      executor.execute(t);
    }

    executor.shutdown();
    /* wait for all the threads to complete */
    while (!executor.isTerminated());

    /* collect the status of each thread */
    for (int i = 0; i < threads.size(); ++i) {
      if (threads.get(i).completed() == false) {
       System.err.println("Thread '" + i +"' didnot finish successfully. Exiting...");
       System.exit(-1);
      }
    }
    System.out.println("Successfully created files in " + outDir);
    return 0;
  }

  /* thread to compare one tablet of table */
  class FormatResultThread implements Runnable {
    FileStatus inFile;
    private boolean completed;

    public FormatResultThread(FileStatus inFile) throws Exception {
      this.inFile = inFile;
    }

    public boolean completed() {
      return completed;
    }


    @Override
    public void run() {
      /* create the out file */
      String outFile = inFile.getPath().getName();
      outFile = outFile + ".txt";

      FSDataOutputStream out = null;
      try {
        out = fs.create(new Path (outDir, outFile));
      } catch (IOException e) {
        System.err.println("Failed to create file " + new Path(outDir, outFile));
        e.printStackTrace();
      }

      SequenceFile.Reader r;
      try {
        r = new SequenceFile.Reader(conf, Reader.file(inFile.getPath()));
        ByteBufWritableComparable key = new ByteBufWritableComparable();
        ByteBufWritableComparable value = new ByteBufWritableComparable();
        while (r.next(key, value)) {
          DBDocumentImpl doc = (DBDocumentImpl) SequenceFileRowColCodec.decode(value.getByteBuf());
          String s = FormatResultNonMR.printDiffRow(key.getByteBuf(), doc);
          out.write(s.getBytes());
        }
        out.close();
        completed = true;
      } catch (IOException e) {
        System.err.println("Failed to read from file " + inFile.getPath());
        e.printStackTrace();
      }
    }
  }
}
