package com.mapr.db.mapreduce.tools.impl;

import java.io.IOException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.ojai.store.QueryCondition;
import org.slf4j.LoggerFactory;

import com.mapr.db.MapRDB;
import com.mapr.db.Table;
import com.mapr.db.TabletInfo;
import com.mapr.db.mapreduce.impl.DiffTableUtils;
import com.mapr.db.rowcol.DBDocumentImpl;

public class DiffTableNonMR  implements FailureTracker{
  private String table1Path;
  private String table2Path;
  private String fields;
  private int numThreads;
  private boolean exitOnFirstMismatch;
  private boolean excludedEmbeddedFamily;
  private Path outDir;
  private FileSystem fs;
  private Configuration conf;
  private Path opsForTable1Dir;
  private Path opsForTable2Dir;
  private boolean shouldExit;
  private static final org.slf4j.Logger logger = LoggerFactory.getLogger(DiffTableNonMR.class);


  public DiffTableNonMR (
      Configuration conf,
      String table1Path,
      String table2Path,
      String fields,
      int numThreads,
      boolean exitOnFirstMismatch,
      boolean excludedEmbeddedFamily,
      String outDir) throws IOException {

    this.table1Path = table1Path;
    this.table2Path = table2Path;
    this.fields = fields;
    this.numThreads = numThreads;
    this.exitOnFirstMismatch = exitOnFirstMismatch;
    this.excludedEmbeddedFamily = excludedEmbeddedFamily;
    this.outDir = new Path(outDir);
    this.conf = conf;
    this.fs = FileSystem.get(conf);
  }

  public int runWithoutMapReduce() throws Exception {
    int err = 0;

    Path[] outPaths = DiffTableUtils.validateAndCreateDirs(fs, outDir);
    if (outPaths == null) {
      err = -1;
      return err;
    }
    opsForTable1Dir = outPaths[0];
    opsForTable2Dir = outPaths[1];

    Table table1 = MapRDB.getTable(table1Path);
    TabletInfo[] tablets = table1.getTabletInfos();

    ExecutorService executor = Executors.newFixedThreadPool(numThreads);
    CompareThread [] threads = new CompareThread[tablets.length];
    for (int i = 0; i < tablets.length; ++i) {
      Path p1 = new Path(opsForTable1Dir, "opsforsrc_" + i + ".diff");
      Path p2 = new Path(opsForTable2Dir, "opsfordst_" + i + ".diff");
      DiffTableCounterCollector c =
          new DiffTableCounterCollector(fs, p1, p2, conf, this);
      threads[i] = new CompareThread(tablets[i].getCondition(), c);
      executor.execute(threads[i]);
    }

    executor.shutdown();
    /* wait for all the threads to complete */
    while (!executor.isTerminated());

    /* collect the status of each thread */
    for (int i = 0; i < threads.length; ++i) {
      if (threads[i].completed() == false) {
       logger.error("Thread '{}' didnot finish successfully. Exiting...", i);
       System.exit(-1);
      }
    }

    /* get the total number of rows processed in each thread and mismatch */
    long numTable1Rows = 0;
    long numTable2Rows = 0;
    long numTable1DiffRows = 0;
    long numTable2DiffRows = 0;
    for (int i = 0; i < threads.length; ++i) {
      numTable1Rows += threads[i].counter.getTable1TotalRows();
      numTable2Rows += threads[i].counter.getTable2TotalRows();
      numTable1DiffRows += threads[i].counter.getTable1DiffRows();
      numTable2DiffRows += threads[i].counter.getTable2DiffRows();
    }

    if ((numTable1DiffRows == 0) &&
        (numTable2DiffRows == 0)) {
      System.out.println("tables '" + table1Path
          + "', and '" + table2Path + "' matched");
      System.out.println("Total number of rows processed " + numTable1Rows);
      return 0;
    }

    System.out.println("tables '" + table1Path
        + "', and '" + table2Path + "' didn't match");

    if (exitOnFirstMismatch) {
      System.out.println("Exiting after finding first mismatch");
    } else {
      System.out.println("Number of rows processed in '" + table1Path + "' : " + numTable1Rows);
      System.out.println("Number of rows processed in '" + table2Path + "' : " + numTable2Rows);
      System.out.println("Mismatch row count in '" + table1Path + "' : "
                          + numTable1DiffRows);
      System.out.println("Mismatch row count in '" + table2Path + "' : "
                          + numTable2DiffRows);
      System.out.println("Rows with mismatch are stored in " + outDir);
    }
    return -1;
  }



  /* thread to compare one tablet of table */
  class CompareThread implements Runnable {
    DocScanner scan1;
    DocScanner scan2;
    String[] fieldList;
    DiffTableCounterCollector counter;
    DiffTableComparator comparator;
    private boolean completed;

    public CompareThread(
        QueryCondition keyCondition,
        DiffTableCounterCollector counter) throws Exception {
      if (fields != null) {
        fieldList = fields.split(",");
      } else {
    	fieldList = null;
      }
      scan1 = new DocScanner(table1Path, keyCondition, fieldList, excludedEmbeddedFamily);
      scan2 = new DocScanner(table2Path, keyCondition, fieldList, excludedEmbeddedFamily);
      this.counter = counter;
      comparator = new DiffTableComparator(table1Path, table2Path, fields, excludedEmbeddedFamily, counter);
    }

    public boolean completed() {
      return completed;
    }

    @Override
    public void run() {
      DBDocumentImpl doc1 = scan1.getNext();
      boolean foundMismatch = false;
      try {
        while (doc1 != null) {
          counter.incTable1Rows();
          foundMismatch = comparator.processNextRow(doc1, scan2);
          if (foundMismatch && exitOnFirstMismatch) {
            comparator.counter.close();
            completed = true;
            return;
          }
          doc1 = scan1.getNext();
        }

        // Process remaining rows in scan2
        foundMismatch = processRemainingRows(scan2);
        scan1.close();
        scan2.close();
        comparator.counter.close();
        completed = true;
      } catch (IOException e) {
        e.printStackTrace();
        System.exit(-1);
      }
    }

    private boolean processRemainingRows(
        com.mapr.db.mapreduce.tools.impl.DocScanner scan) throws IOException {
      boolean foundMismatch = false;
      DBDocumentImpl doc = scan.getNext();
      while (doc != null) {
    	foundMismatch = true;
        counter.incTable2Rows();
        counter.incTable2RowsMismatch(doc);
        if (exitOnFirstMismatch) {
          return foundMismatch;
        }
        doc = scan.getNext();
      }
      return foundMismatch;
    }
  }

  @Override
  public void notifyMismatch() {
    if (exitOnFirstMismatch) {
      shouldExit = true;
    }
  }

  @Override
  public boolean shouldExit() {
    return shouldExit;
  }
}
