package com.mapr.db.mapreduce.tools.impl;

import java.io.IOException;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.SequenceFile;
import org.apache.hadoop.io.SequenceFile.CompressionType;
import org.apache.hadoop.io.SequenceFile.Writer.Option;

import com.mapr.db.impl.IdCodec;
import com.mapr.db.mapreduce.impl.ByteBufWritableComparable;
import com.mapr.db.rowcol.DBDocumentImpl;
import com.mapr.db.rowcol.SequenceFileRowColCodec;

/*
 * Object to be called by the comparator to increment
 * the counters for total number of rows and numrows with
 * mismatch
 */
public class DiffTableCounterCollector {
  long numTable1Rows;
  long numTable2Rows;
  long numTable1DiffRows;
  long numTable2DiffRows;
  FileSystem fs;
  Path opsForTable1Path;
  Path opsForTable2Path;
  boolean shouldExit;
  private Configuration conf;
  SequenceFile.Writer opsForTable1File;
  SequenceFile.Writer opsForTable2File;
  private FailureTracker tracker;

  public DiffTableCounterCollector(
      FileSystem fs,
      Path opsForTable1Path,
      Path opsForTable2Path,
      Configuration conf,
      FailureTracker tracker) {

    this.fs = fs;
    this.opsForTable1Path = opsForTable1Path;
    this.opsForTable2Path = opsForTable2Path;
    this.conf = conf;
    this.tracker = tracker;
  }

  public long getTable1TotalRows() {
    return numTable1Rows;
  }

  public long getTable2TotalRows() {
    return numTable2Rows;
  }

  public long getTable1DiffRows() {
    return numTable1DiffRows;
  }

  public long getTable2DiffRows() {
    return numTable2DiffRows;
  }

  /* called for each row read from table1 */
  public boolean incTable1Rows() {
    ++numTable1Rows;
    return tracker.shouldExit();
  }

  /* called for each row read from table2 */
  public boolean incTable2Rows() {
    ++numTable2Rows;
    return tracker.shouldExit();
  }

  public boolean shouldExit() {
    return tracker.shouldExit();
  }

  /*
   * called when table1 has difference with table2 row
   * table2 row may not exist or is different from table1 row
   * passed doc is table1 row
   * return value - true if the comparison should exit
   */
  boolean incTable1RowsMismatch(DBDocumentImpl doc) throws IOException {
    ++numTable1DiffRows;
    tracker.notifyMismatch();

    /* if the seq file is not created yet then create one */
    if (opsForTable2File == null) {
      Option path = SequenceFile.Writer.file(opsForTable2Path);
      Option keyClass = SequenceFile.Writer.keyClass(ByteBufWritableComparable.class);
      Option valueClass = SequenceFile.Writer.valueClass(ByteBufWritableComparable.class);
      Option compression = SequenceFile.Writer.compression(CompressionType.BLOCK);
      opsForTable2File = SequenceFile.createWriter(conf, path, keyClass, valueClass, compression);
    }
    ByteBufWritableComparable key
    = new ByteBufWritableComparable(IdCodec.encode(doc.getId()));
    ByteBufWritableComparable value =
        new ByteBufWritableComparable(SequenceFileRowColCodec.encode(doc));
    opsForTable2File.append(key, value);
    return tracker.shouldExit();
  }

  boolean incTable2RowsMismatch(DBDocumentImpl doc) throws IOException {
    ++numTable2DiffRows;
    tracker.notifyMismatch();

    /* if the seq file is not created yet then create one */
    if (opsForTable1File == null) {
      Option path = SequenceFile.Writer.file(opsForTable1Path);
      Option keyClass = SequenceFile.Writer.keyClass(ByteBufWritableComparable.class);
      Option valueClass = SequenceFile.Writer.valueClass(ByteBufWritableComparable.class);
      Option compression = SequenceFile.Writer.compression(CompressionType.BLOCK);
      opsForTable1File = SequenceFile.createWriter(conf, path, keyClass, valueClass, compression);
    }
    ByteBufWritableComparable key
    = new ByteBufWritableComparable(IdCodec.encode(doc.getId()));
    ByteBufWritableComparable value =
        new ByteBufWritableComparable(SequenceFileRowColCodec.encode(doc));
    opsForTable1File.append(key, value);
    return tracker.shouldExit();
  }

  // Called at the end driver
  public void close() throws IOException {
    if (opsForTable1File != null) {
      opsForTable1File.close();
      opsForTable1File = null;
    }
    if (opsForTable2File != null) {
      opsForTable2File.close();
      opsForTable2File = null;
    }
  }

  public void incTable1NumDiffRows() {
    ++numTable1DiffRows;
  }

  public void incTable2NumDiffRows() {
    ++numTable2DiffRows;
  }
}
