package com.mapr.db.mapreduce.tools.impl;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.mortbay.log.Log;
import org.ojai.FieldPath;
import org.ojai.Value;

import com.mapr.db.Admin;
import com.mapr.db.FamilyDescriptor;
import com.mapr.db.MapRDB;
import com.mapr.db.TableDescriptor;
import com.mapr.db.impl.Constants;
import com.mapr.db.impl.FamilyDescriptorImpl;
import com.mapr.db.impl.IdCodec;
import com.mapr.db.impl.MapRDBTableImplHelper;
import com.mapr.db.mapreduce.impl.ByteBufWritableComparable;
import com.mapr.db.rowcol.DBDocumentImpl;
import com.mapr.db.rowcol.KeyValueWithTS;
import com.mapr.db.rowcol.RowcolCodec;
import com.mapr.db.rowcol.SequenceFileRowColCodec;
import com.mapr.db.rowcol.SerializationAction;
import com.mapr.db.rowcol.SerializedFamilyInfo;
import com.mapr.db.rowcol.TimeDescriptor;

public class DiffTableComparator {

  String table1Path;
  String table2Path;
  String[] projectedFields;
  public DiffTableCounterCollector counter;

  Map<Integer, FieldPath> table1IdToPathMap;
  Map<Integer, FieldPath> table2IdToPathMap;
  Map<FieldPath, Integer> table1PathToIdMap;
  Map<FieldPath, Integer> table2PathToIdMap;
  Map<Integer, List<String>> table1projectionMap;
  Map<Integer, List<String>> table2projectionMap;


  public DiffTableComparator(
      String table1Path,
      String table2Path,
      String projectedFields,
      boolean excludeEmbeddedFamilies,
      DiffTableCounterCollector c) throws IOException {

    this.table1Path = table1Path;
    this.table2Path = table2Path;
    this.projectedFields = projectedFields != null ? projectedFields.split(",") : null;
    counter = c;

    table1IdToPathMap = new HashMap <Integer, FieldPath>();
    table2IdToPathMap = new HashMap <Integer, FieldPath>();
    table1PathToIdMap = new HashMap <FieldPath, Integer>();
    table2PathToIdMap = new HashMap <FieldPath, Integer>();
    getIdToFieldPathMap(table1Path, table1IdToPathMap, table1PathToIdMap);
    getIdToFieldPathMap(table2Path, table2IdToPathMap, table2PathToIdMap);
    if (this.projectedFields != null) {
      //Both tables must have requested CFs - any mismatch of CF info must be caught
      //at in DiffTablesMeta check
      table1projectionMap = MapRDBTableImplHelper.getMultipleCFQualifiers(table1PathToIdMap,
                                                                      excludeEmbeddedFamilies,
                                                                      this.projectedFields);
      table2projectionMap = MapRDBTableImplHelper.getMultipleCFQualifiers(table2PathToIdMap,
                                          							  excludeEmbeddedFamilies,
                                                                      this.projectedFields);
    } else {
      table1projectionMap = null;
      table2projectionMap = null;
    }

    Log.debug("DifftableComparator table1 {}, id to path map {}",
               table1Path, table1IdToPathMap);
    Log.debug("DifftableComparator table2 {}, id to path map {}",
               table2Path, table2IdToPathMap);

  }

  public Map <Integer, FieldPath> getTable1IdToFieldPathMap() {
    return table1IdToPathMap;
  }

  public Map <Integer, FieldPath> getTable2IdToFieldPathMap() {
    return table2IdToPathMap;
  }

  public Map <FieldPath, Integer> getTable1PathToIdPathMap() {
    return table1PathToIdMap;
  }

  public Map <FieldPath, Integer> getTable2PathToIdPathMap() {
    return table2PathToIdMap;
  }


  // Fills idToPathMap and pathToId map for given table
  public static void getIdToFieldPathMap(
      String table,
      Map<Integer, FieldPath> idToPathMap,
      Map<FieldPath, Integer> pathToIdMap) throws IOException {

    Admin admin = MapRDB.newAdmin();
    TableDescriptor d = admin.getTableDescriptor(table);
    List <FamilyDescriptor>families = d.getFamilies();
    for (FamilyDescriptor family : families) {
      Integer familyId = ((FamilyDescriptorImpl)family).getId();
      idToPathMap.put(familyId, family.getJsonFieldPath());
      pathToIdMap.put(family.getJsonFieldPath(), familyId);
    }
  }

  private RowDiff createRowDiff(Value k, DBDocumentImpl doc1, DBDocumentImpl doc2) {
    ByteBufWritableComparable key = new ByteBufWritableComparable(IdCodec.encode(k));
    ByteBufWritableComparable value1 = null;
    ByteBufWritableComparable value2 = null;
    if (doc1 != null) {
      value1 = new ByteBufWritableComparable(SequenceFileRowColCodec.encode(doc1));
    }
    if (doc2 != null) {
      value2 = new ByteBufWritableComparable(SequenceFileRowColCodec.encode(doc2));
    }
    RowDiff d = new RowDiff();
    d.key = key;
    d.forSrc = value1;
    d.forDst = value2;

    return d;
  }

  public ArrayList<RowDiff> processNextRowAndReturnDiff(DBDocumentImpl doc1,
                           DocScanner scan2) throws IOException {

    ArrayList<RowDiff> diffs = new ArrayList<RowDiff>();
    DBDocumentImpl doc2 = scan2.peekNext();
    boolean shouldExit = false;
    if (doc2 == null) {
      counter.incTable1NumDiffRows();
      diffs.add(createRowDiff(doc1.getId(), null, doc1));
      return diffs;
    }

    ByteBuffer k1 = IdCodec.encode(doc1.getId());
    ByteBuffer k2 = IdCodec.encode(doc2.getId());

    boolean recordMatched = false;
    boolean doMatch = true;

    while (doMatch) {
      shouldExit = counter.shouldExit();
      if (shouldExit) {
        return diffs;
      }
      int res = k1.compareTo(k2);
      if (res == 0) {
        recordMatched = matchDocs(doc1, doc2);
        if (!recordMatched) {
          /* create diffs for both tables and add to array list */
          counter.incTable1NumDiffRows();
          counter.incTable2NumDiffRows();
          diffs.add(createRowDiff(doc1.getId(), doc2, doc1));
        }
        scan2.consume();
        counter.incTable2Rows();
        return diffs;
      } else if (res < 0) {
        //add diff for table 2 and add to diff.
        counter.incTable1NumDiffRows();
        diffs.add(createRowDiff(doc1.getId(), null, doc1));
        recordMatched = false;
        return diffs;
      } else {
        recordMatched = false;
        doMatch = true;
        scan2.consume();
        counter.incTable2Rows();
        counter.incTable2NumDiffRows();
        diffs.add(createRowDiff(doc2.getId(), doc2, null));
        doc2 = scan2.peekNext();
        if (doc2 == null) {
          doMatch = false;
        }
      }
    }
    if (!recordMatched) {
      counter.incTable1NumDiffRows();
      diffs.add(createRowDiff(doc1.getId(), null, doc1));
    }

    return diffs;
  }

  /*
   * Process next document (doc1) and corresponding document
   * from scanner. If the next document in scanner is with
   * higher rowkey than the doc1 then scanner is not advanced
   * return - returns true if there is a mismatch
   */
  public boolean processNextRow(DBDocumentImpl doc1, DocScanner scan) throws IOException {

    DBDocumentImpl doc2 = scan.peekNext();
    boolean shouldExit = false;
    if (doc2 == null) {
      counter.incTable1RowsMismatch(doc1);
      return true;
    }

    ByteBuffer k1 = IdCodec.encode(doc1.getId());
    ByteBuffer k2 = IdCodec.encode(doc2.getId());
    boolean recordMatched = false;
    boolean doMatch = true;

    while (doMatch) {
      shouldExit = counter.shouldExit();
      if (shouldExit) {
        return true;
      }
      int res = k1.compareTo(k2);
      if (res == 0) {
        scan.consume();
        counter.incTable2Rows();
        recordMatched = matchDocs(doc1, doc2);
        if (!recordMatched) {
          counter.incTable1RowsMismatch(doc1);
          counter.incTable2RowsMismatch(doc2);
        }
        return !recordMatched;
      } else if (res < 0) {
        counter.incTable1RowsMismatch(doc1);
        recordMatched = false;
        return true;
      } else {
        /* src document key is more than the dst doc
         * so skip the record from scanner and continue
         * to the next record
         */
        recordMatched = false;
        doMatch = true;
        scan.consume();
        counter.incTable2Rows();
        doMatch = !counter.incTable2RowsMismatch(doc2);
        doc2 = scan.peekNext();
        if (doc2 == null) {
          doMatch = false;
        }
      }
    }

    if (!recordMatched) {
      counter.incTable1RowsMismatch(doc1);
    }
    return !recordMatched;
  }

  private boolean matchDocs(DBDocumentImpl doc1, DBDocumentImpl doc2) throws IOException {

    SerializedFamilyInfo[] doc1Data =
        RowcolCodec.encode(doc1, table1PathToIdMap, false /*isBulkLoad*/, true /*useEncoded*/);

    SerializedFamilyInfo[] doc2Data =
        RowcolCodec.encode(doc2, table2PathToIdMap, false /*isBulkLoad*/, true /*useEncoded*/);

    // Sort based on the column family names so that all matching ones come together
    Arrays.sort(doc1Data, new DBDocComparator(table1IdToPathMap));
    Arrays.sort(doc2Data, new DBDocComparator(table2IdToPathMap));

    int index1 = 0;
    int index2 = 0;

    while ((index1 < doc1Data.length) &&
           (index2 < doc2Data.length)) {

      SerializedFamilyInfo i1 = null;
      SerializedFamilyInfo i2 = null;

      if (index1 < doc1Data.length) {
        i1 = doc1Data[index1];
        if(table1projectionMap != null &&
           !table1projectionMap.containsKey(i1.getFamilyId())) {
          index1++;
          continue;
        }
      }
      if (index2 < doc2Data.length) {
        i2 = doc2Data[index2];
        if(table2projectionMap != null &&
           !table2projectionMap.containsKey(i2.getFamilyId())) {
          index2++;
          continue;
        }
      }

      // There is additional family in one table and its with valid data
      if (i1 == null) {
        if (i2.getAction() != SerializationAction.NO_ACTION) {
          return false;
        }
        index1++;
        index2++;
        continue;
      }

      if (i2 == null) {
        if (i1.getAction() != SerializationAction.NO_ACTION) {
          return false;
        }
        index1++;
        index2++;
        continue;
      }

      // If the familyName is not matching then process the smaller family
      FieldPath family1 = table1IdToPathMap.get(i1.getFamilyId());
      FieldPath family2 = table2IdToPathMap.get(i2.getFamilyId());
      int res = family1.compareTo(family2);
      if (res == 0) {
        DBDocumentImpl familyDoc1 = null;
        DBDocumentImpl familyDoc2 = null;

        if (i1.getAction() != SerializationAction.NO_ACTION) {
          familyDoc1 = (DBDocumentImpl) RowcolCodec.decode(i1.getByteBuffer(),
                    null, true /*excludeId*/, true /*decodeTimestamp*/, true /*preserveDeleteFlags*/);
        }
        if (i2.getAction() != SerializationAction.NO_ACTION) {
          familyDoc2 = (DBDocumentImpl) RowcolCodec.decode(i2.getByteBuffer(), null,
                    true /*excludeId*/, true /*decodeTimestamp*/, true /*preserveDeleteFlags*/);
        }
        index1++;
        index2++;

        // if both the families have no action then they are same
        if (familyDoc1 == null && familyDoc2 == null) {
          continue;
        }


        if ((familyDoc1 == null) && (familyDoc2 != null)) {
          if (TimeDescriptor.isCreateTimeValid(familyDoc2)) {
            return false;
          } else {
            continue;
          }
        }

        if ((familyDoc1 != null) && (familyDoc2 == null)) {
          if (TimeDescriptor.isCreateTimeValid(familyDoc1)) {
            return false;
          } else {
            continue;
          }
        }

        boolean matched = false;

        //No projection or entire family is projected
        if (table1projectionMap == null ||
          table1projectionMap.get(i1.getFamilyId()).isEmpty()) {
          // Match both the documents including their timestamps

          matched = KeyValueWithTS.equals(familyDoc1, familyDoc2,
                                true /*matchTimestamp*/, false /*keepOnlyMismatch*/,
                                null /*parentDelTime1*/, null /*parentDelTime2*/);
        } else {
          //List of projected fieldpaths for each CF on either table are identical
          List<String> projectedQuals1 = table1projectionMap.get(i1.getFamilyId());
          List<String> projectedQuals2 = table2projectionMap.get(i2.getFamilyId());

          if (projectedQuals1.size() != projectedQuals2.size()) {
          String err = "Mismatch in number of projected fieldpaths for tables " + table1Path + " and " + table2Path;
          throw new IOException(err);
          }

          for (String fp : projectedQuals1) {
          if (!projectedQuals2.contains(fp)) {
            String err = "Projection fieldpath mismatch for " + fp + " in Column Family with ID" + i2.getFamilyId() +
                     " in table " + table2Path;
              throw new IOException(err);
          }

          matched = KeyValueWithTS.equals(familyDoc1.getKeyValue(fp), familyDoc2.getKeyValue(fp),
                                      true /*matchTimestamp*/, false /*keepOnlyMismatch*/,
                                      null /*parentDelTime1*/, null /*parentDelTime2*/);
          if (!matched) {
            return false;
          }
          }
        }

        if (!matched) {
          return false;
        }
      } else if (res < 0) {
        // Column family with the matching name is not there in the destination
        // table. Just see if the source table is not null data then we have diff
        if (i1.getAction() != SerializationAction.NO_ACTION) {
          return false;
        }
        index1++;
      } else {
        if (i2.getAction() != SerializationAction.NO_ACTION) {
          return false;
        }
        index2++;
      }
    }

    return true;
  }

  class DBDocComparator implements Comparator <SerializedFamilyInfo> {
    private Map<Integer, FieldPath> pathMap;
    DBDocComparator(Map<Integer, FieldPath> pathMap) {
      this.pathMap = pathMap;
    }

    @Override
    public int compare(SerializedFamilyInfo arg0, SerializedFamilyInfo arg1) {
      return pathMap.get(arg0.getFamilyId()).compareTo(pathMap.get(arg1.getFamilyId()));
    }
  }


}
