package com.mapr.db.mapreduce.tools;

import java.io.IOException;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.NullWritable;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.mapreduce.Mapper;
import org.apache.hadoop.mapreduce.lib.input.SequenceFileInputFormat;
import org.apache.hadoop.mapreduce.lib.output.FileOutputFormat;
import org.apache.hadoop.mapreduce.lib.output.TextOutputFormat;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.db.mapreduce.DBDocumentSerialization;
import com.mapr.db.mapreduce.impl.ByteBufWritableComparable;
import com.mapr.db.mapreduce.tools.impl.FormatResultNonMR;
import com.mapr.db.rowcol.DBDocumentImpl;
import com.mapr.db.rowcol.SequenceFileRowColCodec;



public class FormatResult extends Configured implements Tool{
  private static final Logger LOG = LoggerFactory.getLogger(FormatResult.class);
  private static final String NAME = "FormatResult";

  static boolean mapreduce = false;
  static int numThreads = 0;
  static int defaultThreads = 16;
  static Path inDir;
  static Path outDir;
  static boolean exitOnFirstDiff;

  public static class FormatConverterMapper extends
     Mapper<ByteBufWritableComparable, ByteBufWritableComparable, NullWritable, Text> {

    @Override
    public void map(ByteBufWritableComparable key, ByteBufWritableComparable value, Context context)
      throws IOException, InterruptedException {
        DBDocumentImpl d = (DBDocumentImpl)SequenceFileRowColCodec.decode(value.getByteBuf());
        context.write(NullWritable.get(), new Text(FormatResultNonMR.printDiffRow(key.getByteBuf(), d)));
    }
  }

  public int runFormatResultMR() throws Exception {
    LOG.info("Running FMR Result MAP-Reduce version");
    Configuration conf = this.getConf();

    conf.setStrings("io.serializations", conf.get("io.serializations"),
        DBDocumentSerialization.class.getName());

    Job job = Job.getInstance(conf, NAME);
    job.setJarByClass(FormatResult.class);
    job.setJobName("Convert Sequence File and Output as Text");

    SequenceFileInputFormat.setInputPaths(job, inDir);
    FileOutputFormat.setOutputPath(job, outDir);

    job.setInputFormatClass(SequenceFileInputFormat.class);
    job.setMapperClass(FormatConverterMapper.class);

    job.setOutputKeyClass(NullWritable.class);
    job.setOutputValueClass(Text.class);
    job.setNumReduceTasks(0);
    job.setOutputFormatClass(TextOutputFormat.class);
    job.setSpeculativeExecution(false);

    boolean isJobSuccessful = job.waitForCompletion(true);

    return isJobSuccessful ? 0 : 1;
  }

  @Override
  public int run(String[] args) throws Exception {
    Configuration conf = getConf();

    FileSystem fs = FileSystem.get(conf);
    if (!fs.exists(inDir)) {
      System.err.println("Input dir " + inDir + " doesn't exist");
      System.exit(-1);;
    }

    if (!fs.isDirectory(inDir)) {
      System.err.println("Input path " + inDir + " is not a directory");
      System.exit(-1);;
    }

    if (fs.exists(outDir)) {
      System.err.println("Output dir " + outDir + " already exist");
      System.exit(-1);;
    }


    if (!mapreduce) {
      FormatResultNonMR d =
          new FormatResultNonMR(conf, inDir, outDir, numThreads);
      return d.runWithoutMapReduce();
    } else {
      return runFormatResultMR();
    }

  }


  private static void parseArgs(String args[]) throws Exception {
    for (int i = 0; i < args.length; ++i) {
      if (args[i].equalsIgnoreCase("-h")) {
        Usage(null);
      } else if (args[i].equalsIgnoreCase("-indir")) {
        inDir = new Path(args[++i]);
      } else if (args[i].equalsIgnoreCase("-outdir")) {
        outDir = new Path(args[++i]);
      } else if (args[i].equalsIgnoreCase("-numthreads")) {
        numThreads = Integer.parseInt(args[++i]);
      } else if (args[i].equalsIgnoreCase("-mapreduce")) {
        mapreduce = Boolean.parseBoolean(args[++i]);
      } else {
        Usage(null);
      }
    }

    if (inDir == null) {
      Usage("Missing -indir");
    }

    if (outDir == null) {
      Usage("Missing -outdir");
    }

    if ((mapreduce) && (numThreads != 0)) {
      LOG.warn("numThreads should not be specified with mapreduce=true.");
    }

    if ((!mapreduce) && (numThreads == 0)) {
      numThreads = defaultThreads;
    }


   }

  public static void Usage(String errorMsg) {
    if (errorMsg != null && errorMsg.length() > 0) {
        System.err.println("ERROR: " + errorMsg);
    }

    System.err.println("Usage: FormatResult \n"
                     + "-indir <dir path containing files with doucments>\n"
                     + "-outdir <dir path where files with decoded documents will be created>\n"
                     + "[-mapreduce <false/true> (default:false)]\n"
                     + "[-numthreads <numThreads> (default:16) only valid if mapreduce = false]\n");
    System.exit(1);
  }

  public static void main(String[] args) throws Exception {
    Configuration conf = new Configuration();
    parseArgs(args);

    int ret = 0;
    try {
      ret = ToolRunner.run(conf, new FormatResult(), args);
    } catch (Exception e) {
      ret = 1;
      e.printStackTrace();
    }
    System.exit(ret);
  }
}
