package com.mapr.db.mapreduce.tools;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.SequenceFile;
import org.apache.hadoop.mapred.FileAlreadyExistsException;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.mapreduce.Mapper;
import org.apache.hadoop.mapreduce.lib.output.FileOutputFormat;
import org.apache.hadoop.mapreduce.lib.output.SequenceFileOutputFormat;
import org.apache.hadoop.util.GenericOptionsParser;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.ojai.Document;
import org.ojai.DocumentStream;
import org.ojai.Value;
import org.ojai.store.QueryCondition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.db.MapRDB;
import com.mapr.db.Table;
import com.mapr.db.TabletInfo;
import com.mapr.db.impl.IdCodec;
import com.mapr.db.impl.MapRDBTableImpl;
import com.mapr.db.impl.MapRDBTableImpl.TablePrivateOption;
import com.mapr.db.mapreduce.TableInputFormat;
import com.mapr.db.mapreduce.impl.ByteBufWritableComparable;
import com.mapr.db.mapreduce.impl.DocEmptySerialization;
import com.mapr.db.mapreduce.impl.MapReduceUtilMethods;
import com.mapr.db.rowcol.SequenceFileRowColCodec;
import com.mapr.fs.MapRFileSystem;

public class Export extends Configured implements Tool {

  private static final Logger LOG = LoggerFactory.getLogger(Export.class);
  private final static String NAME = "exporttable";
  private static boolean mapreduce = true;
  private static String tableName = null;
  private static String destPath = null;
  private static boolean preserveTimestamps = true;
  private static boolean getDeletes = true;
  private static boolean readAllCfs = false;
  private static int numThreads = 16;

  private static String columnSpec = null;
  private static String fileNamePrefix = "part";

  class ExporterThread implements Callable<Integer> {
    protected TabletInfo tabletInfo;
    protected int myid;
    protected Configuration config;

    ExporterThread(int id, TabletInfo t, Configuration config) {
      tabletInfo = t;
      myid = id;
      this.config = config;
    }

    @Override
    public Integer call() {
      QueryCondition c = tabletInfo.getCondition();

      MapRDBTableImpl srcTable = null;
      try {
        srcTable = new MapRDBTableImpl(new Path(tableName), config);
        DocumentStream rs = null;
        if (columnSpec != null) {
          //Don't need to return rows that don't qualify for the projection
          srcTable.setPrivateOption(TablePrivateOption.READ_ALL_CFS, readAllCfs);
          rs = srcTable.find(c, MapReduceUtilMethods.getFieldPaths(columnSpec));
        } else {
          rs = srcTable.find(c);
        }

        Iterator<Document> iterator = rs.iterator();

        //create a sequencefile
        if (!destPath.endsWith("/")) {
          destPath += "/";
        }
        Path outputPath = new Path(destPath+fileNamePrefix + Integer.toString(myid));
        FileSystem fs = MapRFileSystem.get(config);
        SequenceFile.Writer writer = SequenceFile.createWriter(fs, config, outputPath, ByteBufWritableComparable.class, ByteBufWritableComparable.class);

        ByteBufWritableComparable key = null;
        ByteBufWritableComparable value = null;

        while (iterator.hasNext()) {
          Document doc = iterator.next();
          key = new ByteBufWritableComparable(IdCodec.encode(doc.getId()));
          value = new ByteBufWritableComparable(SequenceFileRowColCodec.encode(doc));
          writer.append(key, value);
        }

        writer.close();

      } catch (Exception e) {
        LOG.error(NAME + " encountered an exception: " + e.getMessage());
        e.printStackTrace();
        return 1;
      } finally {
        try {
          if (srcTable != null) {
            srcTable.close();
          }
        } catch (Exception e) {
          LOG.error(NAME + " encountered an exception: " + e.getMessage());
          e.printStackTrace();
          return 1;
        }
      }

      return 0;
    }
  }

  public static class Exporter extends Mapper<Value, Document, ByteBufWritableComparable, ByteBufWritableComparable> {
    @Override
    public void map (Value key, Document value, Context context) throws IOException, InterruptedException {
      ByteBufWritableComparable serializedDoc = new ByteBufWritableComparable(SequenceFileRowColCodec.encode(value));
      ByteBufWritableComparable serializedKey = new ByteBufWritableComparable(IdCodec.encode(key));
      context.write(serializedKey, serializedDoc);
    }
  }

  private static Job createSubmittableJob(Configuration conf) throws IOException{
    conf.set(TableInputFormat.INPUT_TABLE, tableName);
    if (columnSpec != null) {
      conf.set(TableInputFormat.FIELD_PATH, columnSpec);
      //Don't need to return rows that don't qualify for the projection
      conf.setBoolean(TableInputFormat.READ_ALL_CFS, readAllCfs);
    }

    conf.setBoolean(MapRDBTableImpl.PRESERVE_TS_STR, preserveTimestamps);
    conf.setBoolean(MapRDBTableImpl.DECOMPRESS_STR, false);
    conf.setBoolean(MapRDBTableImpl.GET_DELETES_STR, getDeletes);

    Path path = new Path(destPath);
    Job job = new Job(conf, NAME + "_" + tableName);

    job.setJarByClass(Export.class);
    Configuration config = job.getConfiguration();
    config.setStrings("io.serializations", conf.get("io.serializations"),
        DocEmptySerialization.class.getName());

    job.setMapperClass(Exporter.class);

    job.setInputFormatClass(TableInputFormat.class);
    MapReduceUtilMethods.setStartStopRow(config);
    job.setSpeculativeExecution(false);

    job.setNumReduceTasks(0);
    job.setOutputFormatClass(SequenceFileOutputFormat.class);
    job.setOutputKeyClass(ByteBufWritableComparable.class);
    job.setOutputValueClass(ByteBufWritableComparable.class);
    FileOutputFormat.setOutputPath(job, path);
    return job;

  }

  private void ParseArgs(String[] args) throws Exception {
    try {
      for (int i = 0; i < args.length; ++i) {
        if (args[i].equalsIgnoreCase("-src")) {
          tableName = args[++i];
        } else if (args[i].equalsIgnoreCase("-dst")) {
          destPath = args[++i];
        } else if (args[i].equalsIgnoreCase("-mapreduce")) {
          mapreduce = Boolean.valueOf(args[++i]);
        } else if (args[i].equalsIgnoreCase("-columns")) {
          columnSpec = args[++i];
        } else {
          System.err.println("Invalid argument: " + args[i]);
          Usage(null);
        }
      }
    } catch (ArrayIndexOutOfBoundsException e) {
      System.err.println("Invalid command line arguments.");
      Usage(null);
    }

    if (tableName == null || destPath == null) {
      Usage("missing -src or -dst.");
    }

    Configuration conf = new Configuration();
    MapRFileSystem mfs = (MapRFileSystem)(FileSystem.get(conf));

    Path tablePath = new Path(tableName);
    if (!mfs.exists(tablePath)) {
      Usage(tablePath + " does not exist");
    }

    if (!mfs.isJsonTable(tablePath)) {
      Usage(tablePath + " is not a JSON table. This tool only supports JSON tables");
    }
  }

  private void Usage(final String errorMsg) {
    if (errorMsg != null && errorMsg.length() > 0) {
      System.err.println("ERROR: " + errorMsg);
    }
    System.err.println("Usage: " + NAME + " [options] "
                       + "-src <Input table name> "
                       + "-dst <dest directory>\n"
                       + "[-columns <JSON Fieldpaths specified as \"path1,...,pathN\">]"
                       + "[-mapreduce : <true|false> (default: true)]"
                       );
    System.exit(1);
  }

  public static void main(String[] args) throws Exception {

    int ret = 0;

    try {
      ret = ToolRunner.run(new Configuration(), new Export(), args);
    } catch (Exception e) {
      ret = 1;
      e.printStackTrace();
    }
    System.exit(ret);
  }

  private int run_NonMR(String[] args) throws Exception {

    Configuration config = getConf();

    Path dirPath = new Path(destPath);
    if (dirPath.getFileSystem(config).exists(dirPath)) {
      throw new FileAlreadyExistsException("Output directory " + dirPath +
                                           " already exists");
    }

    dirPath = dirPath.getFileSystem(config).makeQualified(dirPath);

    config.setBoolean(MapRDBTableImpl.PRESERVE_TS_STR, preserveTimestamps);
    config.setBoolean(MapRDBTableImpl.DECOMPRESS_STR, false);
    config.setBoolean(MapRDBTableImpl.GET_DELETES_STR, getDeletes);

    Table srcTable = MapRDB.getTable(tableName);

    //Calculate splits
    TabletInfo[] tabletInfos = srcTable.getTabletInfos();
    int numSplits = tabletInfos.length;
    //create executor thread pool using that
    long ts = System.currentTimeMillis();
    ExecutorService executor = Executors.newFixedThreadPool(numThreads);
    List<Future> futures = new ArrayList<Future>();

    for (int i = 0; i < numSplits; ++i) {
      Future f = executor.submit(new ExporterThread(i, tabletInfos[i], config));
      futures.add(f);
    }

    int numFailures = 0;
    for (Future f : futures) {
      numFailures += (Integer)f.get();
    }

    executor.shutdown();
    while (!executor.isTerminated());

    if (numFailures == 0) {
      doCleanup();
    }

    return (numFailures == 0 ? 0 : 1);
  }

  private static void doCleanup() {

  }

  @Override
  public int run(String[] args) throws Exception {
    Configuration conf = getConf();
    String[] otherArgs = new GenericOptionsParser(conf, args).getRemainingArgs();
    ParseArgs(otherArgs);

    if (!mapreduce) {
      return run_NonMR(otherArgs);
    }

    Job job = createSubmittableJob(conf);
    int retval = job.waitForCompletion(true)? 0 : 1;
    return retval;
  }
}
