package com.mapr.db.mapreduce.tools;

import java.io.IOException;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
//import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.LongWritable;
import org.apache.hadoop.io.NullWritable;
import org.apache.hadoop.io.SequenceFile;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;

import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.db.Admin;
import com.mapr.db.MapRDB;
import com.mapr.db.impl.TableDescriptorImpl;
import com.mapr.db.mapreduce.impl.ByteBufWritableComparable;
import com.mapr.db.mapreduce.impl.ClusterTablePath;
import com.mapr.db.mapreduce.impl.DiffTableUtils;
import com.mapr.db.mapreduce.impl.MapReduceUtilMethods;
import com.mapr.db.mapreduce.tools.impl.RangeChecksum;
import com.mapr.fs.MapRFileSystem;
import com.mapr.security.JNISecurity;

public class DiffTablesWithCrc extends Configured implements Tool{

  private static final Log LOG = LogFactory.getLog(DiffTablesWithCrc.class);
  public final int NumThreads = 16;

  public final static String NAME = "Difftableswithcrc";
  public final static String SrcChecksumJobNAME = "DiffTablesComputeSrcChecksum";
  public final static String DstChecksumJobNAME = "DiffTablesComputeDstChecksum";

  public static String outputDir = null; // default outputDir
  public static String outputDiffCrcDir = null; // default outputDir for checksum calculation
  public static String outputFileLocation = null;

  static String srcTableName = null;
  static String dstTableName = null;

  static ClusterTablePath outputDirCTPath = null;
  static ClusterTablePath srcCTPath = null;
  static ClusterTablePath dstCTPath = null;

  static String columnSpec = null;
  static boolean excludedEmbeddedFamily = false;
  static boolean exitOnFirstDiff = false;

  static String inputRegionKeyFile = null;
  static String inputSubRegionKeyFile = null;

  static boolean cmpMeta = true;

  static List<ByteBufWritableComparable> mismatchedStartKeys = null;

  public static void compareChecksum(String srcFileUri, String dstFileUri, String mismatchedFile) throws IOException {
    Configuration srcConf = new Configuration();
    Configuration dstConf = new Configuration();
    Configuration localConf = new Configuration();

    Path srcPath = new Path(srcFileUri);
    Path dstPath = new Path(dstFileUri);

    SequenceFile.Reader srcReader = new SequenceFile.Reader(srcConf,
                                        SequenceFile.Reader.file(srcPath));
    SequenceFile.Reader dstReader = new SequenceFile.Reader(dstConf,
                                        SequenceFile.Reader.file(dstPath));
    Path mismatchedFilePath = new Path(mismatchedFile);
    SequenceFile.Writer writer = SequenceFile.createWriter(localConf,
            SequenceFile.Writer.file(mismatchedFilePath),
            SequenceFile.Writer.keyClass(ByteBufWritableComparable.class),
            SequenceFile.Writer.valueClass(NullWritable.class));

    ByteBufWritableComparable srcKey = new ByteBufWritableComparable();
    LongWritable srcVal = new LongWritable();
    ByteBufWritableComparable dstKey = new ByteBufWritableComparable();
    LongWritable dstVal = new LongWritable();
    boolean srcHasNext = srcReader.next(srcKey, srcVal);
    boolean dstHasNext = dstReader.next(dstKey, dstVal);
    while (srcHasNext && dstHasNext) {

      int cmpret = srcKey.compareTo(dstKey);
      if (cmpret < 0) {
        // dst has a key range missing.
        LOG.debug(srcFileUri + " has an extra key range with start key (" + srcKey.toString() + ")");
        writer.append(srcKey, NullWritable.get());
        srcHasNext = srcReader.next(srcKey, srcVal);

      } else if (cmpret > 0) {
        // src has a key range missing.
        LOG.debug(dstFileUri + " has an extra key range with start key (" + dstKey.toString() + ")");
        writer.append(dstKey, NullWritable.get());
        dstHasNext = dstReader.next(dstKey, dstVal);
      } else { //srcKey == dstKey

        if (!srcVal.equals(dstVal)) {
          // src and dst have different checksum.
          LOG.debug(dstFileUri + " checksum(" + srcVal.toString() + ") and "
                 + srcFileUri + "  checksum(" + dstVal.toString() + ") are different for range start key ("
                 + srcKey.toString() + ")");
          writer.append(srcKey, NullWritable.get());
        }
        srcHasNext = srcReader.next(srcKey, srcVal);
        dstHasNext = dstReader.next(dstKey, dstVal);
      }
    }

    while (srcHasNext) {
      // Finished checksum for dst key ranges.
      LOG.debug(srcFileUri + " has an extra key range with start key " + srcKey.toString());
      writer.append(srcKey, NullWritable.get());
      srcHasNext = srcReader.next(srcKey, srcVal);
    }

    while (dstHasNext) {
      // Finished checksum for src key ranges.
      LOG.debug(dstFileUri + " has an extra key range with start key " + dstKey.toString());
      writer.append(dstKey, NullWritable.get());
      dstHasNext = dstReader.next(dstKey, dstVal);
    }

    writer.close();
    srcReader.close();
    dstReader.close();
  }

  private static boolean doCommandLine(final String[] args) {
    if (args.length <= 0) {
      printUsage(null);
      return false;
    }

    try {
      for (int i = 0; i < args.length; i++) {
        String cmd = args[i];
        if (cmd.equalsIgnoreCase("-h") || cmd.startsWith("--h")) {
          printUsage(null);
          return false;
        } else if (cmd.equalsIgnoreCase("-src")) {
          if (!checkNextArg(args, i))
            return false;
          srcTableName = args[++i];
          srcCTPath = ClusterTablePath.parse(srcTableName);
        } else if (cmd.equalsIgnoreCase("-dst")) {
          if (!checkNextArg(args, i))
            return false;
          dstTableName = args[++i];
          dstCTPath = ClusterTablePath.parse(dstTableName);
        } else if (args[i].equalsIgnoreCase("-columns")) {
          if (!checkNextArg(args, i))
            return false;
          columnSpec = args[++i];
        } else if (args[i].equalsIgnoreCase("-exclude_embedded_families")) {
          if (!checkNextArg(args, i))
            return false;
          excludedEmbeddedFamily = Boolean.valueOf(args[++i]);
        } else if (cmd.equalsIgnoreCase("-outdir")) {
          if (!checkNextArg(args, i))
            return false;
          outputDir = args[++i];
        } else if (cmd.equalsIgnoreCase("-first_exit")){
          exitOnFirstDiff = true;
        } else if (cmd.equalsIgnoreCase("-useRegionKeyFile")){
          if (!checkNextArg(args, i))
            return false;
          inputRegionKeyFile = args[++i];
        } else if (cmd.equalsIgnoreCase("-useSubRegionKeyFile")){
          if (!checkNextArg(args, i))
            return false;
          inputSubRegionKeyFile = args[++i];
        } else if (args[i].equalsIgnoreCase("-mapreduce")){
          if (!checkNextArg(args, i))
            return false;
          String isMapreduce = args[++i];
          if(isMapreduce.equalsIgnoreCase("false")) {
            printUsage("Non-mapreduce version is not supported.");
            return false;
          }
        } else if (args[i].equalsIgnoreCase("-cmpmeta")) {
          cmpMeta = Boolean.valueOf(args[++i]);
        } else {
          printUsage("unrecognized argument " + args[i]);
          return false;
        }
      }

      if (srcTableName == null) {
        printUsage("Missing -src.");
        return false;
      }
      if (dstTableName == null) {
        printUsage("Missing -dst.");
        return false;
      }

      Configuration conf = new Configuration();

      MapRFileSystem mfs = (MapRFileSystem)(FileSystem.get(conf));

      Path srcPath = new Path(srcTableName);
      Path dstPath = new Path(dstTableName);

      if (!mfs.exists(srcPath)) {
        printUsage(srcPath + " does not exist");
        return false;
      }

      if (!mfs.isJsonTable(srcPath)) {
        printUsage(srcPath + " is not a JSON table. This tool only supports JSON tables");
        return false;
      }

      if (!mfs.exists(dstPath)) {
        printUsage(dstPath + " does not exist");
        return false;
      }

      if (!mfs.isJsonTable(dstPath)) {
        printUsage(dstPath + " is not a JSON table. This tool only supports JSON tables");
        return false;
      }

      if (ClusterTablePath.equal(srcCTPath, dstCTPath)) {
        System.out.println("The tables match. " + srcTableName + " and " + dstTableName + " are the same table.");
        System.exit(0);
      }

      if (outputDir == null) {
        printUsage("Missing -outdir");
        return false;
      }

      if (!outputDir.endsWith("/")) { outputDir = outputDir + "/";}
      outputDirCTPath = ClusterTablePath.parse(outputDir);
      // check if user given output directory exists
      if (DiffTableUtils.checkPathExists(conf, outputDirCTPath)) {
        printUsage("Output directory " + outputDir + " already exists");
        return false;
      }
      outputDiffCrcDir = DiffTableUtils.getTmpDirName(conf) + outputDir;

      columnSpec = MapReduceUtilMethods.processColumnSpec(columnSpec, srcTableName);
      LOG.info("Comparing " + (columnSpec != null ? columnSpec : "all") +" column families from "
             + srcTableName + " to " + dstTableName);

      Admin admin = MapRDB.newAdmin();
      TableDescriptorImpl desc = (TableDescriptorImpl) admin.getTableDescriptor(srcTableName);
      if (desc.isStream()) {
        excludedEmbeddedFamily = true;
      }

    } catch (Exception e) {
      e.printStackTrace();
      printUsage("Can't start because " + e.getMessage());
      return false;
    }
    return true;
  }

  private static boolean checkNextArg(String[] args, int i) {
    //If a new option is add, remember to add its first character in the regex.
    //For example, if the option is called -abc, then change the regex to be
    //"-[acdefmnosu][a-zA-Z_]*"
    if (i + 1 >= args.length || args[i+1].matches("-[cdefmnosu][a-zA-Z_]*")) {
      printUsage("Missing argument after " + args[i]);
      return false;
    }
    return true;
  }

  private static void printUsage(final String errorMsg) {

    if (errorMsg != null && errorMsg.length() > 0) {
      System.err.println("ERROR: " + errorMsg);
    }

    System.err.println("Usage: maprdb difftableswithcrc\n"
        + "-src <source table path>\n"
        + "-dst <destination table path>\n"
        + "-outdir <output directory>\n"
        + "[-first_exit] Exit when first difference is found.\n"
        + "[-columns <comma separated list of field paths> ]\n"
        + "[-exclude_embedded_families <true|false>] (default: false)\n"
        +    "  Don't include the  other column families with path embedded in specified columns\n"
        //Not supported yet.
        //+ "[-mapreduce] <true|false> (default: true)]\n"
        //+ "[-numthreads <numThreads> (default:16, valid only when -mapreduce is false)]\n"
        + "[-cmpmeta <true|false> (default: true)]\n");
    //Internal use, do not print it to user.
    //System.err.println("  [-userInputRegionKeyFile] Specify the split regions that the "
        //+ "checksum mapper job should cover.");
    //System.err.println("  [-userInputSubRegionKeyFile] Specify the small ranges that a "
        //+ "checksum should be calculated on.");
  }

  @Override
  public int run(String[] args) throws Exception {

    if (!doCommandLine(args)) {
      System.exit(-1);
    }

    int ret = 0;
    if (cmpMeta && ((ret = compareMeta(args)) != 0)) {
      return ret;
    } else if (!cmpMeta) {
      System.out.println("Skip metadata check.");
    }

    Configuration conf = this.getConf();

    // Both src and dst will use the src key range file.
    String srcTableNameWithoutSpecialChar = ((srcCTPath.clusterName==null ? "" : srcCTPath.clusterName)+srcCTPath.tablePathName).replace("/", "").replace(":", "");
    String dstTableNameWithoutSpecialChar = ((dstCTPath.clusterName==null ? "" : dstCTPath.clusterName) +dstCTPath.tablePathName).replace("/", "").replace(":", "");

    String srcOutDiffCrcDirName = srcCTPath.getClusterUri().toString() + outputDiffCrcDir;
    Configuration srcConf = new Configuration();
    String srcFullClusterName = srcCTPath.getClusterUri().toString();
    if (srcFullClusterName != null) {
      srcConf.set("fs.defaultFS", srcFullClusterName);
    }
    Path srcOutDiffCrcDirPath = new Path(srcOutDiffCrcDirName);
    FileSystem srcFs = srcOutDiffCrcDirPath.getFileSystem(srcConf);
    //This is temp directory for checksum calculation, delete if it exists
    if (srcFs.exists(srcOutDiffCrcDirPath)) {
      srcFs.delete(srcOutDiffCrcDirPath, true /*true for recursive delete*/);
    }
    String srcKeyRangeDirName = new String(srcOutDiffCrcDirName + srcTableNameWithoutSpecialChar + ".keyrange/");
    String srcRegionKeyFilePathName = srcKeyRangeDirName + DiffTableUtils.getTabletKeyFileName(srcTableNameWithoutSpecialChar);
    String srcSubRegionKeyFilePathName = srcKeyRangeDirName + DiffTableUtils.getSegKeyFileName(srcTableNameWithoutSpecialChar);

    String dstOutDiffCrcDirName =  dstCTPath.getClusterUri().toString() + outputDiffCrcDir;
    Configuration dstConf = new Configuration();
    String dstFullClusterName = dstCTPath.getClusterUri().toString();
    if (dstFullClusterName != null) {
      dstConf.set("fs.defaultFS", dstFullClusterName);
    }
    Path dstOutDiffCrcDirPath = new Path(dstOutDiffCrcDirName);
    FileSystem dstFs = dstOutDiffCrcDirPath.getFileSystem(dstConf);
    //This is temp directory for checksum calculation, delete if it exists
    if (dstFs.exists(dstOutDiffCrcDirPath)) {
      dstFs.delete(dstOutDiffCrcDirPath, true /*true for recursive delete*/);
    }

    String dstKeyRangeDirName = new String(dstOutDiffCrcDirName +dstTableNameWithoutSpecialChar + ".keyrange/");
    String dstRegionKeyFilePathName = dstKeyRangeDirName + DiffTableUtils.getTabletKeyFileName(dstTableNameWithoutSpecialChar);
    String dstSubRegionKeyFilePathName = dstKeyRangeDirName + DiffTableUtils.getSegKeyFileName(dstTableNameWithoutSpecialChar);
    String diffTableOutDirPathName = outputDir;

    LOG.info("DiffTables output directory:" + diffTableOutDirPathName
            + "\nsrc side key range directory:" + srcKeyRangeDirName + ", mapper job split region file:"
            + srcRegionKeyFilePathName + ", checksum calculation subregion file:" + srcSubRegionKeyFilePathName
            + "\ndst side key range directory:" + dstKeyRangeDirName + ", mapper job split region file:"
            + dstRegionKeyFilePathName + ", checksum calculation subregion file:" + dstSubRegionKeyFilePathName);

    outputFileLocation = "DiffTables output are located at " + diffTableOutDirPathName
            + "\nsource table checksum output are located at "
            + srcOutDiffCrcDirName
            +"\ndestination table checksum output are located at "
            + dstOutDiffCrcDirName;

    if (inputRegionKeyFile != null) {
      // if user provide one, use it as input region
      DiffTableUtils.copyKeyRangeFile(conf, inputRegionKeyFile, conf, srcRegionKeyFilePathName);
      DiffTableUtils.copyKeyRangeFile(conf, inputRegionKeyFile, conf, dstRegionKeyFilePathName);
    } else {
      // if user does not provide one, use the tablet as input region
      DiffTableUtils.writeTabletKeyRange(srcRegionKeyFilePathName, srcTableName);
      //DiffTableUtils.writeTabletKeyRange(dstRegionKeyFilePathName, srcTableName);
      DiffTableUtils.copyKeyRangeFile(conf, srcRegionKeyFilePathName, conf, dstRegionKeyFilePathName);
    }

    int segnum = 0;
    if (inputSubRegionKeyFile != null) {
      // if user provide one, use it as input region
      DiffTableUtils.copyKeyRangeFile(conf, inputSubRegionKeyFile, conf, srcSubRegionKeyFilePathName);
      DiffTableUtils.copyKeyRangeFile(conf, inputSubRegionKeyFile, conf, dstSubRegionKeyFilePathName);
    } else {
      // if user does not provide one, use the tablet as input region
      segnum = DiffTableUtils.writeSegKeyRange(srcSubRegionKeyFilePathName, srcTableName);
      //dst uses the same sub region keys.
      DiffTableUtils.copyKeyRangeFile(conf, srcSubRegionKeyFilePathName, conf, dstSubRegionKeyFilePathName);
    }

    String localCluster = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();
    boolean isSecureCluster = JNISecurity.IsSecurityEnabled(localCluster);

    if (segnum == 0) {
      //src table does not have a segment, no need to calculate the checksums. Here we do not do the check for dst table, since
      //empty table is a corner case, we do not want to introduce extra code in normal code path for corner cases.
      return launchDiffTablesJob(diffTableOutDirPathName, null, null, args, isSecureCluster);
    }

    LOG.info(localCluster+" is "+(isSecureCluster?"":"not ") +"secure cluster.");
    if (isSecureCluster) {
      srcConf.setBoolean("fs.maprfs.impl.disable.cache", true);
      dstConf.setBoolean("fs.maprfs.impl.disable.cache", true);
    }

    //submit src side job to compute checksum
    String srcChecksumDirUri = srcOutDiffCrcDirName;
    String[] srcRCArgs = RangeChecksum.convertFromDiffTablesArg(true, // use src table
                                                                srcRegionKeyFilePathName, srcSubRegionKeyFilePathName,
                                                                null, // no included key regions
                                                                srcChecksumDirUri,
                                                                SrcChecksumJobNAME,
                                                                args);

    //submit dst side job to compute checksum
    String dstChecksumDirUri = dstOutDiffCrcDirName;
    String[] dstRCArgs = RangeChecksum.convertFromDiffTablesArg(false, // use dst table
                                                                dstRegionKeyFilePathName, dstSubRegionKeyFilePathName,
                                                                null, // no included key regions
                                                                dstChecksumDirUri,
                                                                DstChecksumJobNAME,
                                                                args);


    // XXX - The underlying infrastructure can't handle submitting jobs to
    // remote clusters.  So, with a tremendous hack, we modify the default
    // cluster name to trick the lower levels into talking to the right
    // destination.  The basic flow is:
    //   1) set cluster name to cluster where source job will run
    //   2) submit source job
    //   3) set cluster name to cluster where destination job will run
    //   4) submit destination job
    //   5) set cluster name to cluster where source job is running
    //   6) wait for source job to complete
    //   7) set cluster name to cluster where destination job is running
    //   8) wait for destination job to complete
    //   9) reset cluster name back to its original value
    // Keep in mind that no cluster means to use the default cluster, which is
    // the default for the current cluster name.
    if (srcCTPath.clusterName != null) {
      if (srcCTPath.clusterName.compareTo(CLDBRpcCommonUtils.getInstance().getCurrentClusterName()) != 0) {
        CLDBRpcCommonUtils.getInstance().setCurrentClusterName(srcCTPath.clusterName);
      }
    }

    Job srcJob = RangeChecksum.createSubmittableJob(srcConf, srcRCArgs);
    if (srcJob == null) {
      LOG.error("Failed to create job on cluster " + (srcCTPath.clusterName == null? "local" : srcCTPath.clusterName));
      return 1;	//TODO: define some meaningful error code?
    }
    srcJob.submit();

    if (dstCTPath.clusterName != null) {
      if (dstCTPath.clusterName.compareTo(CLDBRpcCommonUtils.getInstance().getCurrentClusterName()) != 0) {
        CLDBRpcCommonUtils.getInstance().setCurrentClusterName(dstCTPath.clusterName);
      }
    } else {
      CLDBRpcCommonUtils.getInstance().resetCurrentClusterName();
    }

    Job dstJob = RangeChecksum.createSubmittableJob(dstConf, dstRCArgs);
    if (dstJob == null) {
      LOG.error("Failed to create job on cluster " + (dstCTPath.clusterName == null? "local" : dstCTPath.clusterName));
      return 1;
    }
    dstJob.submit();

    if (srcCTPath.clusterName != null) {
      if (srcCTPath.clusterName.compareTo(CLDBRpcCommonUtils.getInstance().getCurrentClusterName()) != 0) {
         CLDBRpcCommonUtils.getInstance().setCurrentClusterName(srcCTPath.clusterName);
      }
    } else {
      CLDBRpcCommonUtils.getInstance().resetCurrentClusterName();
    }
    ret = srcJob.waitForCompletion(true) ? 0 : 1;
    if (ret != 0) {
      LOG.error("Wait for job to complete failed on cluster " + (srcCTPath.clusterName == null? "local" : srcCTPath.clusterName));
      return ret;
    }

    if (dstCTPath.clusterName != null) {
      if (dstCTPath.clusterName.compareTo(CLDBRpcCommonUtils.getInstance().getCurrentClusterName()) != 0) {
        CLDBRpcCommonUtils.getInstance().setCurrentClusterName(dstCTPath.clusterName);
      }
    } else {
      CLDBRpcCommonUtils.getInstance().resetCurrentClusterName();
    }
    ret = dstJob.waitForCompletion(true) ? 0 : 1;
    if (ret != 0) {
      LOG.error("Wait for job to complete failed on cluster " + (dstCTPath.clusterName == null? "local" : dstCTPath.clusterName));
      return ret;
    }

    CLDBRpcCommonUtils.getInstance().resetCurrentClusterName();
    // XXX - Hack zone over

    //read the checksum files and compare them. Since we only have one reducer for the checksum computing,
    //the checksum file name is "part-r-00000"
    //TODO: find a way to get this hard coded file name generated from hadoop configure if possible.
    String srcChecksumFileUri = RangeChecksum.getOutputPathName(srcChecksumDirUri, srcTableName) + "/" + "part-r-00000";
    String destChecksumFileUri = RangeChecksum.getOutputPathName(dstChecksumDirUri, dstTableName) + "/" + "part-r-00000";
    String mismatchedPathName = new String(outputDiffCrcDir
        + ((srcCTPath.clusterName==null ? "" : srcCTPath.clusterName)+srcCTPath.tablePathName).replace("/", "").replace(":", "")
        + ((dstCTPath.clusterName==null ? "" : dstCTPath.clusterName)+dstCTPath.tablePathName).replace("/", "").replace(":", "") + ".mismatchedkeyrange");


    LOG.info("srcChecksumFileUri=(" + srcChecksumFileUri + "), destChecksumFileUri=(" + destChecksumFileUri +
             "), mismatchedPathName=(" + mismatchedPathName + "), diffTableOutDirPathName =(" + diffTableOutDirPathName + ")");

    compareChecksum(srcChecksumFileUri, destChecksumFileUri, mismatchedPathName);

    //read the mismatched checksum files and launch DiffTables to compare row by row for each range.
    List<ByteBufWritableComparable> mismatchedStartKeys = DiffTableUtils.readKeyRange(conf, new Path(mismatchedPathName));
    if (mismatchedStartKeys.isEmpty()) {
      LOG.info("Succeed. Checksum for each range matches, skip row by row comparison. ");
      //touch an empty file to indicate the job succeed.
      DiffTableUtils.writeStringToFile(conf, "", diffTableOutDirPathName + DiffTableUtils.SucceedFileName);
      return 0;
    }

    return launchDiffTablesJob(diffTableOutDirPathName, srcSubRegionKeyFilePathName,
                               mismatchedPathName, args, isSecureCluster);
  }
  
  public int launchDiffTablesJob(String diffTableOutDirPathName, String srcSubRegionKeyFilePathName,
                                 String mismatchedPathName, String[] args, boolean isSecureCluster)
  {
    int ret = 0;
    //submit DiffTableRanges job to compare each row
    String[] diffRangeArgs = DiffTableUtils.diffTablesWithCrcArgToDiffTablesArg(
        diffTableOutDirPathName, srcSubRegionKeyFilePathName, mismatchedPathName, args);
    // verify the arguments, parseArgs will exit the program if verification failed.

    try {
      DiffTables diffTables = new DiffTables();
      diffTables.parseArgs(diffRangeArgs);

      Configuration diffTableConf = new Configuration();
      if (isSecureCluster) {
        diffTableConf.setBoolean("fs.maprfs.impl.disable.cache", true);
      }
      ret = ToolRunner.run(diffTableConf, diffTables, diffRangeArgs);
    } catch (Exception e) {
      ret = 1;
      e.printStackTrace();
    }
    if (ret != 0) {
      LOG.error("DiffTables job failed on this cluster.");
    }
    return ret;
  }

  private int compareMeta(String[] args) throws Exception {
    int ret = ToolRunner.run(getConf(), new DiffTablesMeta(true), args);
    if (ret == DiffTablesMeta.DIFFERENT_METADATA_RET) {
      System.out.println("ERROR: Metadata is different.");
      System.out.println("To skip metadata comparison, use the option -cmpmeta false.");
      System.exit(ret);
    } else if (ret == DiffTablesMeta.SAME_METADATA_RET) {
      System.out.println("DiffTablesMeta completed. Metadata of the two tables is same.");
    }
    return ret;
  }

  public static void main(String[] args) throws Exception {

    Configuration conf = new Configuration();
    int ret = 0;
    try {
      ret = ToolRunner.run(conf, new DiffTablesWithCrc(), args);
    } catch (Exception e) {
      ret = 1;
      e.printStackTrace();
      if (outputFileLocation != null) {
        System.err.println(outputFileLocation);
      }
    }
    System.exit(ret);
  }
}
