package com.mapr.db.mapreduce.tools;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

import com.mapr.db.Admin;
import com.mapr.db.FamilyDescriptor;
import com.mapr.db.MapRDB;
import com.mapr.db.Table;
import com.mapr.db.TableDescriptor;
import com.mapr.db.impl.MapRDBTableImpl;
import com.mapr.db.impl.TableDescriptorImpl;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.proto.Dbserver.AccessControlExpression;
import com.mapr.fs.proto.Dbserver.TableAces;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;

public class DiffTablesMeta extends Configured implements Tool{
  public static final String NAME = "DiffTablesMeta";
  public static final int METATYPE_NONE = 0;
  public static final int METATYPE_CFS =  0x1;
  public static final int METATYPE_ACES = 0x1 << 1;
  public static final int METATYPE_ATTRS = 0x1 << 2;
  public static final int METATYPE_ALL =
    METATYPE_CFS | METATYPE_ACES | METATYPE_ATTRS;

  public static final int SAME_METADATA_RET = 0;
  public static final int DIFFERENT_METADATA_RET = 1;

  private static String srcPath = null;
  private static String dstPath = null;
  private static MapRDBTableImpl srcTable = null;
  private static MapRDBTableImpl dstTable = null;
  private static TableDescriptorImpl srcDesc = null;
  private static TableDescriptorImpl dstDesc = null;


  static int metaTypeMask = METATYPE_NONE;

  static Configuration config = null;

  private boolean callFromDiffTables = true;
  public DiffTablesMeta(boolean callFromDiffTables){
    this.callFromDiffTables = callFromDiffTables;
  }

  public static boolean compareTableMeta(String src, String dst, int metaTypeMask)
    throws Exception {
    Admin maprAdmin = MapRDB.newAdmin();
    List<String> mismatchErrors = new ArrayList<String>();

    //1. Check for table existence and type
    if (!maprAdmin.tableExists(srcPath)) {
      throw new IOException("Table " + srcPath + " does not exist.");
    } else {
      try {
        srcTable = (MapRDBTableImpl)MapRDB.getTable(srcPath);
        srcDesc = (TableDescriptorImpl)srcTable.getTableDescriptor();
      } catch (Exception e) {
        new IOException("Failed to read table " + srcTable + " due to exception: " + e.getMessage());
      }
    }

    if (!maprAdmin.tableExists(dstPath)) {
      throw new IOException("Table " + dstPath + " does not exist.");
    } else {
      try {
        dstTable = (MapRDBTableImpl)MapRDB.getTable(dstPath);
        dstDesc = (TableDescriptorImpl)dstTable.getTableDescriptor();
      } catch (Exception e) {
        throw new IOException("Failed to read table " + dstTable + " due to exception: " + e.getMessage());
      }
    }

    if (srcDesc.isStream() != dstDesc.isStream()) {
      mismatchErrors.add("Table: types are different: " +
                         srcDesc.getPath() + " is" + (srcDesc.isStream() ? " a Stream, " : " a JSON Table, " +
                         dstDesc.getPath() + " is" + (dstDesc.isStream() ? " a Stream." : " a JSON Table.")));
    }

    //2. Compare Column Families and JSON Paths
    if ((metaTypeMask & METATYPE_CFS) > 0) {
      List<FamilyDescriptor> srcFams = srcDesc.getFamilies();
      List<FamilyDescriptor> dstFams = dstDesc.getFamilies();
      if (srcFams.size() != dstFams.size()) {
        mismatchErrors.add("Column family: Counts are different: " +
                            srcDesc.getPath() + " has " + srcFams.size() + " Column Families, " +
                            dstDesc.getPath() + " has " + dstFams.size() + " Column Families.");
      }

      for (ListIterator<FamilyDescriptor> srcIter = srcFams.listIterator(); srcIter.hasNext(); ) {
        boolean foundMatch = false;
        FamilyDescriptor srcFam = srcIter.next();
        for (ListIterator<FamilyDescriptor> dstIter = dstFams.listIterator(); dstIter.hasNext(); ) {
          FamilyDescriptor dstFam = dstIter.next();
          if (srcFam.getName().equals(dstFam.getName())) {
            foundMatch = true;
            if (!srcFam.equals(dstFam)) {
              mismatchErrors.add("Column Family: " + srcFam.getName() + " has different properties " +
                                 " in " + srcDesc.getPath() + " and " + dstDesc.getPath());
            }
            srcIter.remove();
            dstIter.remove();
          }
        }
        if (!foundMatch) {
          mismatchErrors.add("Column Family: " + srcFam.getName() + " with Json Path " +
                             srcFam.getJsonFieldPath() + " not found in " + dstDesc.getPath() + ".");
        }
      }
      if (!dstFams.isEmpty()) {
        for(FamilyDescriptor f : dstFams) {
          mismatchErrors.add("Column Family:" + f.getName() + " with Json Path " +
                             f.getJsonFieldPath() + " not found in " + srcDesc.getPath() + ".");
        }
      }
    }

    //TODO JJS Bug 20046 - Compare Table Default, CF aces
    if ((metaTypeMask & METATYPE_ACES) > 0) {
      TableAces.Builder srcAces = srcDesc.getTableAces();
      TableAces.Builder dstAces = dstDesc.getTableAces();
      if (srcAces.getAcesCount() != dstAces.getAcesCount()) {
        mismatchErrors.add("Table Aces: Number of aces are different. " + srcTable.getName() + " has " + srcAces.getAcesCount() +
                           " while " + dstTable.getName() + " has " + dstAces.getAcesCount() + " aces.");
      }

      List<AccessControlExpression> srcAceList = new ArrayList<AccessControlExpression>(srcAces.getAcesList());
      List<AccessControlExpression> dstAceList = new ArrayList<AccessControlExpression>(dstAces.getAcesList());
      for (AccessControlExpression srcAce : srcAceList) {
        boolean found = false;
        for (Iterator<AccessControlExpression> dstAceItr = dstAceList.iterator(); dstAceItr.hasNext();) {
          AccessControlExpression dstAce = dstAceItr.next();
          if (srcAce.getAccessType() == dstAce.getAccessType()) {
            found = true;
            if (!srcAce.getBooleanExpression().equals(dstAce.getBooleanExpression())) {
              mismatchErrors.add("Table Aces: Boolean expressions are different for access type " + srcAce.getAccessType() + ". " +
                                 srcTable.getName() + " has " + srcAce.getBooleanExpression().toStringUtf8() + " while " + dstTable.getName() + " has " + 
                                 dstAce.getBooleanExpression().toStringUtf8() + ".");
            }
            dstAceItr.remove();
          }
          if (!found) {
            mismatchErrors.add("Table Aces: ACE of access type " + srcAce.getAccessType() + " not found in " + dstTable.getName() + ".");
          }
        }
      }
      if(!dstAceList.isEmpty()) {
        for (AccessControlExpression dstAce : dstAceList) {
          mismatchErrors.add("Table aces: ACE of access type " + dstAce.getAccessType() + " not found in " + srcTable.getName() + ".");
        }
      }
    }

    if (mismatchErrors.size() > 0) {
      System.out.println("DiffTablesMeta Output:");
      for(String errMsg : mismatchErrors) {
        System.out.println(errMsg);
      }
      return false;
    }
    return true;
  }

  private boolean ParseArgs(final String[] args) {
    for (int i = 0; i < args.length; ++i) {
      if (args[i].equals("-h") || args[i].startsWith("--h")) {
        printUsage();
        return false;
      } else if (args[i].equalsIgnoreCase("-src")) {
        srcPath = args[++i];
      } else if (args[i].equalsIgnoreCase("-dst")) {
        dstPath = args[++i];
      } else if (args[i].equalsIgnoreCase("-columns")) {
        metaTypeMask |= METATYPE_CFS;
      } else if (args[i].equalsIgnoreCase("-Aces")) {
        metaTypeMask |= METATYPE_ACES;
      } else {
        if(callFromDiffTables)
          continue;
        printUsage();
      }
    }
    if (srcPath == null || dstPath == null)
      printUsage();
    return true;
  }

  private void printUsage() {
    if(callFromDiffTables)
      DiffTables.Usage(null);
    System.err.println("Usage: " + NAME + " -src tableA -dst tableB [options].\n" +
                       "Options:" +
                       "  [-columns] compare column families\n" +
                       "  [-Aces] compare Aces\n" +
                       "By default checks all of them.\n");
    System.exit(1);
  }

  @Override
  public int run(String[] args) throws Exception {
    ParseArgs(args);
    return compareTableMeta(srcPath, dstPath, metaTypeMask == 0
        ? METATYPE_ALL : metaTypeMask) ? SAME_METADATA_RET : DIFFERENT_METADATA_RET;
  }

  public static void main(String[] args) throws Exception {
    int ret = 0;
    try {
      ret = ToolRunner.run(new Configuration(), new DiffTablesMeta(false), args);
      if(ret == SAME_METADATA_RET){
        System.out.println("DiffTablesMeta completed. Metadata of the two tables is same.");
      } else if(ret == DIFFERENT_METADATA_RET){
        System.err.println("ERROR: Metadata is different.");
      }
    }catch (Exception e) {
      ret = 1;
      e.printStackTrace();
    }
    System.exit(ret);
  }
}
