package com.mapr.db.mapreduce.tools;

import java.io.IOException;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.mapreduce.InputSplit;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.mapreduce.RecordWriter;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.GenericOptionsParser;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.ojai.Document;
import org.ojai.DocumentStream;
import org.ojai.Value;
import org.ojai.store.QueryCondition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.db.Admin;
import com.mapr.db.MapRDB;
import com.mapr.db.Table;
import com.mapr.db.TableDescriptor;
import com.mapr.db.TabletInfo;
import com.mapr.db.impl.AdminImpl;
import com.mapr.db.impl.ConditionImpl;
import com.mapr.db.impl.ConditionNode.RowkeyRange;
import com.mapr.db.impl.MapRDBTableImpl.TablePrivateOption;
import com.mapr.db.impl.Constants;
import com.mapr.db.impl.IdCodec;
import com.mapr.db.impl.MapRDBTableImpl;
import com.mapr.db.impl.TableDescriptorImpl;
import com.mapr.db.impl.TabletInfoImpl;
import com.mapr.db.mapreduce.BulkLoadOutputFormat;
import com.mapr.db.mapreduce.BulkLoadRecordWriter;
import com.mapr.db.mapreduce.TableInputFormat;
import com.mapr.db.mapreduce.TableOutputFormat;
import com.mapr.db.mapreduce.impl.JsonImportMapper;
import com.mapr.db.mapreduce.impl.MapReduceConstants;
import com.mapr.db.mapreduce.impl.MapReduceUtilMethods;
import com.mapr.db.mapreduce.impl.MarlinSplitter;
import com.mapr.db.mapreduce.impl.TableSplit;
import com.mapr.db.rowcol.DBDocumentImpl;
import com.mapr.db.rowcol.IdValueComparator;
import com.mapr.db.rowcol.KeyValue;
import com.mapr.fs.MapRFileSystem;

public class CopyTable extends Configured implements Tool {
  private static final Logger LOG = LoggerFactory.getLogger(CopyTable.class);
  private static final String NAME = "CopyTable";
  public final static String TABLE_NAME = "import.table.name";
  private static String srcPath;
  private static String dstPath;
  private static boolean bulkLoad = false;
  private static boolean mapreduce = true;
  private static boolean isSuccess = false;
  private static boolean preserveTimestamps = true;
  private static boolean keepDeletes = true;
  private static boolean cmpMeta = true;
  private static boolean excludeEmbeddedCF = false;
  private static boolean isMarlin = false;
  private static boolean readAllCfs = false;

  //Bug 19484: Allow multiple columns to specify projection
  private static String columnSpec = null;
  private static String startRow = null;
  private static String stopRow = null;
  private int numThreads = 16;


  abstract class BaseLoaderThread implements Callable<Integer> {
    protected TabletInfo tabletInfo;
    protected int myid;
    protected Configuration config;

    protected BaseLoaderThread(int id, TabletInfo t, Configuration config) {
      this.myid = id;
      this.tabletInfo = t;
      this.config = config;
    }
  }

  class LoaderThread extends BaseLoaderThread {
    LoaderThread(int id, TabletInfo t, Configuration config) {
      super(id, t, config);
    }

    @Override
    public Integer call() {
      RecordWriter<Value, Document> writer = null;
      QueryCondition c = tabletInfo.getCondition();

      MapRDBTableImpl srcTable = null;
      MapRDBTableImpl destTable = null;
      try {
        srcTable = new MapRDBTableImpl(new Path(srcPath), config);
        DocumentStream rs = null;
        if (columnSpec != null) {
          rs = srcTable.find(c, columnSpec.split(",", -1));
        } else {
          rs = srcTable.find(c);
        }

        Iterator<Document> iterator = rs.iterator();
        if (bulkLoad) {
          // bulkload part : not tested
          // TODO : finish testing with bulkload -true
          writer = new BulkLoadRecordWriter(getConf(), new Path(dstPath));
          while(iterator.hasNext()) {
            Document r = iterator.next();
            writer.write(r.getId(), r);
          }
          writer.close(null);
        } else {
          destTable = new MapRDBTableImpl(new Path(dstPath), config);
          while (iterator.hasNext()) {
            Document r = iterator.next();
            destTable.insertOrReplace(r.getId(), r);
          }
        }
      } catch (Exception e) {
        LOG.error(NAME + " encountered an exception: " + e.getMessage());
        e.printStackTrace();
        return 1;
      } finally {
        try {
          if (srcTable != null) {
            srcTable.close();
          }
          if (destTable != null) {
            destTable.flush();
            destTable.close();
          }
        } catch (Exception e) {
          LOG.error(NAME + " encountered an exception: " + e.getMessage());
          e.printStackTrace();
        }
      }
      return 0;
    }
  }

  @SuppressWarnings("deprecation")
  private Job createSubmittableJob(String[] otherArgs) throws IOException {
    Job job = new Job(getConf(), NAME + "_" + srcPath);
    job.setJarByClass(CopyTable.class);

    Configuration conf = job.getConfiguration();
    conf.set(TableInputFormat.INPUT_TABLE, srcPath);
    conf.setBoolean(MapRDBTableImpl.PRESERVE_TS_STR, preserveTimestamps);
    conf.setBoolean(MapReduceConstants.PRESERVE_TS, preserveTimestamps);
    conf.setBoolean(MapRDBTableImpl.GET_DELETES_STR, keepDeletes);
    conf.setBoolean(MapRDBTableImpl.DECOMPRESS_STR, false);
    conf.setBoolean(TableInputFormat.EXCLUDE_EMBEDDEDFAMILY, excludeEmbeddedCF);
    conf.setBoolean(TableInputFormat.READ_ALL_CFS, readAllCfs);
    conf.setBoolean(Constants.BUFFERWRITE_STR, false);

    if (startRow != null)
      conf.set(TableInputFormat.START_ROW, startRow);

    if (stopRow != null)
      conf.set(TableInputFormat.STOP_ROW, stopRow);

    if (columnSpec != null) {
      conf.set(TableInputFormat.FIELD_PATH, columnSpec);
    }

    job.setInputFormatClass(TableInputFormat.class);
    job.setSpeculativeExecution(false);

    conf.set(TABLE_NAME, dstPath);
    job.setMapperClass(JsonImportMapper.class);

    MapReduceUtilMethods.setStartStopRow(conf);

    job.setOutputKeyClass(KeyValue.class);
    job.setSortComparatorClass(IdValueComparator.class);
    job.setOutputValueClass(DBDocumentImpl.class);

    setupCopyTableSource();
    setupCopyTableDestination();

    if (!bulkLoad) {
      job.setOutputFormatClass(TableOutputFormat.class);
      conf.set(TableOutputFormat.OUTPUT_TABLE, dstPath);
    } else {
      job.setOutputFormatClass(BulkLoadOutputFormat.class);
      conf.set(BulkLoadOutputFormat.OUTPUT_TABLE, dstPath);
    }

    job.setNumReduceTasks(0);
    return job;
  }

  private void createTableForCopy(Admin maprAdmin,
                                  String dstPath, String srcPath)
    throws IOException {
    TableDescriptor dstTableDesc = maprAdmin.getTableDescriptor(srcPath);
    dstTableDesc.setPath(dstPath);
    dstTableDesc.setBulkLoad(bulkLoad);

    // Get the list of splits from the source table.
    Table srcTable = MapRDB.getTable(srcPath);
    TabletInfo[] tabletInfos = srcTable.getTabletInfos();
    List<Value> startKeys = new ArrayList<Value>(tabletInfos.length);
    boolean isFirst = true;
    for (TabletInfo ti : tabletInfos) {
      if (isFirst) {
        // skip first tablet with -INF startkey
        isFirst = false;
        continue;
      }
      List<RowkeyRange> range =
        ((ConditionImpl)ti.getCondition()).getRowkeyRanges();
      startKeys.add(IdCodec.decode(range.get(0).getStartRow()));
    }
    Value[] splits = null;
    if (startKeys.size() > 0)
      splits = startKeys.toArray(new Value[startKeys.size()]);

    ((AdminImpl)maprAdmin).createTable(dstTableDesc, splits);
  }

  private void Usage(String errMsg) {
    if(errMsg != null) {
      System.err.println("ERROR: " + errMsg);
    }
    System.err.println("Usage: " + NAME + " [Options] " +
                       "-src <source table path> " +
                       "-dst <destination table path>\n" +
                       "Options:\n" +
                       "[-fromID <start key>]\n" +
                       "[-toID <end key>]\n" +
                       "[-columns <JSON Fieldpaths specified as \"path1,...,pathN\">]\n" +
                       "[-bulkload <true|false> (default: false)]\n" +
                       "[-mapreduce <true|false> (default: true)]\n" +
                       "[-cmpmeta <true|false> (default: true)]\n" +
                       "[-numthreads <number of threads> (default: 16)");
    System.exit(1);
  }

  private void ParseArgs(String args[]) throws Exception {
  List<String> cmpArgs = new ArrayList<String>();
    for (int i = 0; i < args.length; ++i) {
      if (args[i].equalsIgnoreCase("-src")) {
        cmpArgs.add(args[i]);
        srcPath = args[++i];
        cmpArgs.add(args[i]);
      } else if (args[i].equalsIgnoreCase("-dst")) {
        cmpArgs.add(args[i]);
        dstPath = args[++i];
        cmpArgs.add(args[i]);
      } else if (args[i].equalsIgnoreCase("-columns")) {
        columnSpec = args[++i];
      } else if (args[i].equalsIgnoreCase("-bulkload")) {
        boolean bulkLoadFlag = Boolean.valueOf(args[++i]);
        if (!MapReduceUtilMethods.checkBulkloadStatus(bulkLoadFlag, dstPath)) {
          Usage("Table "+dstPath+" is in bulkload mode and can't work with bulkload = false option.");
        }
        bulkLoad = bulkLoadFlag;
      } else if (args[i].equalsIgnoreCase("-startRow")) {
        startRow = args[++i];
      } else if (args[i].equalsIgnoreCase("-stopRow")) {
        stopRow = args[++i];
      } else if (args[i].equalsIgnoreCase("-mapreduce")) {
        mapreduce = Boolean.valueOf(args[++i]);
      } else if (args[i].equalsIgnoreCase("-numthreads")) {
        numThreads = Integer.parseInt(args[++i]);
      } else if (args[i].equalsIgnoreCase("-preserve_ts")) {  //Hidden option
        preserveTimestamps = Boolean.valueOf(args[++i]);
      } else if (args[i].equalsIgnoreCase("-cmpmeta")) {
        cmpMeta = Boolean.valueOf(args[++i]);
      } else {
        Usage(null);
      }
    }



    if (srcPath == null || dstPath == null)
      Usage("missing -src or -dst.");

    if((startRow != null && stopRow != null) &&
        startRow.compareTo(stopRow) < 0)
      Usage("stopRow is smaller than startRow.");

    Admin admin = MapRDB.newAdmin();

    if (preserveTimestamps) {
      if (admin.tableExists(dstPath) && cmpMeta) {
        //Add cmd arg to compare CFs and ACEs as well
        cmpArgs.add("-columns");
        cmpArgs.add("-Aces");
        int ret = ToolRunner.run(new Configuration(), new DiffTablesMeta(false), cmpArgs.toArray(new String[cmpArgs.size()]));
        if(ret == DiffTablesMeta.SAME_METADATA_RET){
          LOG.info("Metadata of the two tables matches.");
        } else if(ret == DiffTablesMeta.DIFFERENT_METADATA_RET){
          throw new Exception("Metadata of " + srcPath + " and " + dstPath + " is different.");
        }
      }
    } else {
      keepDeletes = false;
    }

    MapRFileSystem mfs = (MapRFileSystem)(FileSystem.get(new Configuration()));
    Path sPath = new Path(srcPath);

    if (!mfs.exists(sPath)) {
      Usage(sPath + " does not exist");
    }

    if (!mfs.isJsonTable(sPath)) {
      Usage(sPath + " is not a JSON table. This tool only supports JSON tables");
    }

    columnSpec = MapReduceUtilMethods.processColumnSpec(columnSpec, srcPath);
    LOG.info("Copying {} column families from {} to {}.",
        (columnSpec != null ? columnSpec : "all"), srcPath, dstPath);

    TableDescriptorImpl desc = (TableDescriptorImpl) admin.getTableDescriptor(srcPath);
    if (desc.isStream()) {
      excludeEmbeddedCF = true;
      isMarlin = true;
    }

    TableDescriptorImpl destDesc;
    if (admin.tableExists(dstPath)) {
      destDesc = (TableDescriptorImpl) admin.getTableDescriptor(dstPath);

      if ((!bulkLoad) && (destDesc.isBulkLoad())) {
        LOG.info("Default bulkload is false...setting it to true");
        bulkLoad = true;
      }
    }
  }

  private void setupCopyTableSource() throws IOException {
    //Marlin does NOT support bulkload
    Admin maprAdmin = MapRDB.newAdmin();
    if (((TableDescriptorImpl)maprAdmin.getTableDescriptor(srcPath)).isStream()) {
      bulkLoad = false;
    }
  }

  private void setupCopyTableDestination() throws IOException {
    Admin maprAdmin = MapRDB.newAdmin();
    if (!maprAdmin.tableExists(dstPath)) {
      createTableForCopy(maprAdmin, dstPath, srcPath);
    }
  }

  private void Cleanup() throws IOException {
    Admin maprAdmin = MapRDB.newAdmin();
    TableDescriptor dstTableDesc = maprAdmin.getTableDescriptor(dstPath);
    if (dstTableDesc.isBulkLoad()) {
      dstTableDesc.setBulkLoad(false);
      maprAdmin.alterTable(dstTableDesc);
    }
  }

  private int run_NoMR(String[] args) throws IOException, Exception {
    Configuration config = getConf();
    config.setBoolean(MapRDBTableImpl.PRESERVE_TS_STR, preserveTimestamps);
    config.setBoolean(MapRDBTableImpl.GET_DELETES_STR, keepDeletes);
    config.setBoolean(MapRDBTableImpl.EXCLUDE_EMBEDDEDFAMILY_STR, excludeEmbeddedCF);
    config.setBoolean(MapRDBTableImpl.DECOMPRESS_STR, false);
    config.setBoolean(Constants.BUFFERWRITE_STR, true);
    config.setBoolean(MapRDBTableImpl.READ_ALL_CFS_STR, readAllCfs);

    setupCopyTableSource();
    setupCopyTableDestination();

    Table srcTable = MapRDB.getTable(srcPath);

    //Calculate splits
    TabletInfo[] tabletInfos = srcTable.getTabletInfos();

    if (isMarlin) { // this is marlin
      List<InputSplit> splits = MarlinSplitter.getMarlinSplits(srcTable.getName(), tabletInfos);
      tabletInfos = new TabletInfo[splits.size()];

      int index = 0;
      for(InputSplit split : splits) {
        TableSplit thisSplit = (TableSplit) split;
        tabletInfos[index] = new TabletInfoImpl(thisSplit.getCondition(),
                                                thisSplit.getLocations(),
                                                thisSplit.getLength(),
                                                0 /* estimated num rows */);
        index++;
      }
    }

    srcTable.close();

    int numSplits = tabletInfos.length;

    //create executor thread pool using that
    long ts = System.currentTimeMillis();
    ExecutorService executor = Executors.newFixedThreadPool(numThreads);
    List<Future> futures = new ArrayList<Future>();

    for (int i = 0; i < numSplits; ++i) {
      Future f = executor.submit(new LoaderThread(i, tabletInfos[i], config));
      futures.add(f);
    }

    int numFailures = 0;
    for (Future f : futures) {
      numFailures += (Integer)f.get();
    }

    executor.shutdown();
    while (!executor.isTerminated());

    if (numFailures == 0) {
      Cleanup();
    }
    return (numFailures == 0 ? 0 : 1);
  }

  @Override
  public int run(String[] args) throws Exception {
    String[] otherArgs = new GenericOptionsParser(getConf(), args).getRemainingArgs();
    if (otherArgs.length < 2) {
      Usage("Wrong number of arguments: " + otherArgs.length);
      System.exit(-1);
    }
    ParseArgs(otherArgs);

    if (!mapreduce) {
      return run_NoMR(otherArgs);
    }

    Job job = createSubmittableJob(otherArgs);
    job.submit();
    System.out.println("job_id: "+ job.getJobID().toString());
    if (!job.waitForCompletion(true)) {
      LOG.error("CopyTable MapReduce job failed !!");
      return 1;
    }
    Cleanup();
    return 0;
  }

  public static boolean copy(final String[] args, String user) throws Exception {
    //By default, mapreduce is always set to true and -mapreduce is optional
    //Run as logged-in user instead of mapr user
    UserGroupInformation ugi = createUser(user);
    LOG.info("Running copytable job " + mapreduce + " as user: " + user);
    if (!mapreduce) {
      // This is to keep runcopytable happy
      System.out.println("No map-reduce");
    }
    ugi.doAs(new PrivilegedExceptionAction<Void>() {
        @Override
        public Void run() throws Exception {
          try {
            final int status = ToolRunner.run(new Configuration(), new CopyTable(), args);
            if (status==0)
              setJobSuccessful();
          } catch (Exception e) {
            LOG.error("Exception while running copytable job: " + e);
            throw new Exception(e.getMessage());
          }
          return null;
        }
    });
    return isSuccess;
  }

  public static void setJobSuccessful() {
    isSuccess = true;
  }

  public static void main(String[] args) throws Exception {
    int ret = 0;
    try {
      ret = ToolRunner.run(new Configuration(), new CopyTable(), args);
    } catch (Exception e) {
      ret = 1;
      e.printStackTrace();
    }
    System.exit(ret);
  }

  private static UserGroupInformation createUser(String user) throws IOException {
    return UserGroupInformation.createRemoteUser(user);
  }
}
