/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.streams.tests.xcr;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.List;

import org.apache.hadoop.conf.Configuration;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.Ignore;
import org.junit.experimental.categories.Category;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.streams.Admin;
import com.mapr.streams.StreamDescriptor;
import com.mapr.streams.Streams;

import com.mapr.streams.producer.ProducerPerformance;
import com.mapr.streams.listener.ListenerPerformance;

import com.mapr.tests.BaseTest;
import com.mapr.tests.annotations.StressTest;
import java.lang.ProcessBuilder;
import java.lang.Process;

@Category(StressTest.class)
public class CopyTableTest extends BaseTest {
  private static final Logger _logger = LoggerFactory.getLogger(CopyTableTest.class);
  private static final String SRCSTREAM = "/jtest-src-" + CopyTableTest.class.getSimpleName();
  private static final String DSTSTREAM = "/jtest-dst-" + CopyTableTest.class.getSimpleName();
  private static final String SRCSTREAMMR = SRCSTREAM+"MR";
  private static final String DSTSTREAMMR = DSTSTREAM+"MR";
  private static final String SRCSTREAMNOMR = SRCSTREAM+"NOMR";
  private static final String DSTSTREAMNOMR = DSTSTREAM+"NOMR";
  private static Admin madmin;

  // Need to send enough messages to trigger splits to test that messages across tablets for the
  // same topic partition are delivered in order.
  private static final int numParts = 1;
  private static final int numMsgs = 100000;
  private static final int numTopics = 30;

  private static boolean NoMRfailed = false;
  private static boolean MRfailed = false;

  @BeforeClass
  public static void setupTestClass() throws Exception {
    Configuration conf = new Configuration();
    madmin = Streams.newAdmin(conf);

    try {
      madmin.deleteStream(SRCSTREAMMR);
    } catch (Exception e) {}

    try {
      madmin.deleteStream(DSTSTREAMMR);
    } catch (Exception e) {}

    try {
      madmin.deleteStream(SRCSTREAMNOMR);
    } catch (Exception e) {}

    try {
      madmin.deleteStream(DSTSTREAMNOMR);
    } catch (Exception e) {}

    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(numParts);
    madmin.createStream(SRCSTREAMMR, sdesc);
    // Now setup copy table.
    ProcessBuilder proc = new ProcessBuilder("maprcli", "table", "edit", "-path", SRCSTREAMMR, "-regionsizemb", "512");
    Process process = proc.start();
    int errorCode = process.waitFor();
    System.out.println("Set region size, completed with return code " + errorCode);
    madmin.createStream(DSTSTREAMMR, sdesc);

    madmin.createStream(SRCSTREAMNOMR, sdesc);
    // Now setup copy table.
    proc = new ProcessBuilder("maprcli", "table", "edit", "-path", SRCSTREAMNOMR, "-regionsizemb", "512");
    process = proc.start();
    errorCode = process.waitFor();
    System.out.println("Set region size, completed with return code " + errorCode);
    madmin.createStream(DSTSTREAMNOMR, sdesc);
  }

  @AfterClass
  public static void cleanupTest() throws Exception {
    if (!MRfailed) {
      madmin.deleteStream(SRCSTREAMMR);
      madmin.deleteStream(DSTSTREAMMR);
    }
    if (!NoMRfailed) {
      madmin.deleteStream(SRCSTREAMNOMR);
      madmin.deleteStream(DSTSTREAMNOMR);
    }
  }

  @Test
  public void testCopyTableMROrder() throws IOException {
    // first produce all the messages and then setup copy table
    assertTrue(ProducerPerformance.runStressTest(SRCSTREAMMR,
                                                 numMsgs,
                                                 numTopics,
                                                 numParts,
                                                 1 /* numbatches */,
                                                 10*1000 /*batch sleep*/,
                                                 true /* verify */,
                                                 false /* multiple flushers */,
                                                 true /* print progress */,
                                                 false /* roundrobin*/,
                                                 false /*hashkey*/));

    try {

      // kick start a listener
      ListenerPerformance lp = new ListenerPerformance(DSTSTREAMMR,
                                                       SRCSTREAMMR,
                                                       numTopics,
                                                       numParts,
                                                       numMsgs,
                                                       1,
                                                       true /* verify */,
                                                       true /* keys in order */,
                                                       false /*allowDuplicateKeys */,
                                                       false /*isTracingEnabled*/,
                                                       true /*printProgress*/,
                                                       null /*groupId*/,
                                                       "CopyTableTest",
                                                       false /*topicSubscriptions*/);

      Thread lt = new Thread(lp);
      lt.start();

      // Now setup copy table.
      ProcessBuilder proc = new ProcessBuilder("hadoop", "jar", "/opt/mapr/lib/mapr-tools-5.2.2-mapr.jar",
                                               "com.mapr.db.mapreduce.tools.CopyTable", "-src", SRCSTREAMMR, "-dst",
                                               DSTSTREAMMR, "-bulkload", "false", "-mapreduce", "true");
      Process copyTableProc = proc.start();
      lt.join();

      MRfailed = !lp.status;
      assertTrue(lp.status);

      copyTableProc.waitFor();
      System.out.println("CopyTable process exit code: " + copyTableProc.exitValue());
    } catch (InterruptedException e) {
      System.out.println(e);
      MRfailed = true;
      assertTrue(false);
    }
  }

  @Test
  public void testCopyTableNonMROrder() throws IOException {
    // first produce all the messages and then setup copy table
    assertTrue(ProducerPerformance.runStressTest(SRCSTREAMNOMR,
                                                 numMsgs,
                                                 numTopics,
                                                 numParts,
                                                 1 /* numbatches */,
                                                 10*1000 /*batch sleep*/,
                                                 true /* verify */,
                                                 false /* multiple flushers */,
                                                 true /* print progress */,
                                                 false /* roundrobin*/,
                                                 false /*hashkey*/));

    try {

      // kick start a listener
      ListenerPerformance lp = new ListenerPerformance(DSTSTREAMNOMR,
                                                       SRCSTREAMNOMR,
                                                       numTopics,
                                                       numParts,
                                                       numMsgs,
                                                       1,
                                                       true /* verify */,
                                                       true /* keys in order */,
                                                       false /*allowDuplicateKeys */,
                                                       false /*isTracingEnabled*/,
                                                       true /*printProgress*/,
                                                       null /*groupId*/,
                                                       "CopyTableTest",
                                                       false /*topicSubscriptions*/);

      Thread lt = new Thread(lp);
      lt.start();

      // Now setup copy table.
      ProcessBuilder proc = new ProcessBuilder("hadoop", "jar", "/opt/mapr/lib/mapr-tools-5.2.2-mapr.jar",
                                               "com.mapr.db.mapreduce.tools.CopyTable", "-src", SRCSTREAMNOMR, "-dst",
                                               DSTSTREAMNOMR, "-bulkload", "false", "-mapreduce", "false");
      Process copyTableProc = proc.start();

      copyTableProc.waitFor();
      System.out.println("CopyTable process exit code: " + copyTableProc.exitValue());
      lt.join();

      NoMRfailed = !lp.status;

      assertTrue(lp.status);
    } catch (InterruptedException e) {
      System.out.println(e);
      NoMRfailed = true;
      assertTrue(false);
    }
  }

}
