/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.streams.tests.xcr;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.List;

import org.apache.hadoop.conf.Configuration;
import org.junit.After;
import org.junit.BeforeClass;
import org.junit.Before;
import org.junit.Test;
import org.junit.Ignore;
import org.junit.experimental.categories.Category;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.streams.Admin;
import com.mapr.streams.StreamDescriptor;
import com.mapr.streams.Streams;

import com.mapr.streams.producer.ProducerPerformance;
import com.mapr.streams.listener.ListenerPerformance;

import com.mapr.tests.BaseTest;
import com.mapr.tests.annotations.StressTest;
import java.lang.ProcessBuilder;
import java.lang.Process;

@Category(StressTest.class)
public class AsyncReplTest extends BaseTest {
  private static final Logger _logger = LoggerFactory.getLogger(AsyncReplTest.class);
  private static final String SRCSTREAM = "/jtest-src-" + AsyncReplTest.class.getSimpleName();
  private static final String DSTSTREAM = "/jtest-dst-" + AsyncReplTest.class.getSimpleName();
  private static Admin madmin;

  // Need to send enough messages to trigger splits to test that messages across tablets for the
  // same topic partition are delivered in order.
  private static final int numParts = 1;
  private static final int numMsgs = 100000;
  private static final int numTopics = 30;

  @BeforeClass
  public static void setupTestClass() throws Exception {
    Configuration conf = new Configuration();
    madmin = Streams.newAdmin(conf);

    try {
      madmin.deleteStream(SRCSTREAM);
    } catch (Exception e) {}

    try {
      madmin.deleteStream(DSTSTREAM);
    } catch (Exception e) {}
  }

  @Before
  public void setupTest() throws Exception {
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(numParts);
    madmin.createStream(SRCSTREAM, sdesc);
    // Now setup copy table.
    ProcessBuilder proc = new ProcessBuilder("maprcli", "table", "edit", "-path", SRCSTREAM, "-regionsizemb", "512");
    Process process = proc.start();
    int errorCode = process.waitFor();
    System.out.println("Set region size, completed with return code " + errorCode);

    // Don't create the destination stream, will just use autosetup
  }

  @After
  public void cleanupTest() throws Exception {
    madmin.deleteStream(SRCSTREAM);
    madmin.deleteStream(DSTSTREAM);
  }

  @Test
  public void testAsyncReplAutoSetupOrder() throws IOException {

    try {
      // Now setup copy table.
      ProcessBuilder proc = new ProcessBuilder("maprcli", "stream", "replica",
                                               "autosetup", "-path", SRCSTREAM,
                                               "-replica", DSTSTREAM);
      Process autosetup = proc.start();
      autosetup.waitFor();
      System.out.println("Replica autosetup process exit code: " + autosetup.exitValue());


      runProducerAndListener();
    } catch (InterruptedException e) {
      System.out.println(e);
      assertTrue(false);
    }
  }

  @Test
  public void testAsyncReplManualSetupOrder() throws IOException {

    try {
      StreamDescriptor sdesc = Streams.newStreamDescriptor();
      sdesc.setDefaultPartitions(numParts);
      madmin.createStream(DSTSTREAM, sdesc);

      // Now setup copy table.
      ProcessBuilder proc = new ProcessBuilder("maprcli", "stream", "replica",
                                               "add", "-path", SRCSTREAM,
                                               "-replica", DSTSTREAM);
      Process replicaAdd  = proc.start();
      replicaAdd.waitFor();
      System.out.println("Replica add process exit code: " + replicaAdd.exitValue());

      proc = new ProcessBuilder("maprcli", "stream", "upstream",
                                "add", "-path", DSTSTREAM,
                                "-upstream", SRCSTREAM);
      Process upstreamAdd = proc.start();
      upstreamAdd.waitFor();
      System.out.println("Upstream add process exit code: " + upstreamAdd.exitValue());

      runProducerAndListener();

    } catch (InterruptedException e) {
      System.out.println(e);
      assertTrue(false);
    }
  }

  private void runProducerAndListener() throws InterruptedException {
    // wait 5 seconds
    Thread.sleep(5000);

    // Only print listener progress, not producer
    ProducerRunnable pp = new ProducerRunnable(SRCSTREAM,
                                               numMsgs,
                                               numTopics,
                                               numParts,
                                               1 /*numbatches*/,
                                               10*1000 /*batch sleep*/,
                                               true /* verify */,
                                               false /* multiple flushers */,
                                               false /* print progress */,
                                               false /* roundrobin*/,
                                               false /*hashkey*/);
    Thread pt = new Thread(pp);
    pt.start();

    // kick start a listener
    ListenerPerformance lp = new ListenerPerformance(DSTSTREAM,
                                                     SRCSTREAM,
                                                     numTopics,
                                                     numParts,
                                                     numMsgs,
                                                     1 /*numbatches*/,
                                                     true /* verify */,
                                                     true /* keys in order */,
                                                     false /*allowDuplicateKeys */,
                                                     false /*isTracingEnabled*/,
                                                     true /*printProgress*/,
                                                     null /*groupId*/,
                                                     "AsyncReplTest",
                                                     false /*topicSubscriptions*/);

    Thread lt = new Thread(lp);
    lt.start();

    pt.join();
    lt.join();
    assertTrue(pp.status);
    assertTrue(lp.status);
  }

  private static class ProducerRunnable implements Runnable {
    public String stream;
    public int nmsgs;
    public int ntopics;
    public int npartitions;
    public int nbatches;
    public long batchsleepms;
    public boolean verifyKeys;
    public boolean mflushers;
    public boolean printProgress;
    public boolean roundrobin;
    public boolean hashkey;

    public boolean status;

    public ProducerRunnable(String stream, int nmsgs, int ntopics, int nparts,
                            int nbatches, long batchSleepMs, boolean verifyKeys,
                            boolean mflushers, boolean print, boolean rr, boolean hash) {
      this.stream = stream;
      this.nmsgs = nmsgs;
      this.ntopics = ntopics;
      this.npartitions = nparts;
      this.nbatches = nbatches;
      this.batchsleepms = batchSleepMs;
      this.verifyKeys = verifyKeys;
      this.mflushers = mflushers;
      this.printProgress = print;
      this.roundrobin = rr;
      this.hashkey = hash;
      this.status = false;
    }

    public void run() {
      try {
        status = ProducerPerformance.runStressTest(stream,
                                                   nmsgs,
                                                   ntopics,
                                                   npartitions,
                                                   nbatches,
                                                   batchsleepms,
                                                   verifyKeys,
                                                   mflushers,
                                                   printProgress,
                                                   roundrobin,
                                                   hashkey);
      } catch (Exception e) {
        System.out.println(e);
        status = false;
      }
    }
  }

}
