/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.streams.tests.producer;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.List;

import org.apache.hadoop.conf.Configuration;
import org.junit.AfterClass;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.streams.Admin;
import com.mapr.streams.Streams;
import com.mapr.streams.StreamDescriptor;

import com.mapr.streams.producer.ProducerPerformance;
import com.mapr.tests.BaseTest;
import com.mapr.tests.annotations.ClusterTest;

@Category(ClusterTest.class)
public class ProducerStressTest extends BaseTest {
  private static final Logger _logger = LoggerFactory.getLogger(ProducerStressTest.class);
  private static final String STREAM = "/jtest-" + ProducerStressTest.class.getSimpleName();
  private static Admin madmin;

  private static final int numParts = 4;

  @BeforeClass
  public static void setupTestClass() throws Exception {
    final Configuration conf = new Configuration();
    madmin = Streams.newAdmin(conf);

    try {
      madmin.deleteStream(STREAM);
    } catch (Exception e) {}
  }

  @Before
  public void setupTest() throws Exception {
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(numParts);
    madmin.createStream(STREAM, sdesc);
  }

  @After
  public void cleanupTest() throws Exception {
    madmin.deleteStream(STREAM);
  }

  @Test
  public void testProducerPerformancePartition() throws IOException {
    assertTrue(ProducerPerformance.runStressTest(STREAM,
                                                 100000 /*nmsgs per partition */,
                                                 2 /*ntopics*/,
                                                 numParts /*npartitions*/,
                                                 1 /*nbatches*/,
                                                 10*1000 /*ms between batches*/,
                                                 true /*needsVerify*/,
                                                 true /*multiple flushers */,
                                                 true /*progress*/,
                                                 false /*roundrobin*/,
                                                 false /*hashkey*/));
  }

  @Test
  public void testProducerPerformanceRoundRobin() throws IOException {
    assertTrue(ProducerPerformance.runStressTest(STREAM,
                                                 100000 /*nmsgs per partition */,
                                                 2 /*ntopics*/,
                                                 numParts /*npartitions*/,
                                                 1 /*nbatches*/,
                                                 10*1000 /*ms between batches*/,
                                                 false /*needsVerify*/,
                                                 true /*multiple flushers */,
                                                 true /*progress*/,
                                                 true /*roundrobin*/,
                                                 false /*hashkey*/));
  }

  @Test
  public void testProducerPerformanceHashKey() throws IOException {
    assertTrue(ProducerPerformance.runStressTest(STREAM,
                                                 100000 /*nmsgs per partition */,
                                                 2 /*ntopics*/,
                                                 numParts /*npartitions*/,
                                                 1 /*nbatches*/,
                                                 10*1000 /*ms between batches*/,
                                                 false /*needsVerify*/,
                                                 true /*multiple flushers */,
                                                 true /*progress*/,
                                                 false /*roundrobin*/,
                                                 true /*hashkey*/));
  }
}
