/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.streams.tests.producer;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.concurrent.Future;
import java.util.Map;
import org.apache.hadoop.conf.Configuration;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import java.lang.Thread;
import org.junit.experimental.categories.Category;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.util.Properties;
import com.mapr.streams.Admin;
import com.mapr.streams.Streams;
import com.mapr.streams.StreamDescriptor;
import org.apache.kafka.clients.producer.Callback;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.common.PartitionInfo;
import com.mapr.fs.proto.Marlinserver.MarlinConfigDefaults;
import org.apache.kafka.clients.producer.StreamsPartitioner;
import com.mapr.streams.tests.producer.TestPartitioner;

import com.mapr.tests.BaseTest;
import com.mapr.tests.annotations.ClusterTest;

@Category(ClusterTest.class)
public class ProducerPartitionerTest extends BaseTest {
  private static final Logger _logger = LoggerFactory.getLogger(ProducerPartitionerTest.class);
  private static final String STREAM = "/jtest-" + ProducerPartitionerTest.class.getSimpleName();
  private static final String TOPIC = "testtopic";
  private static Admin madmin;
  private static KafkaProducer producer;
  private static Properties props;
  public static int msgValueLength = 200;
  public static final byte[] value = new byte[msgValueLength];
  public static final byte[] key = "abc".getBytes();
  public static int numPartitions = 10;

  @BeforeClass
  public static void setupTest() throws Exception {
    final Configuration conf = new Configuration();
    madmin = Streams.newAdmin(conf);

    try {
      madmin.deleteStream(STREAM);
    } catch (Exception e) {}
  }

  @Before
  public void setupTable() throws Exception {
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(numPartitions);
    sdesc.setAutoCreateTopics(true);
    madmin.createStream(STREAM, sdesc);
  }

  @After
  public void cleanupTest() throws Exception {
    madmin.deleteStream(STREAM);
  }

  @Test
  public void testSendWithDefaultPartitioner() throws IOException {
    MarlinConfigDefaults cdef = MarlinConfigDefaults.getDefaultInstance();
    props = new Properties();
    props.put("key.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    props.put("value.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    props.put(cdef.getParallelFlushersPerPartition(), true);
    props.put(cdef.getMetadataMaxAge(), 100);  // want to exercise metadata refresher
    props.put(cdef.getBufferTime(), 500);  // flush quickly for testing
    // Use default partitioner

    KafkaProducer producer = new KafkaProducer<byte[], byte[]>(props);

    ProducerRecord<byte[], byte[]> record =
      new ProducerRecord<byte[], byte[]>(STREAM+":"+TOPIC, 1, key, value);
    TestCallback callback1 = new TestCallback(1, false);
    Future<RecordMetadata> future1 = producer.send(record, callback1);
    try {
      future1.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback1.verify());

    record = new ProducerRecord<byte[], byte[]>(STREAM+":"+TOPIC, key, value);
    TestCallback callback2 = new TestCallback(false);
    Future<RecordMetadata> future2 = producer.send(record, callback2);
    try {
      future2.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback2.verify());

    record =
      new ProducerRecord<byte[], byte[]>(STREAM+":"+TOPIC, key, value);
    TestCallback callback3 = new TestCallback(callback2.partitionid(), false);
    Future<RecordMetadata> future3 = producer.send(record, callback3);
    try {
      future3.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback3.verify());

    producer.close();
    // just to check
    producer.flush();
    producer.close();
  }

  @Test
  public void testSendWithCustomPartitioner() throws IOException {
    MarlinConfigDefaults cdef = MarlinConfigDefaults.getDefaultInstance();
    props = new Properties();
    props.put("key.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    props.put("value.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    props.put(cdef.getParallelFlushersPerPartition(), true);
    props.put(cdef.getMetadataMaxAge(), 100);  // want to exercise metadata refresher
    props.put(cdef.getBufferTime(), 500);  // flush quickly for testing
    props.put(cdef.getPartitioner(),"com.mapr.streams.tests.producer.TestPartitioner");

    KafkaProducer producer = new KafkaProducer<byte[], byte[]>(props);

    for (int i = 1; i < numPartitions; i++) {
      ProducerRecord<byte[], byte[]> record =
        new ProducerRecord<byte[], byte[]>(STREAM+":"+TOPIC, key, value);
      TestCallback callback = new TestCallback(i, false);
      Future<RecordMetadata> future = producer.send(record, callback);
      try {
        future.get();
      } catch (Exception e) {
        System.out.println(e);
      }
      assertTrue(callback.verify());
    }

    ProducerRecord<byte[], byte[]> record =
        new ProducerRecord<byte[], byte[]>(STREAM+":"+TOPIC, key, value);
    TestCallback callback = new TestCallback(-1, true);
    Future<RecordMetadata> future = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    producer.close();

    // Just so that we always pass this.
    producer.flush();
    producer.close();
  }

  @Test
  public void testSendWithKeyBasedPartitioner() throws IOException {
    MarlinConfigDefaults cdef = MarlinConfigDefaults.getDefaultInstance();
    props = new Properties();
    props.put("key.serializer", "org.apache.kafka.common.serialization.StringSerializer");
    props.put("value.serializer", "org.apache.kafka.common.serialization.StringSerializer");
    props.put(cdef.getParallelFlushersPerPartition(), true);
    props.put(cdef.getMetadataMaxAge(), 100);  // want to exercise metadata refresher
    props.put(cdef.getBufferTime(), 500);  // flush quickly for testing
    props.put(cdef.getPartitioner(),"com.mapr.streams.tests.producer.KeyBasedPartitioner");

    KafkaProducer producer = new KafkaProducer<String, String>(props);

    int numMsgsPerKey = 1000;
    String keyA = "abc";
    String keyB = "123456789";
    String value = "this is a value";
    Future[] futuresA = new Future[numMsgsPerKey];
    Future[] futuresB = new Future[numMsgsPerKey];

    for (int i = 0; i < numMsgsPerKey ; i++) {
      ProducerRecord<String, String> recordA =
        new ProducerRecord<String, String>(STREAM+":"+TOPIC, keyA, value);
      ProducerRecord<String, String> recordB =
        new ProducerRecord<String, String>(STREAM+":"+TOPIC, keyB, value);
      futuresA[i] = producer.send(recordA);
      futuresB[i] = producer.send(recordB);
    }

    producer.flush();
    producer.close();

    long lastOffsetA = -1;
    long lastOffsetB = -1;
    int feedIDA = 0;
    int feedIDB = 1;
    try {
      for (int i = 0; i < numMsgsPerKey; i++) {
        Future<RecordMetadata> futureA = futuresA[i];
        //System.out.println("A " + lastOffsetA + " < " + futureA.get().offset() + ", feed " + futureA.get().partition());
        assertTrue(lastOffsetA < futureA.get().offset());
        assertTrue(feedIDA == futureA.get().partition());
        lastOffsetA = futureA.get().offset();
        Future<RecordMetadata> futureB = futuresB[i];
        //System.out.println("B " + lastOffsetB + " < " + futureB.get().offset() + ", feed " + futureB.get().partition());
        assertTrue(lastOffsetB < futureB.get().offset());
        assertTrue(feedIDB == futureB.get().partition());
        lastOffsetB = futureB.get().offset();
      }
    } catch (Exception e) {
      System.out.println(e);
      assertTrue(false);
    }
  }

  private static final class TestCallback implements Callback {
    private boolean error;
    private int expectedFeedID;
    private Exception exceptionReceived;
    private RecordMetadata metadataReceived;
    private boolean callbackCompleted;
    private boolean checkfeedID;

    public TestCallback(boolean errors) {
      this.checkfeedID = false;
      this.error = errors;
      this.callbackCompleted = false;
    }
    public TestCallback(int feedid, boolean errors) {
      this.checkfeedID = true;
      this.expectedFeedID = feedid;
      this.error = errors;
      this.callbackCompleted = false;
    }

    public void onCompletion(RecordMetadata metadata,
                             Exception exception) {
      exceptionReceived = exception;
      metadataReceived = metadata;
      synchronized(this) {
        this.callbackCompleted = true;
        try {
          this.notifyAll();
        } catch (Exception e) {
          System.out.println(e);
        }
      }
    }

    public int partitionid() {
      return metadataReceived.partition();
    }

    public boolean verify() {
      synchronized(this) {
        if (!callbackCompleted) {
          try {
            this.wait();
          } catch (Exception e) {
            System.out.println(e);
          }
        }
      }
      boolean verified = true;
      if (error && exceptionReceived == null) {
        System.out.println("Did not get exception when expected");
        verified = false;
      } else if (!error && exceptionReceived != null) {
        System.out.println("Received exception " + exceptionReceived + " but expected none");
        verified = false;
      }

      if (checkfeedID && expectedFeedID != metadataReceived.partition()) {
        System.out.println("Received partition " + metadataReceived.partition() + " but expected " + expectedFeedID);
        verified = false;
      }

      return verified;
    }
  }
}
