/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.streams.tests.producer;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.assertFalse;

import java.io.IOException;
import java.net.URI;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.concurrent.Future;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.Path;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.After;
import org.junit.Test;
import java.lang.Thread;
import org.junit.experimental.categories.Category;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.util.Properties;
import com.mapr.streams.Admin;
import com.mapr.streams.impl.admin.MarlinAdminImpl;
import com.mapr.streams.StreamDescriptor;
import com.mapr.streams.Streams;
import org.apache.kafka.clients.producer.Callback;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.common.PartitionInfo;

import com.mapr.fs.jni.MapRConstants;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.proto.Marlinserver.MarlinConfigDefaults;
import com.mapr.fs.proto.Marlinserver.MarlinTopicMetaEntry;

import com.mapr.tests.BaseTest;
import com.mapr.tests.annotations.ClusterTest;

@Category(ClusterTest.class)
public class ProducerErrorTest extends BaseTest {
  private static final Logger _logger = LoggerFactory.getLogger(ProducerErrorTest.class);
  private static final String STREAM = "/jtest-" + ProducerErrorTest.class.getSimpleName();
  private static final String STREAMFUTURES = STREAM + "-futures";
  private static final String STREAMKEY = STREAM + "-key";
  private static final String STREAMAUTO = STREAM + "-auto";
  private static final String STREAMNA = STREAM + "-noauto";
  private static final String STREAMMETA = STREAM + "-meta";
  private static final String STREAMENOENT = STREAM + "-enoent";
  private static final String STREAMENOENTAUTO = STREAM + "-enoent-auto";
  private static final String STREAMDELETE = STREAM + "-deletestream";
  private static final String TOPIC = "testtopic";
  private static KafkaProducer producer;
  private static Properties props;
  public static int msgValueLength = 200;
  public static final byte[] value = new byte[msgValueLength];
  public static final byte[] key = "abc".getBytes();

  @BeforeClass
  public static void setupTest() throws Exception {
    Configuration conf = new Configuration();
    Admin madmin = Streams.newAdmin(conf);
    MarlinConfigDefaults cdef = MarlinConfigDefaults.getDefaultInstance();
    props = new Properties();
    props.put("key.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    props.put("value.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    props.put(cdef.getParallelFlushersPerPartition(), false);
    props.put(cdef.getMetadataMaxAge(), 1000);  // want to exercise metadata refresher
    props.put(cdef.getBufferTime(), 500);  // flush quickly for testing

    try {
      madmin.deleteStream(STREAMFUTURES);
    } catch (Exception e) {}
    try {
      madmin.deleteStream(STREAMKEY);
    } catch (Exception e) {}
    try {
      madmin.deleteStream(STREAMAUTO);
    } catch (Exception e) {}
    try {
      madmin.deleteStream(STREAMNA);
    } catch (Exception e) {}
    try {
      madmin.deleteStream(STREAMMETA);
    } catch (Exception e) {}
    try {
      madmin.deleteStream(STREAMENOENT);
    } catch (Exception e) {}
    try {
      madmin.deleteStream(STREAMENOENTAUTO);
    } catch (Exception e) {}
    try {
      madmin.deleteStream(STREAMDELETE);
    } catch (Exception e) {}

    madmin.close();
  }

  @AfterClass
  public static void cleanupTest() throws Exception {
    Configuration conf = new Configuration();
    Admin madmin = Streams.newAdmin(conf);
    try {
      madmin.deleteStream(STREAMFUTURES);
      madmin.deleteStream(STREAMKEY);
      madmin.deleteStream(STREAMAUTO);
      madmin.deleteStream(STREAMNA);
      madmin.deleteStream(STREAMMETA);
      madmin.deleteStream(STREAMENOENT);
      madmin.deleteStream(STREAMENOENTAUTO);
      madmin.deleteStream(STREAMDELETE);
    } catch (Exception e) {}
    madmin.close();
  }

  @Test
  public void testStreamDelete() throws IOException {
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(1);
    sdesc.setAutoCreateTopics(false);

    Configuration conf = new Configuration();
    Admin madmin = Streams.newAdmin(conf);
    madmin.createStream(STREAMDELETE, sdesc);
    madmin.close();

    KafkaProducer producer = new KafkaProducer<byte[], byte[]>(props);

    // No autocreate, so should fail
    ProducerRecord<byte[], byte[]> record =
      new ProducerRecord<byte[], byte[]>(STREAMDELETE+":"+TOPIC, 0, key, value);
    TestCallback callback = new TestCallback(-1, true);
    Future<RecordMetadata> future  = producer.send(record, callback); 
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    madmin = Streams.newAdmin(conf);
    madmin.createTopic(STREAMDELETE, TOPIC, 1);
    madmin.close();

    // Wait for metadata refresher to collect the updated information
    try {
      Thread.sleep(2100);
    } catch (Exception e) {
      System.out.println(e);
    }

    // now topic exists, should succeed
    record = new ProducerRecord<byte[], byte[]>(STREAMDELETE+":"+TOPIC, 0, key, value);
    callback = new TestCallback(0, false);
    future  = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    madmin = Streams.newAdmin(conf);
    madmin.deleteStream(STREAMDELETE);
    madmin.close();

    // now should fail since stream deleted
    record = new ProducerRecord<byte[], byte[]>(STREAMDELETE+":"+TOPIC, 0, key, value);
    callback = new TestCallback(true);
    future  = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println("Failed as stream deleted " + e);
    }
    assertTrue(callback.verify());

    // now recreate with auto create on!
    sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(1);
    sdesc.setAutoCreateTopics(true);

    madmin = Streams.newAdmin(conf);
    madmin.createStream(STREAMDELETE, sdesc);
    madmin.close();

    // now topic and stream exists!, can fail because of stale stream data
    record = new ProducerRecord<byte[], byte[]>(STREAMDELETE+":"+TOPIC, 0, key, value);
    callback = new TestCallback(false);
    future  = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println("Failed on new stream " + e);
    }

    // Wait for metadata refresher to collect the updated information, but stale stream is not handled
    try {
      Thread.sleep(2100);
    } catch (Exception e) {
      System.out.println(e);
    }

    // should not fail!
    record = new ProducerRecord<byte[], byte[]>(STREAMDELETE+":"+TOPIC, 0, key, value);
    callback = new TestCallback(0, false);
    future  = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    producer.close();
  }

  @Test
  public void testFuturesAndInvalidTopicNames() throws IOException {
    try {
      StreamDescriptor sdesc = Streams.newStreamDescriptor();
      sdesc.setDefaultPartitions(1);
      sdesc.setAutoCreateTopics(true);

      Configuration conf = new Configuration();
      Admin madmin = Streams.newAdmin(conf);
      madmin.createStream(STREAMFUTURES, sdesc);
      madmin.close();
      // Topic doesn't exist, so producer send should auto create it with one partition.
      KafkaProducer producer = new KafkaProducer<byte[], byte[]>(props);

      Future[] futures = new Future[2];

      for (int i = 0; i < 2; i++) {
        String key = "key-value" + i;
        String msg = "msg-value" + i;
        ProducerRecord<byte[], byte[]> record =
          new ProducerRecord<byte[], byte[]>(STREAMFUTURES+":"+TOPIC,
                                             key.getBytes(),
                                             msg.getBytes());
        futures[i] = producer.send(record);
      }
      producer.flush();

      for (Future<RecordMetadata> future : futures) {
        future.get();
      }

      futures = new Future[5000];

      for (int i = 0; i < 5000; i++) {
        String key = "key-value" + i;
        String msg = "msg-value" + i;
        ProducerRecord<byte[], byte[]> record =
          new ProducerRecord<byte[], byte[]>(STREAMFUTURES+":"+TOPIC,
                                             key.getBytes(),
                                             msg.getBytes());
        futures[i] = producer.send(record);
      }
      producer.flush();

      for (Future<RecordMetadata> future : futures) {
        future.get();
      }

      // now try to send to topics with fullpath and non-fullpath
      String clusterName = null;
      try {
        MapRFileSystem fs = new MapRFileSystem();
        fs.initialize(new URI(MapRConstants.MAPRFS_PREFIX), conf);

        FileStatus contents[] = fs.listStatus(new Path("/mapr"));
        clusterName = contents[0].getPath().getName();
      } catch (Exception e) {
        System.out.println("Exception " + e + " occurred");
        throw e;
      }

      String fulltopicpath = "/mapr/" + clusterName +
                             STREAMFUTURES + ":" + "fullpath";
      String nonfullpath = STREAMFUTURES + ":" + "fullpath";

      futures = new Future[10000];

      for (int i = 0; i < 5000; i++) {
        String key = "key-value" + i;
        String msg = "msg-value" + i;
        ProducerRecord<byte[], byte[]> record =
          new ProducerRecord<byte[], byte[]>(fulltopicpath,
                                             key.getBytes(),
                                             msg.getBytes());
        futures[2*i] = producer.send(record);
        record =
          new ProducerRecord<byte[], byte[]>(nonfullpath,
                                             key.getBytes(),
                                             msg.getBytes());
        futures[2*i + 1] = producer.send(record);
      }
      producer.flush();

      long prevoffset = 0;
      for (Future<RecordMetadata> future : futures) {
        assert(prevoffset < future.get().offset());
        prevoffset = future.get().offset();
      }


      String invalidtopic = STREAMFUTURES + ":" + "@";  // "@" is invalid topic
      TestCallback testcallback = new TestCallback(-1, true);

      ProducerRecord<byte[], byte[]> record =
        new ProducerRecord<byte[], byte[]>(invalidtopic, 0,
                                           "key".getBytes(), "value".getBytes());
      System.out.println("Sending to invalid topic, " + invalidtopic);
      Future<RecordMetadata> invalidFuture = producer.send(record, testcallback);

      boolean exceptionCaught = false;
      try {
        invalidFuture.get();
      } catch (Exception e) {
        exceptionCaught = true;
      }
      assertTrue(exceptionCaught);
      assertTrue(testcallback.verify());

      invalidtopic = STREAMFUTURES;
      testcallback = new TestCallback(-1, true);
      System.out.println("Sending to invalid topic, " + invalidtopic);
      record = new ProducerRecord<byte[], byte[]>(invalidtopic, 0,
                                                  "key".getBytes(),
                                                  "value".getBytes());
      invalidFuture = producer.send(record, testcallback);

      exceptionCaught = false;
      try {
        invalidFuture.get();
      } catch (Exception e) {
        exceptionCaught = true;
      }
      assertTrue(exceptionCaught);
      assertTrue(testcallback.verify());

      String dottopic = STREAMFUTURES + ":" + "topic.dot"; // . is valid
      testcallback = new TestCallback(0, false);
      System.out.println("Sending to topic with . in its name, " + dottopic);
      record = new ProducerRecord<byte[], byte[]>(dottopic, 0,
                                                  "key".getBytes(),
                                                  "value".getBytes());
      Future validFuture = producer.send(record, testcallback);

      exceptionCaught = false;
      try {
        validFuture.get();
      } catch (Exception e) {
        exceptionCaught = true;
      }
      assertTrue(!exceptionCaught);
      assertTrue(testcallback.verify());

      producer.close();
    } catch (Exception e) {
      System.out.println(e);
    }
  }

  @Test
  public void testSendWithKey() throws IOException {
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(10);
    sdesc.setAutoCreateTopics(true);

    Configuration conf = new Configuration();
    Admin madmin = Streams.newAdmin(conf);
    madmin.createStream(STREAMKEY, sdesc);
    madmin.close();
    // Topic doesn't exist, so producer send should auto create it with one partition.
    KafkaProducer producer = new KafkaProducer<byte[], byte[]>(props);

    // send to key "abc"
    ProducerRecord<byte[], byte[]> record =
      new ProducerRecord<byte[], byte[]>(STREAMKEY+":"+TOPIC, key, value);
    TestCallback callback1 = new TestCallback(false);
    Future<RecordMetadata> future1 = producer.send(record, callback1);
    try {
      future1.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback1.verify());

    // send to key "ab"
    record = new ProducerRecord<byte[], byte[]>(STREAMKEY+":"+TOPIC, "ab".getBytes(), value);
    TestCallback callback2 = new TestCallback(false);
    Future<RecordMetadata> future2 = producer.send(record, callback2);
    try {
      future2.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback2.verify());

    // send to key "abc"
    record =
      new ProducerRecord<byte[], byte[]>(STREAMKEY+":"+TOPIC, key, value);
    TestCallback callback3 = new TestCallback(callback1.partitionid(), false);
    Future<RecordMetadata> future3 = producer.send(record, callback3);
    try {
      future3.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback3.verify());

    // send to key "ab"
    record =
      new ProducerRecord<byte[], byte[]>(STREAMKEY+":"+TOPIC, "ab".getBytes(), value);
    TestCallback callback4 = new TestCallback(callback2.partitionid(), false);
    Future<RecordMetadata> future4 = producer.send(record, callback4);
    try {
      future4.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback4.verify());

    producer.close();

    // just to check
    producer.flush();
    producer.close();
  }

  @Test
  public void testSendMessageAutoCreate() throws IOException {
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(1);
    sdesc.setAutoCreateTopics(true);

    Configuration conf = new Configuration();
    Admin madmin = Streams.newAdmin(conf);
    madmin.createStream(STREAMAUTO, sdesc);
    madmin.close();

    // Topic doesn't exist, so producer send should auto create it with one partition.
    KafkaProducer producer = new KafkaProducer<byte[], byte[]>(props);
    producer.close();

    // send message after close
    ProducerRecord<byte[], byte[]> record =
      new ProducerRecord<byte[], byte[]>(STREAMAUTO+":"+TOPIC, 0, key, value);
    Future<RecordMetadata> future = producer.send(record);
    boolean exceptionCaught = false;
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
      exceptionCaught = true;
    }
    assertTrue(exceptionCaught);

    producer = new KafkaProducer<byte[], byte[]>(props);

    // specify a message partition
    record = new ProducerRecord<byte[], byte[]>(STREAMAUTO+":"+TOPIC, 0, key, value);
    TestCallback callback = new TestCallback(0, false);
    future = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    List<PartitionInfo> partList = producer.partitionsFor(STREAMAUTO+":"+TOPIC);
    assertTrue(partList.size() == 1);
    assertTrue(partList.get(0).partition() == 0);

    // don't specify the partition id
    record = new ProducerRecord<byte[], byte[]>(STREAMAUTO+":"+TOPIC, key, value);
    callback = new TestCallback(0, false);
    future = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    // send to nonexisting partition
    record = new ProducerRecord<byte[], byte[]>(STREAMAUTO+":"+TOPIC, 1, key, value);
    callback = new TestCallback(-1, true);
    future = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    // send empty value
    record = new ProducerRecord<byte[], byte[]>(STREAMAUTO+":"+TOPIC, 1, key, null);
    callback = new TestCallback(-1, true);
    future = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    producer.close();

    // Just so that we always pass this.
    producer.flush();
    producer.close();

    // send a message after a close and see
    record = new ProducerRecord<byte[], byte[]>(STREAMAUTO+":"+TOPIC, 1, key, null);
    callback = new TestCallback(-1, true);
    exceptionCaught = false;
    future = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
      exceptionCaught = true;
    }
    assertTrue(exceptionCaught);

  }

  @Test
  public void testSendMessageNoAutoCreate() throws IOException {
    KafkaProducer producer = new KafkaProducer<byte[], byte[]>(props);

    // send before creating the stream
    ProducerRecord<byte[], byte[]> record =
      new ProducerRecord<byte[], byte[]>(STREAMNA+":"+TOPIC, 0, key, value);
    TestCallback callback = new TestCallback(-1, true);
    Future<RecordMetadata> future = producer.send(record, callback); 
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(1);
    sdesc.setAutoCreateTopics(false);

    Configuration conf = new Configuration();
    Admin madmin = Streams.newAdmin(conf);
    madmin.createStream(STREAMNA, sdesc);
    madmin.close();

    // send after creating stream
    record = new ProducerRecord<byte[], byte[]>(STREAMNA+":"+TOPIC, 0, key, value);
    callback = new TestCallback(-1, true);
    future = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    // send without partition id -- make sure it doesn't get anything back!
    record = new ProducerRecord<byte[], byte[]>(STREAMNA+":"+TOPIC, key, value);
    callback = new TestCallback(-1, true);
    future = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());
    producer.close();

    // Just so that we always pass this.
    producer.flush();
    producer.close();
  }

  @Test
  public void testMetadataRefresher() throws IOException {
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(1);
    sdesc.setAutoCreateTopics(false);

    Configuration conf = new Configuration();
    Admin madmin = Streams.newAdmin(conf);
    madmin.createStream(STREAMMETA, sdesc);
    madmin.close();

    KafkaProducer producer = new KafkaProducer<byte[], byte[]>(props);
    ProducerRecord<byte[], byte[]> record =
      new ProducerRecord<byte[], byte[]>(STREAMMETA+":"+TOPIC, 1, key, value);
    TestCallback callback = new TestCallback(-1, true);
    Future<RecordMetadata> future  = producer.send(record, callback); 
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    // create topic
    madmin = Streams.newAdmin(conf);
    madmin.createTopic(STREAMMETA, TOPIC, 1);
    madmin.close();

    // send a record again but with index > num partitions
    callback = new TestCallback(-1, true);
    record = new ProducerRecord<byte[], byte[]>(STREAMMETA+":"+TOPIC, 1, key, value);
    future = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    // change topic to have 3 feeds (0, 1, 2)
    madmin = Streams.newAdmin(conf);
    madmin.editTopic(STREAMMETA, TOPIC, 3);
    madmin.close();

    // Wait for metadata refresher to collect the updated information
    try {
      Thread.sleep(2100);
    } catch (Exception e) {
      System.out.println(e);
    }

    // send the same record again
    callback = new TestCallback(1, false);
    record =
      new ProducerRecord<byte[], byte[]>(STREAMMETA+":"+TOPIC, 1, key, value);
    future = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    // delete topic
    madmin = Streams.newAdmin(conf);
    madmin.deleteTopic(STREAMMETA, TOPIC);
    madmin.close();

    // Wait for metadata refresher to collect the updated information
    try {
      Thread.sleep(2100);
    } catch (Exception e) {
      System.out.println(e);
    }

    // send the same record again
    callback = new TestCallback(-1, true);
    record =
      new ProducerRecord<byte[], byte[]>(STREAMMETA+":"+TOPIC, 1, key, value);
    future = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    madmin = Streams.newAdmin(conf);
    madmin.createTopic(STREAMMETA, TOPIC, 2);
    madmin.close();

    callback = new TestCallback(1, false);
    record =
      new ProducerRecord<byte[], byte[]>(STREAMMETA+":"+TOPIC, 1, key, value);
    future = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    callback = new TestCallback(-1, true);
    record =
      new ProducerRecord<byte[], byte[]>(STREAMMETA+":"+TOPIC, 2, key, value);
    future = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    producer.close();

    // Just so that we always pass this.
    producer.flush();
    producer.close();
  }

  @Test
  public void testENOENTWithoutAutoCreateFromServer() throws IOException {
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(1);
    sdesc.setAutoCreateTopics(false);

    Configuration conf = new Configuration();
    Admin madmin = Streams.newAdmin(conf);
    madmin.createStream(STREAMENOENT, sdesc);
    madmin.close();

    MarlinConfigDefaults cdef = MarlinConfigDefaults.getDefaultInstance();
    Properties myprops = new Properties();
    myprops.put("key.serializer",
                "org.apache.kafka.common.serialization.ByteArraySerializer");
    myprops.put("value.serializer",
                "org.apache.kafka.common.serialization.ByteArraySerializer");
    myprops.put(cdef.getParallelFlushersPerPartition(), true);
    myprops.put(cdef.getMetadataMaxAge(), 5*60*1000);  // don't want refresher to refresh
    myprops.put(cdef.getBufferTime(), 3000);

    // create topic
    madmin = Streams.newAdmin(conf);
    madmin.createTopic(STREAMENOENT, TOPIC, 1);
    madmin.close();

    KafkaProducer producer = new KafkaProducer<byte[], byte[]>(myprops);
    ProducerRecord<byte[], byte[]> record =
      new ProducerRecord<byte[], byte[]>(STREAMENOENT+":"+TOPIC, 0, key, value);
    TestCallback callback = new TestCallback(0, false);
    Future<RecordMetadata> future  = producer.send(record, callback); 
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    madmin = Streams.newAdmin(conf);
    madmin.deleteTopic(STREAMENOENT, TOPIC);
    madmin.close();

    try {
      Thread.sleep(2100); // Wait for the topic delete to go through
    } catch (Exception e) {
      System.out.println(e);
    }

    // send a record again -- this record would have landed in the partition 0 queue,
    // but will be rejected by ENOENT.  So, we will have an exception (true) but still
    // have a valid partition of 0.
    callback = new TestCallback(0, true);
    record = new ProducerRecord<byte[], byte[]>(STREAMENOENT+":"+TOPIC, 0, key, value);
    future = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    try {
      Thread.sleep(2100); // Wait for the topic delete to go through
    } catch (Exception e) {
      System.out.println(e);
    }

    // Make sure that the topic is still deleted!
    boolean exceptionCaught = false;
    MarlinTopicMetaEntry topicEntry = null;

    madmin = Streams.newAdmin(conf);
    try {
      MarlinAdminImpl admin = (MarlinAdminImpl)madmin;
      topicEntry = admin.getTopicMetaEntry(STREAMENOENT+":"+TOPIC);
    } catch (Exception e) {
      exceptionCaught = true;
    }
    madmin.close();

    assertTrue(exceptionCaught || topicEntry.getIsDeleted());

    // send a record again -- this record would not land in any partition queue, since
    // topic is deleted and will not be accepted.
    callback = new TestCallback(-1, true);
    record = new ProducerRecord<byte[], byte[]>(STREAMENOENT+":"+TOPIC, 0, key, value);
    future = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    producer.flush();
    producer.close();
  }


  @Test
  public void testENOENTWithAutoCreateFromServer() throws IOException {
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(1);
    sdesc.setAutoCreateTopics(true);

    Configuration conf = new Configuration();
    Admin madmin = Streams.newAdmin(conf);
    madmin.createStream(STREAMENOENTAUTO, sdesc);
    madmin.close();


    MarlinConfigDefaults cdef = MarlinConfigDefaults.getDefaultInstance();
    Properties myprops = new Properties();
    myprops.put("key.serializer",
                "org.apache.kafka.common.serialization.ByteArraySerializer");
    myprops.put("value.serializer",
                "org.apache.kafka.common.serialization.ByteArraySerializer");
    myprops.put(cdef.getParallelFlushersPerPartition(), true);
    myprops.put(cdef.getMetadataMaxAge(), 5*60*1000);  // don't want refresher to refresh
    myprops.put(cdef.getBufferTime(), 3000);

    KafkaProducer producer = new KafkaProducer<byte[], byte[]>(myprops);
    ProducerRecord<byte[], byte[]> record =
      new ProducerRecord<byte[], byte[]>(STREAMENOENTAUTO+":"+TOPIC, 0, key, value);
    TestCallback callback = new TestCallback(0, false);
    Future<RecordMetadata> future  = producer.send(record, callback); 
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    // Get the topic unique number!
    boolean exceptionCaught = false;
    MarlinTopicMetaEntry topicEntryFirst = null;
    madmin = Streams.newAdmin(conf);
    MarlinAdminImpl admin = (MarlinAdminImpl)madmin;
    try {
      topicEntryFirst = admin.getTopicMetaEntry(STREAMENOENTAUTO+":"+TOPIC);
    } catch (Exception e) {
      exceptionCaught = true;
    }
    madmin.close();
    assertFalse(exceptionCaught);

    madmin = Streams.newAdmin(conf);
    madmin.deleteTopic(STREAMENOENTAUTO, TOPIC);
    madmin.close();

    try {
      Thread.sleep(2100); // Wait for the topic delete to go through
    } catch (Exception e) {
      System.out.println(e);
    }

    // send a record.  Topic is deleted, so we should get a ENOENT.  At this point,
    // producer does a topic metadata refresh, but this topic is done for. :(
    callback = new TestCallback(0, true);
    record = new ProducerRecord<byte[], byte[]>(STREAMENOENTAUTO+":"+TOPIC, 0, key, value);
    future = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    try {
      Thread.sleep(2100); // Wait for the topic delete to go through
    } catch (Exception e) {
      System.out.println(e);
    }

    // send a record again -- producer topic metadata should have been updated,
    // and marked the topic as deleted.  Now, producer should recreate the topic.
    callback = new TestCallback(0, false);
    record = new ProducerRecord<byte[], byte[]>(STREAMENOENTAUTO+":"+TOPIC, 0, key, value);
    future = producer.send(record, callback);
    try {
      future.get();
    } catch (Exception e) {
      System.out.println(e);
    }
    assertTrue(callback.verify());

    // Now get the second topic's unique number!
    exceptionCaught = false;
    MarlinTopicMetaEntry topicEntrySecond = null;

    madmin = Streams.newAdmin(conf);
    try {
      MarlinAdminImpl iadmin = (MarlinAdminImpl)madmin;
      topicEntrySecond = iadmin.getTopicMetaEntry(STREAMENOENTAUTO+":"+TOPIC);
    } catch (Exception e) {
      exceptionCaught = true;
    }
    madmin.close();
    assertFalse(exceptionCaught);

    assertTrue(topicEntryFirst.getTopicUniq() != topicEntrySecond.getTopicUniq());

    producer.flush();
    producer.close();
  }


  private static final class TestCallback implements Callback {
    private boolean error;
    private int expectedFeedID;
    private Exception exceptionReceived;
    private RecordMetadata metadataReceived;
    private boolean callbackCompleted;
    private boolean checkfeedID;

    public TestCallback(boolean errors) {
      this.checkfeedID = false;
      this.error = errors;
      this.callbackCompleted = false;
    }
    public TestCallback(int feedid, boolean errors) {
      this.checkfeedID = true;
      this.expectedFeedID = feedid;
      this.error = errors;
      this.callbackCompleted = false;
    }

    public void onCompletion(RecordMetadata metadata,
                             Exception exception) {
      exceptionReceived = exception;
      metadataReceived = metadata;
      synchronized(this) {
        this.callbackCompleted = true;
        try {
          this.notifyAll();
        } catch (Exception e) {
          System.out.println(e);
        }
      }
    }

    public int partitionid() {
      return metadataReceived.partition();
    }

    public boolean verify() {
      synchronized(this) {
        if (!callbackCompleted) {
          try {
            this.wait();
          } catch (Exception e) {
            System.out.println(e);
          }
        }
      }
      boolean verified = true;
      if (error && exceptionReceived == null) {
        System.out.println("Did not get exception when expected");
        verified = false;
      } else if (!error && exceptionReceived != null) {
        System.out.println("Received exception " + exceptionReceived + " but expected none");
        verified = false;
      }

      if (checkfeedID && expectedFeedID != metadataReceived.partition()) {
        System.out.println("Received partition " + metadataReceived.partition() + " but expected " + expectedFeedID);
        verified = false;
      }

      return verified;
    }
  }
}
