/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.streams.tests.producer;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.List;

import org.apache.hadoop.conf.Configuration;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.streams.Admin;
import com.mapr.streams.StreamDescriptor;
import com.mapr.streams.Streams;

import com.mapr.streams.producer.ProducerPerformance;
import com.mapr.tests.BaseTest;
import com.mapr.tests.annotations.ClusterTest;

@Category(ClusterTest.class)
public class ProducerBasicTest extends BaseTest {
  private static final Logger _logger = LoggerFactory.getLogger(ProducerBasicTest.class);
  private static final String STREAM = "/jtest-" + ProducerBasicTest.class.getSimpleName();
  private static Admin madmin;

  private static final int numParts = 1;

  @BeforeClass
  public static void setupTest() throws Exception {
    final Configuration conf = new Configuration();
    madmin = Streams.newAdmin(conf);

    try{
      madmin.deleteStream(STREAM);
    } catch (Exception e) {}

    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(numParts);
    madmin.createStream(STREAM, sdesc);
  }

  @AfterClass
  public static void cleanupTest() throws Exception {
    madmin.deleteStream(STREAM);
  }

  @Test
  public void testSendOneSmallMessage() throws IOException {
    assertTrue(ProducerPerformance.runBasicTest(STREAM,
                                                1 /*nmsgs per partition */,
                                                1 /*ntopics*/,
                                                0 /*nslowtopics */,
                                                numParts /*npartitions*/,
                                                1 /*nbatches*/,
                                                10*1000 /*ms between batches*/,
                                                true /* multiple flushers */,
                                                false /* progress */,
                                                200 /* msg size */,
                                                32*1024*1024 /* pool size */,
                                                5*60*1000 /* metadata refresh*/));
  }

  @Test
  public void testSendOneLargeMessage() throws IOException {
    assertTrue(ProducerPerformance.runBasicTest(STREAM,
                                                1 /*nmsgs per partition */,
                                                1 /*ntopics*/,
                                                0 /*nslowtopics */,
                                                numParts /*npartitions*/,
                                                1 /*nbatches*/,
                                                10*1000 /*ms between batches*/,
                                                true /*multiple flushers */,
                                                false /* progress */,
                                                2*1024*1024 /* msg size */,
                                                1*1024*1024 /* pool size */,
                                                5*60*1000 /* metadata refresh*/));
  }

  @Test
  public void testSendOneMediumMessage() throws IOException {
    /* msg is 90% of the pool size ( > threshold, and < pool max size) */
    assertTrue(ProducerPerformance.runBasicTest(STREAM,
                                                1 /*nmsgs per partition */,
                                                1 /*ntopics*/,
                                                0 /*nslowtopics */,
                                                numParts /*npartitions*/,
                                                1 /*nbatches*/,
                                                10*1000 /*ms between batches*/,
                                                true /*multiple flushers */,
                                                false /* progress */,
                                                9*1024*1024/10 /* msg size */,
                                                1*1024*1024 /* pool size */,
                                                5*60*1000 /* metadata refresh*/));
  }

  @Test
  public void testSendThousandTopics() throws IOException {
    assertTrue(ProducerPerformance.runBasicTest(STREAM,
                                                1000 /*nmsgs per partition */,
                                                1000 /*ntopics*/,
                                                0 /*nslowtopics */,
                                                numParts /*npartitions*/,
                                                1 /*nbatches*/,
                                                10*1000 /*ms between batches*/,
                                                true /*multiple flushers */,
                                                true /* progress */,
                                                200 /* msg size */,
                                                32*1024*1024 /* pool size */,
                                                5*60*1000 /* metadata refresh*/));
  }

  @Test
  public void testSendThousandTopicsWithRefresh() throws IOException {
    assertTrue(ProducerPerformance.runBasicTest(STREAM,
                                                1000 /*nmsgs per partition */,
                                                1011 /*ntopics*/,
                                                0 /*nslowtopics */,
                                                numParts /*npartitions*/,
                                                1 /*nbatches*/,
                                                10*1000 /*ms between batches*/,
                                                true /*multiple flushers */,
                                                true /* progress */,
                                                200 /* msg size */,
                                                32*1024*1024 /* pool size */,
                                                1000 /* metadata refresh*/));
  }


  @Test
  public void testSendTenTopicsMediumMessages() throws IOException {
    assertTrue(ProducerPerformance.runBasicTest(STREAM,
                                                100 /*nmsgs per partition */,
                                                10 /*ntopics*/,
                                                0 /*nslowtopics */,
                                                numParts /*npartitions*/,
                                                1 /*nbatches*/,
                                                10*1000 /*ms between batches*/,
                                                true /*multiple flushers */,
                                                true /* progress */,
                                                9*1024*1024/10 /* msg size */,
                                                1*1024*1024 /* pool size */,
                                                5*60*1000 /* metadata refresh*/));
  }

  @Test
  public void testSendTenTopicsLargeMessages() throws IOException {
    assertTrue(ProducerPerformance.runBasicTest(STREAM,
                                                100 /*nmsgs per partition */,
                                                10 /*ntopics*/,
                                                0 /*nslowtopics */,
                                                numParts /*npartitions*/,
                                                1 /*nbatches*/,
                                                10*1000 /*ms between batches*/,
                                                true /*multiple flushers */,
                                                true /* progress */,
                                                2*1024*1024 /* msg size */,
                                                1*1024*1024 /* pool size */,
                                                5*60*1000 /* metadata refresh*/));
  }

  @Test
  public void testMixedSpeedTopics() throws IOException {
    assertTrue(ProducerPerformance.runBasicTest(STREAM,
                                                10000 /*nmsgs per partition */,
                                                100 /*ntopics*/,
                                                100 /*nslowtopics */,
                                                numParts /*npartitions*/,
                                                1 /*nbatches*/,
                                                10*1000 /*ms between batches*/,
                                                true /*multiple flushers */,
                                                true /* progress */,
                                                200 /* msg size */,
                                                32*1024*1024 /* pool size */,
                                                5*60*1000 /* metadata refresh*/));
  }



}
