/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.streams.tests.ojai;

import static org.junit.Assert.assertEquals;

import java.io.IOException;
import java.util.Iterator;

import org.apache.hadoop.conf.Configuration;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.ojai.Document;
import org.ojai.DocumentConstants;
import org.ojai.DocumentStream;
import org.ojai.store.QueryCondition;
import org.ojai.store.QueryCondition.Op;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.db.MapRDB;
import com.mapr.db.Table;
import com.mapr.streams.Admin;
import com.mapr.streams.demo.drill.Producer;
import com.mapr.streams.impl.StreamsDocument;
import com.mapr.streams.impl.StreamsDocumentTranslator;
import com.mapr.streams.StreamDescriptor;
import com.mapr.streams.Streams;
import com.mapr.tests.BaseTest;
import com.mapr.tests.annotations.ClusterTest;

@Category(ClusterTest.class)
public class TestStreamScanWithCondition extends BaseTest {
  private static final Logger _logger = LoggerFactory.getLogger(TestStreamScanWithCondition.class);

  private static final int NUM_PARTITION = 10;
  private static final String TMP_TEST_SCAN_WITH_CONDITION_STREAM = "/tmp/TestScanWithCondition_stream";

  private static Admin admin;

  @BeforeClass
  public static void setupTests_ScanWithCondition() throws IOException {
    admin = Streams.newAdmin(new Configuration());
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(NUM_PARTITION);
    try {
      admin.deleteStream(TMP_TEST_SCAN_WITH_CONDITION_STREAM);
    } catch (Exception e) {}
    admin.createStream(TMP_TEST_SCAN_WITH_CONDITION_STREAM, sdesc);

    Producer p = new Producer(TMP_TEST_SCAN_WITH_CONDITION_STREAM, NUM_PARTITION, 50000);
    p.run();
  }

  @AfterClass
  public static void cleanupTests_ScanWithCondition() {
    try {
      if (admin != null) {
        admin.deleteStream(TMP_TEST_SCAN_WITH_CONDITION_STREAM);
      }
    } catch (Exception e) {}
  }

  @Test
  public void testScanWithCondition() throws Exception {
    try (Table t = MapRDB.getTable(TMP_TEST_SCAN_WITH_CONDITION_STREAM);) {
      QueryCondition splitCond = MapRDB.newCondition()
          .or()
            .and()
              .is(DocumentConstants.ID_KEY, Op.GREATER_OR_EQUAL, "p001")
              .is(DocumentConstants.ID_KEY, Op.LESS, "p002")
            .close()
            .and()
              .is(DocumentConstants.ID_KEY, Op.GREATER_OR_EQUAL, "p003")
              .is(DocumentConstants.ID_KEY, Op.LESS, "p004")
            .close()
          .close()
        .build();

      QueryCondition c = MapRDB.newCondition()
          .and()
            .is(DocumentConstants.ID_KEY, Op.GREATER_OR_EQUAL, "p001tUS_SOUTH:000000010000000000000000")
            .is(DocumentConstants.ID_KEY, Op.LESS, "p001tUS_SOUTH:000000017fffffffffffffff")
          .close()
        .build();

      QueryCondition prodCond = MapRDB.newCondition()
          .is("p", Op.EQUAL, "root")
        .build();

      QueryCondition fullCond = MapRDB.newCondition()
          .and()
            .condition(splitCond)
            .condition(c)
            .condition(prodCond)
          .close()
        .build();

      _logger.info("Full Condition: {}", fullCond);
      DocumentStream rs = t.find(fullCond);
      Iterator<Document> iter = rs.iterator();
      while (iter.hasNext()) {
        Document rec = iter.next();
        StreamsDocumentTranslator translator = new StreamsDocumentTranslator(rec, null/*projectedPathIds*/);
        while (translator.hasNext()) {
          StreamsDocument r = (StreamsDocument) translator.next();
          _logger.info("{}", r);
          assertEquals(1, r.getInt("partition"));
          assertEquals("US_SOUTH", r.getString("topic"));
          assertEquals("root", r.getString("producer"));
        }
      }
    }
  }

}
