/* Copyright (c) 2017 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.streams.tests.listener;
import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.lang.RuntimeException;
import java.nio.ByteBuffer;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.ArrayList;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.Map;
import java.util.concurrent.locks.*;

import org.apache.hadoop.conf.Configuration;
import org.apache.kafka.clients.consumer.NoOffsetForPartitionException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.RecordTooLargeException;
import org.apache.kafka.common.errors.UnknownTopicOrPartitionException;

import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.streams.Admin;
import com.mapr.streams.StreamDescriptor;
import com.mapr.streams.Streams;
import com.mapr.streams.producer.Producer;
import com.mapr.streams.listener.Listener;
import com.mapr.streams.listener.Listener.TED_ACTION;
import com.mapr.streams.tests.producer.ProducerMultiTest.CountCallback;
import com.mapr.streams.tests.producer.SendMessagesToProducer;
import com.mapr.streams.impl.admin.TopicFeedInfo;
import com.mapr.streams.impl.admin.MarlinAdminImpl;

import com.mapr.tests.BaseTest;
import com.mapr.tests.annotations.ClusterTest;

import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.producer.Callback;
import org.apache.kafka.clients.producer.KafkaProducer;

import java.util.Arrays;
import java.util.concurrent.TimeUnit;

@Category(ClusterTest.class)
public class OffsetsGapTest extends BaseTest {
 private static final Logger _logger = LoggerFactory.getLogger(OffsetsGapTest.class);
 private static final String STREAM = "/jtest-" + OffsetsGapTest.class.getSimpleName();
 private static Admin madmin;

 private static final int numParts = 1;
 private static final int defaultNumParts = 5;
 private static final int numMsgs = 1000000;
 private static final int numTopics = 1;
 private String topic= "t";
 private String topicName = ":t";
 private KafkaProducer kafkaProducer;
 private KafkaConsumer kafkaConsumer;
 private boolean produceDone = false;

 @BeforeClass
  public static void setupTestClass() throws Exception {
    Configuration conf = new Configuration();
    madmin = Streams.newAdmin(conf);

    try {
      madmin.deleteStream(STREAM);
    } catch (Exception e) {}
  }

  @Before
  public void setupTest() throws Exception {
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(defaultNumParts);
    madmin.createStream(STREAM, sdesc);
    // Now setup copy table.
    ProcessBuilder proc = new ProcessBuilder("maprcli", "table", "edit", "-path", STREAM, "-regionsizemb", "256");
    Process process = proc.start();
    int errorCode = process.waitFor();
    _logger.info("Set region size, completed with return code " + errorCode);

    //Producer
    Properties props = new Properties();
    props.put("key.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    props.put("value.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    kafkaProducer = new KafkaProducer<byte[], byte[]>(props);

    //Listener
    Properties listenerProps = new Properties();
    listenerProps.put("key.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    listenerProps.put("value.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    listenerProps.put("fetch.min.bytes", "1");
    listenerProps.put("auto.offset.reset", "earliest");
    listenerProps.put("group.id", "mygroup");
    kafkaConsumer = new KafkaConsumer<byte[], byte[]>(listenerProps);
  }

  @After
  public void cleanupTest() throws Exception {
    madmin.deleteStream(STREAM);
  }

  @Test
  public void testGapsOnTabletSplit() throws Exception {
    CountCallback cb = new CountCallback(numMsgs * numParts);
    SendMessagesToProducer producer = new SendMessagesToProducer(kafkaProducer, cb,
                                                                 STREAM + topicName, numParts, numMsgs, 20000 /*kv size*/);
    producer.run();

    List<TopicPartition> topicPartitionList = new ArrayList<TopicPartition>();

    for (int i = 0; i < numParts; i++) {
      topicPartitionList.add(new TopicPartition(STREAM + topicName, i));
    }

    kafkaConsumer.assign(topicPartitionList);

    for (TopicPartition tp : topicPartitionList) {
      kafkaConsumer.seekToEnd(tp);
      long endPos = kafkaConsumer.position(tp);
      _logger.info("endPos " + endPos + "numMsgs " + numMsgs);
      assertTrue(endPos == numMsgs);
    }
  }

  @Test
  public void testSeekDuringTabletSplit() throws Exception {
    Lock lock = new ReentrantLock();
    madmin.createTopic(STREAM, topic);
    Thread seeker = new Thread(new Seeker(lock));
    seeker.start();
    CountCallback cb = new CountCallback(numMsgs * numParts);
    SendMessagesToProducer producer = new SendMessagesToProducer(kafkaProducer, cb,
                                                                 STREAM + topicName, numParts, numMsgs, 20000 /*kv size*/);
    producer.run();
    lock.lock();
    produceDone = true;
    lock.unlock();
    seeker.join();

  }

  private class Seeker implements Runnable {
    List<TopicPartition> topicPartitionList;
    Lock lock;
    public Seeker(Lock l) {
      topicPartitionList = new ArrayList<TopicPartition>();
      for (int i = 0; i < numParts; i++) {
        topicPartitionList.add(new TopicPartition(STREAM + topicName, i));
      }
      lock = l;
    }

    @Override
    public void run() {
    kafkaConsumer.assign(topicPartitionList);
      while (true) {
        for (TopicPartition tp : topicPartitionList) {
          kafkaConsumer.seekToEnd(tp);
          long endPos = kafkaConsumer.position(tp);
          _logger.info("endPos " + endPos);
        }
        lock.lock();
        if (produceDone) {
          lock.unlock();
          for (TopicPartition tp : topicPartitionList) {
            kafkaConsumer.seekToEnd(tp);
            long endPos = kafkaConsumer.position(tp);
            _logger.info("endPos " + endPos);
            assertTrue(endPos == numMsgs);
          }
          return;
        }
        lock.unlock();
      }
    }
  }

  @Test
  public void testPollOnTabletSplit() throws Exception {
    CountCallback cb = new CountCallback(numMsgs * numParts);
    SendMessagesToProducer producer = new SendMessagesToProducer(kafkaProducer, cb,
                                                                 STREAM + topicName, numParts, numMsgs, 20000 /*kv size*/);
    producer.run();

    List<TopicPartition> topicPartitionList = new ArrayList<TopicPartition>();

    for (int i = 0; i < numParts; i++) {
      topicPartitionList.add(new TopicPartition(STREAM + topicName, i));
    }

    kafkaConsumer.assign(topicPartitionList);

    int numRecs = 0;
    long lastOffset = 0L;
    boolean outerBreak = false;
    while (outerBreak == false) {
      ConsumerRecords<String, byte[]> recs = kafkaConsumer.poll(100);
      for (ConsumerRecord<String, byte[]> rec : recs) {
        numRecs++;
        if (numRecs == numMsgs) {
          lastOffset = rec.offset();
          outerBreak = true;
          break;
        }
      }
    }
    assertTrue(lastOffset == numMsgs - 1);
  }

  @Test
  public void testSeekAfterRestart() throws Exception {
    int localNumMsgs = 100;
    CountCallback cb = new CountCallback(localNumMsgs * numParts);
    SendMessagesToProducer producer = new SendMessagesToProducer(kafkaProducer, cb,
                                                                 STREAM + topicName, numParts, localNumMsgs, 20000 /*kv size*/);
    producer.run();

    ProcessBuilder proc = new ProcessBuilder("sudo", "service", "mapr-warden", "restart");
    Process process = proc.start();
    int errorCode = process.waitFor();




    List<TopicPartition> topicPartitionList = new ArrayList<TopicPartition>();

    for (int i = 0; i < numParts; i++) {
      topicPartitionList.add(new TopicPartition(STREAM + topicName, i));
    }

    kafkaConsumer.assign(topicPartitionList);

    for (TopicPartition tp : topicPartitionList) {
      kafkaConsumer.seekToEnd(tp);
      long endPos = kafkaConsumer.position(tp);
      _logger.info("endPos " + endPos + "localNumMsgs " + localNumMsgs);
      assertTrue(endPos == localNumMsgs);
    }
  }

  @Test
  public void testSeekAfterBucketSwitch() throws Exception {
    int localNumMsgs = 10000;
    CountCallback cb = new CountCallback(localNumMsgs * numParts);
    ProcessBuilder proc = new ProcessBuilder("/opt/mapr/server/mrconfig", "set", "config", "streams.saveseq.maxmsgs", "8193");
    Process process = proc.start();
    int errorCode = process.waitFor();
    _logger.info("Set streams.saveseq.maxmsgs, completed with return code " + errorCode);

    SendMessagesToProducer producer = new SendMessagesToProducer(kafkaProducer, cb,
                                                                 STREAM + topicName, numParts, localNumMsgs, 100 /*kv size*/);
    producer.run();

    proc = new ProcessBuilder("/opt/mapr/server/mrconfig", "set", "config", "streams.saveseq.maxmsgs", "524288");
    process = proc.start();
    errorCode = process.waitFor();
    _logger.info("Set streams.saveseq.maxmsgs, completed with return code " + errorCode);

    List<TopicPartition> topicPartitionList = new ArrayList<TopicPartition>();

    for (int i = 0; i < numParts; i++) {
      topicPartitionList.add(new TopicPartition(STREAM + topicName, i));
    }

    kafkaConsumer.assign(topicPartitionList);

    for (TopicPartition tp : topicPartitionList) {
      kafkaConsumer.seekToEnd(tp);
      long endPos = kafkaConsumer.position(tp);
      _logger.info("endPos " + endPos + "localNumMsgs " + localNumMsgs);
      assertTrue(endPos == localNumMsgs);
    }
  }

  @Test
  public void testListTopicsAfterRestart() throws Exception {
    int localNumMsgs = 100;
    int localNumParts = 5;
    CountCallback cb = new CountCallback(localNumMsgs * numParts);
    SendMessagesToProducer producer = new SendMessagesToProducer(kafkaProducer, cb,
                                                                 STREAM + topicName, localNumParts, localNumMsgs, 20000 /*kv size*/);
    producer.run();

    ProcessBuilder proc = new ProcessBuilder("sudo", "service", "mapr-warden", "restart");
    Process process = proc.start();
    int errorCode = process.waitFor();

    MarlinAdminImpl admin = (MarlinAdminImpl) madmin;
    Map<String, List<TopicFeedInfo>> tfMap = admin.listTopics(STREAM);
    assertTrue(tfMap.isEmpty() == false);
    for (List<TopicFeedInfo> feedInfoList : tfMap.values()) {
      for (TopicFeedInfo fi : feedInfoList) {
        _logger.info("statmaxseq " + fi.stat().getMaxSeq() + "localNumMsgs " + localNumMsgs);
        assertTrue(fi.stat().getMaxSeq() == localNumMsgs - 1);
      }
    }
  }
}
