/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.streams.tests.listener;

import static org.junit.Assert.assertTrue;
import java.io.IOException;
import java.util.*;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.regex.Pattern;

import org.apache.hadoop.conf.Configuration;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRebalanceListener;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;

import com.mapr.tests.BaseTest;
import com.mapr.tests.annotations.ClusterTest;

import com.mapr.streams.Admin;
import com.mapr.streams.Streams;
import com.mapr.streams.StreamDescriptor;
import com.mapr.fs.proto.Marlinserver.MarlinConfigDefaults;


@Category(ClusterTest.class)
public class ListenerRegexTest extends BaseTest {
  private static final Logger _logger = LoggerFactory.getLogger(ListenerRegexTest.class);
  private static final String STREAM = "/jtest-" + ListenerRegexTest.class.getSimpleName();
  private static final String STREAMLG = STREAM + "-LG";
  private static final String STREAMFP = STREAM + "-FP";
  private static final String STREAMFPLG = STREAM + "-FPLG";
  private static final String STREAMOVERLAP = STREAM + "-OVER";
  private static final String STREAMDOT = STREAM + "-DOT";
  private static final String STREAMOR = STREAM + "-OR";
  private static final String STREAMTWOLISTENER = STREAM + "-TWO";
  private static Admin madmin;
  private static final int numParts = 1;

  @BeforeClass
  public static void setupTestClass() throws Exception {
    final Configuration conf = new Configuration();
    madmin = Streams.newAdmin(conf);

    //Cleanup all stale streams
    try {
      madmin.deleteStream(STREAM);
    } catch (Exception e) {}
    try {
      madmin.deleteStream(STREAMLG);
    } catch (Exception e) {}
    try {
      madmin.deleteStream(STREAMFP);
    } catch (Exception e) {}
    try {
      madmin.deleteStream(STREAMFPLG);
    } catch (Exception e) {}
    try {
      madmin.deleteStream(STREAMOVERLAP);
    } catch (Exception e) {}
    try {
      madmin.deleteStream(STREAMDOT);
    } catch (Exception e) {}
    try {
      madmin.deleteStream(STREAMOR);
    } catch (Exception e) {}
    try {
      madmin.deleteStream(STREAMTWOLISTENER);
    } catch (Exception e) {}

    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(numParts);
    madmin.createStream(STREAM, sdesc);
    madmin.createStream(STREAMLG, sdesc);
    madmin.createStream(STREAMFP, sdesc);
    madmin.createStream(STREAMFPLG, sdesc);
    madmin.createStream(STREAMOVERLAP, sdesc);
    madmin.createStream(STREAMDOT, sdesc);
    madmin.createStream(STREAMOR, sdesc);
    madmin.createStream(STREAMTWOLISTENER, sdesc);
  }

  @AfterClass
  public static void cleanupTestClass() throws Exception {
    madmin.deleteStream(STREAM);
    madmin.deleteStream(STREAMLG);
    madmin.deleteStream(STREAMFP);
    madmin.deleteStream(STREAMFPLG);
    madmin.deleteStream(STREAMOVERLAP);
    madmin.deleteStream(STREAMDOT);
    madmin.deleteStream(STREAMOR);
    madmin.deleteStream(STREAMTWOLISTENER);
  }

  @Test
  public void testTopicOverlapRegexSubscription() throws IOException {
    Properties props = new Properties();
    MarlinConfigDefaults cdef = MarlinConfigDefaults.getDefaultInstance();
    props.put("key.deserializer",
              "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    props.put("value.deserializer",
              "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    props.put("auto.offset.reset", "earliest");
    props.put(cdef.getMetadataMaxAge(), 3000);

    KafkaConsumer consumer = new KafkaConsumer<byte[], byte[]>(props);

    madmin.createTopic(STREAMOVERLAP, "topic0", 1);
    madmin.createTopic(STREAMOVERLAP, "topic1", 1);
    madmin.createTopic(STREAMOVERLAP, "topic2", 1);
    madmin.createTopic(STREAMOVERLAP, "topic3", 1);
    madmin.createTopic(STREAMOVERLAP, "topic4", 1);
    madmin.createTopic(STREAMOVERLAP, "topic1A", 1);
    madmin.createTopic(STREAMOVERLAP, "topic22", 1);

    Set<String> regexMatchSet = new HashSet<String>();
    regexMatchSet.add(STREAMOVERLAP+":topic0");
    regexMatchSet.add(STREAMOVERLAP+":topic1");
    regexMatchSet.add(STREAMOVERLAP+":topic2");
    Pattern regex = Pattern.compile(STREAMOVERLAP+":topic[0-2]$");
    consumer.subscribe(regex, null);
    Set<TopicPartition> subscribeList = consumer.assignment();
    Set<String> subscribedSet = consumer.subscription();
    Iterator<TopicPartition> iter = subscribeList.iterator();
    while(iter.hasNext()) {
      System.out.println("subscribe topic[0-2]$: " + iter.next());
    }
    assertTrue(subscribeList.size() == 3);
    assertTrue(subscribedSet.equals(regexMatchSet));

    consumer.poll(100);

    // the next subscribe should overwrite the previous subscribe
    regexMatchSet.clear();
    regexMatchSet.add(STREAMOVERLAP+":topic1");
    regexMatchSet.add(STREAMOVERLAP+":topic2");
    regexMatchSet.add(STREAMOVERLAP+":topic3");
    regexMatchSet.add(STREAMOVERLAP+":topic4");
    regex = Pattern.compile(STREAMOVERLAP+":topic[1-4]$");
    consumer.subscribe(regex, null);
    subscribeList = consumer.assignment();
    subscribedSet = consumer.subscription();
    iter = subscribeList.iterator();
    while(iter.hasNext()) {
      System.out.println("subscribe topic[1-4]$: " + iter.next());
    }
    assertTrue(subscribeList.size() == 4);
    assertTrue(subscribedSet.equals(regexMatchSet));

    consumer.poll(100);

    consumer.unsubscribe();
    subscribeList = consumer.assignment();
    subscribedSet = consumer.subscription();
    iter = subscribeList.iterator();
    while(iter.hasNext()) {
      System.out.println("unsubscribe: " + iter.next());
    }
    assertTrue(subscribeList.size() == 0);
    assertTrue(subscribedSet.isEmpty());

    consumer.poll(100);

    regexMatchSet.clear();
    regexMatchSet.add(STREAMOVERLAP+":topic0");
    regexMatchSet.add(STREAMOVERLAP+":topic1");
    regexMatchSet.add(STREAMOVERLAP+":topic2");
    regex = Pattern.compile(STREAMOVERLAP+":topic[0-2]$");
    consumer.subscribe(regex, null);
    subscribeList = consumer.assignment();
    subscribedSet = consumer.subscription();
    iter = subscribeList.iterator();
    while(iter.hasNext()) {
      System.out.println("subscribe topic[0-2]$: " + iter.next());
    }
    assertTrue(subscribeList.size() == 3);
    assertTrue(subscribedSet.equals(regexMatchSet));

    consumer.poll(100);

    regexMatchSet.clear();
    regexMatchSet.add(STREAMOVERLAP+":topic0");
    regexMatchSet.add(STREAMOVERLAP+":topic1");
    regexMatchSet.add(STREAMOVERLAP+":topic2");
    regexMatchSet.add(STREAMOVERLAP+":topic3");
    regexMatchSet.add(STREAMOVERLAP+":topic4");
    regex = Pattern.compile(STREAMOVERLAP+":topic[0-4]$");
    consumer.subscribe(regex, null);
    subscribeList = consumer.assignment();
    subscribedSet = consumer.subscription();
    iter = subscribeList.iterator();
    while(iter.hasNext()) {
      System.out.println("subscribe topic[0-4]$: " + iter.next());
    }
    assertTrue(subscribeList.size() == 5);
    assertTrue(subscribedSet.equals(regexMatchSet));

    consumer.poll(100);

    consumer.unsubscribe();
    subscribeList = consumer.assignment();
    subscribedSet = consumer.subscription();
    iter = subscribeList.iterator();
    while(iter.hasNext()) {
      System.out.println("unsubscribe: " + iter.next());
    }
    assertTrue(subscribeList.size() == 0);
    assertTrue(subscribedSet.isEmpty());

    consumer.poll(100);
    consumer.close();
  }

  @Test
  public void testTopicDotRegexSubscription() throws IOException {
    Properties props = new Properties();
    MarlinConfigDefaults cdef = MarlinConfigDefaults.getDefaultInstance();
    props.put("key.deserializer",
              "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    props.put("value.deserializer",
              "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    props.put("auto.offset.reset", "earliest");
    props.put(cdef.getMetadataMaxAge(), 100);

    KafkaConsumer consumer = new KafkaConsumer<byte[], byte[]>(props);

    List<String> subscrTopics = new ArrayList<String>();
    subscrTopics.add("t-pic");
    subscrTopics.add("t.pic");
    subscrTopics.add("t_pic");
    subscrTopics.add("tApic");
    subscrTopics.add("tbpic");
    subscrTopics.add("t1pic");
    subscrTopics.add("t1picABC");
    subscrTopics.add("xad");
    subscrTopics.add("x123");
    subscrTopics.add("x");
    subscrTopics.add("abcx123");
    subscrTopics.add("-x123");

    for(String topic : subscrTopics) {
     madmin.createTopic(STREAMDOT, topic, 1);
    }

    List<String> topics = new ArrayList<String>();

    boolean failNullSubscription = false;
    try {
      topics.add((String) null);
      consumer.subscribe(topics);
    } catch (Exception e) {
      System.out.println("subscribe to null topic string... " + e);
      failNullSubscription = true;
    }
    assertTrue(failNullSubscription);

    List<String> subscribeArrayNull = new ArrayList<String>();
    subscribeArrayNull.add(STREAMDOT+":t.pic");
    subscribeArrayNull.add(null);
    subscribeArrayNull.add(STREAMDOT+":testing");

    failNullSubscription = false;
    try {
      consumer.subscribe(subscribeArrayNull);
    } catch (Exception e) {
      System.out.println("subscribe to null topic in array failed... " + e);
      failNullSubscription = true;
    }
    assertTrue(failNullSubscription);

    subscribeArrayNull = new ArrayList<String>();

    failNullSubscription = false;
    consumer.subscribe(subscribeArrayNull);
    Set<TopicPartition> subscribeList = consumer.assignment();
    Iterator<TopicPartition> iter = subscribeList.iterator();
    while(iter.hasNext()) {
      System.out.println("subscribe unset list: " + iter.next());
    }
    assertTrue(subscribeList.size() == 0);
    assertTrue(consumer.subscription().size() == 0);

    subscribeArrayNull.clear();
    subscribeArrayNull.add(STREAMDOT+":DOESNOTEXIST");
    subscribeArrayNull.add(STREAMDOT+":t1pic");
    subscribeArrayNull.add(STREAMDOT+":x123");
    consumer.subscribe(subscribeArrayNull);
    subscribeList = consumer.assignment();
    iter = subscribeList.iterator();
    while(iter.hasNext()) {
      System.out.println("subscribe valid list: " + iter.next());
    }
    assertTrue(subscribeList.size() == 2);
    assertTrue(consumer.subscription().size() == 3);

    consumer.unsubscribe();

    subscribeArrayNull.clear();
    subscribeArrayNull.add(STREAMDOT+":t.pic");
    consumer.subscribe(subscribeArrayNull);
    subscribeList = consumer.assignment();
    subscribeList = consumer.assignment();
    iter = subscribeList.iterator();
    while(iter.hasNext()) {
      System.out.println("subscribe topic t.pic: " + iter.next());
    }
    assertTrue(subscribeList.size() == 1);
    assertTrue(consumer.subscription().size() == 1);

    consumer.unsubscribe();

    Pattern regex;
    Pattern topicRegex;
    int expectedMatch = 0;
    List<String> patterns = new ArrayList<String> ();
    patterns.add("t.pic");
    patterns.add("t.pic$");
    patterns.add("t.{1}pic");
    patterns.add("t.{1}pic$");
    patterns.add("x.*$");
    patterns.add("^x.*$");
    patterns.add(".*$");
    patterns.add("^[:alpha:]*$");
    patterns.add("^\\w*$");
    patterns.add(".*::.*");
    patterns.add(".*:.*");
    //TODO(bug 20755)
    //patterns.add("^[[:alpha:]]*$");
    //patterns.add("[[:alpha:]]*");

    for (String pattern : patterns) {
      regex = Pattern.compile(STREAMDOT + ":" + pattern);
      topicRegex = Pattern.compile(pattern);
      expectedMatch = 0;
      for (String topic : subscrTopics) {
        if (topicRegex.matcher(topic).matches()) {
          ++expectedMatch;
        }
      }

      consumer.subscribe(regex, null);
      subscribeList = consumer.assignment();
      iter = subscribeList.iterator();
      while(iter.hasNext()) {
        System.out.println("subscribe regex " + pattern + ": " + iter.next());
      }
      if (subscribeList.size() != expectedMatch) {
        _logger.info("Expected matches for topic pattern " + pattern);
        for (String topic : subscrTopics) {
          if (topicRegex.matcher(topic).matches()) {
            System.out.println(pattern + " matched " + topic);
          }
        }
      }
      assertTrue(subscribeList.size() == expectedMatch);
      consumer.unsubscribe();
      subscribeList = consumer.assignment();
      assertTrue(subscribeList.size() == 0);
    }

    //Test for invalid pattern
    regex = Pattern.compile(STREAMDOT+":*");
    Exception ex = null;
    try {
      consumer.subscribe(regex, null);
    } catch (Exception e) {
      ex = e;
    }
    assertTrue(ex instanceof IllegalArgumentException);
    consumer.unsubscribe();

    consumer.close();
  }

  @Test
  public void testTopicORRegexSubscription() throws IOException {
    Properties props = new Properties();
    MarlinConfigDefaults cdef = MarlinConfigDefaults.getDefaultInstance();
    props.put("key.deserializer",
              "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    props.put("value.deserializer",
              "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    props.put("auto.offset.reset", "earliest");
    props.put(cdef.getMetadataMaxAge(), 100);


    List<String> subscrTopics = new ArrayList<String>();
    subscrTopics.add("t-pic");
    subscrTopics.add("t.pic");
    subscrTopics.add("t_pic");
    subscrTopics.add("abxyzpic");
    subscrTopics.add("abpic");
    subscrTopics.add("t1pic");
    subscrTopics.add("t1picABC");
    subscrTopics.add("xad");
    subscrTopics.add("x123");
    subscrTopics.add("x");
    subscrTopics.add("abcx123");
    subscrTopics.add("-x123");

    for(String topic : subscrTopics) {
     madmin.createTopic(STREAMOR, topic, 1);
    }

    Pattern regex;
    Pattern topicRegex;
    int expectedMatch = 0;
    String pattern = new String ("t.*|x.*");

    regex = Pattern.compile(STREAMOR + ":" + pattern);
    topicRegex = Pattern.compile(pattern);
    expectedMatch = 0;
    for (String topic : subscrTopics) {
      if (topicRegex.matcher(topic).matches()) {
        ++expectedMatch;
      }
    }

    KafkaConsumer consumer = new KafkaConsumer<byte[], byte[]>(props);
    consumer.subscribe(regex, null);
    Set<TopicPartition> subscribeList = consumer.assignment();
    Iterator<TopicPartition> iter = subscribeList.iterator();
    subscribeList = consumer.assignment();
    iter = subscribeList.iterator();
    while(iter.hasNext()) {
      System.out.println("subscribe regex " + pattern + ": " + iter.next());
    }
    System.out.println( "subscribeList.size():" + subscribeList.size());
    if (subscribeList.size() != expectedMatch) {
      _logger.info("Expected matches for topic pattern " + pattern);
      for (String topic : subscrTopics) {
        if (topicRegex.matcher(topic).matches()) {
            System.out.println(pattern + " matched " + topic);
        }
      }
    }
    assertTrue(subscribeList.size() == expectedMatch);
    consumer.unsubscribe();
    subscribeList = consumer.assignment();
    assertTrue(subscribeList.size() == 0);

    consumer.unsubscribe();
    consumer.close();
  }

  @Test
  public void testTopicRegexSubscription() throws IOException {
    Properties props = new Properties();
    MarlinConfigDefaults cdef = MarlinConfigDefaults.getDefaultInstance();
    props.put("key.deserializer",
              "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    props.put("value.deserializer",
              "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    props.put("auto.offset.reset", "earliest");
    props.put(cdef.getMetadataMaxAge(), 100);

    KafkaConsumer consumer = new KafkaConsumer<byte[], byte[]>(props);

    runRegexTestWithSingleListener(consumer, STREAM, null);
  }

  @Test
  public void testListenerGroupTopicRegexSubscription() throws IOException {
    Properties props = new Properties();
    MarlinConfigDefaults cdef = MarlinConfigDefaults.getDefaultInstance();
    props.put("key.deserializer",
              "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    props.put("value.deserializer",
              "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    props.put("auto.offset.reset", "earliest");
    props.put("group.id", "regexgroup");
    props.put(cdef.getMetadataMaxAge(), 100);

    RebalanceCb cb = new RebalanceCb();
    ByteArrayDeserializer keyD = new ByteArrayDeserializer();
    ByteArrayDeserializer valueD = new ByteArrayDeserializer();

    KafkaConsumer consumer = new KafkaConsumer<byte[], byte[]>(props, keyD, valueD);

    runRegexTestWithSingleListener(consumer, STREAMLG, cb);
  }

  @Test
  public void testListenerGroupTwoListenersRegex() throws IOException {
    Properties props = new Properties();
    MarlinConfigDefaults cdef = MarlinConfigDefaults.getDefaultInstance();
    props.put("key.deserializer",
              "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    props.put("value.deserializer",
              "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    props.put("auto.offset.reset", "earliest");
    props.put("group.id", "regexgroup");
    props.put(cdef.getMetadataMaxAge(), 100);

    RebalanceCb cb1 = new RebalanceCb();
    ByteArrayDeserializer keyD1 = new ByteArrayDeserializer();
    ByteArrayDeserializer valueD1 = new ByteArrayDeserializer();

    KafkaConsumer consumer1 = new KafkaConsumer<byte[], byte[]>(props, keyD1, valueD1);

    RebalanceCb cb2 = new RebalanceCb();
    ByteArrayDeserializer keyD2 = new ByteArrayDeserializer();
    ByteArrayDeserializer valueD2 = new ByteArrayDeserializer();

    KafkaConsumer consumer2 = new KafkaConsumer<byte[], byte[]>(props, keyD2, valueD2);

    madmin.createTopic(STREAMTWOLISTENER, "topic0", 2);
    madmin.createTopic(STREAMTWOLISTENER, "topic1", 2);
    madmin.createTopic(STREAMTWOLISTENER, "topic2", 2);
    madmin.createTopic(STREAMTWOLISTENER, "topic3", 2);
    madmin.createTopic(STREAMTWOLISTENER, "topic4", 2);
    madmin.createTopic(STREAMTWOLISTENER, "topic1A", 2);
    madmin.createTopic(STREAMTWOLISTENER, "topic22", 2);
    madmin.createTopic(STREAMTWOLISTENER, "abc", 2);
    madmin.createTopic(STREAMTWOLISTENER, "12345", 2);

    // Array subscription
    Pattern regex = Pattern.compile(STREAMTWOLISTENER+":^topic.*$");

    consumer1.subscribe(regex, cb1);

    int numTries = 5;
    while (numTries > 0) {
      try {
        Thread.sleep(150);
      } catch (Exception e) {
        System.out.println(e);
      }
      cb1.assignDone();
      if (consumer1.assignment().size() == 7*2)
        break;
    }

    cb1.clear();
    cb2.clear();

    Set<TopicPartition> subscribeList1 = consumer1.assignment();
    Iterator<TopicPartition> iter1 = subscribeList1.iterator();
    while(iter1.hasNext()) {
      System.out.println("consumer 1 subscribe regexArr, consumer 1: " + iter1.next());
    }
    Set<TopicPartition> subscribeList2 = consumer2.assignment();
    Iterator<TopicPartition> iter2 = subscribeList2.iterator();
    while(iter2.hasNext()) {
      System.out.println("consumer 1 subscribe regexArr, consumer 2: " + iter2.next());
    }
    assertTrue(subscribeList1.size() == 7*2);
    assertTrue(subscribeList2.size() == 0);

    consumer2.subscribe(regex, cb2);

    numTries = 5;
    while (numTries > 0) {
      try {
        Thread.sleep(150);
      } catch (Exception e) {
        System.out.println(e);
      }
      cb2.assignDone();
      if (consumer1.assignment().size() + consumer2.assignment().size() == 7*2)
        break;
    }

    cb1.clear();
    cb2.clear();

    subscribeList1 = consumer1.assignment();
    iter1 = subscribeList1.iterator();
    while(iter1.hasNext()) {
      System.out.println("consumer 2 subscribe regexArr, consumer 1: " + iter1.next());
    }
    subscribeList2 = consumer2.assignment();
    iter2 = subscribeList2.iterator();
    while(iter2.hasNext()) {
      System.out.println("consumer 2 subscribe regexArr, consumer 2: " + iter2.next());
    }
    assertTrue(subscribeList1.size() + subscribeList2.size()  == 7*2);

    consumer1.unsubscribe();


    numTries = 5;
    while (numTries > 0) {
      try {
        Thread.sleep(150);
      } catch (Exception e) {
        System.out.println(e);
      }
      cb1.revokeDone();
      if (consumer1.assignment().size() == 0)
        break;
    }

    numTries = 5;
    while (numTries > 0) {
      try {
        Thread.sleep(150);
      } catch (Exception e) {
        System.out.println(e);
      }
      cb2.assignDone();
      if (consumer2.assignment().size() == 7*2)
        break;
    }

    cb1.clear();
    cb2.clear();

    subscribeList1 = consumer1.assignment();
    iter1 = subscribeList1.iterator();
    while(iter1.hasNext()) {
      System.out.println("consumer 1 unsubscribe regexArr, consumer 1: " + iter1.next());
    }
    subscribeList2 = consumer2.assignment();
    iter2 = subscribeList2.iterator();
    while(iter2.hasNext()) {
      System.out.println("consumer 1 unsubscribe regexArr, consumer 2: " + iter2.next());
    }
    assertTrue(subscribeList1.size() == 0);
    assertTrue(subscribeList2.size() == 7*2);

    consumer2.unsubscribe();

    numTries = 5;
    while (numTries > 0) {
      try {
        Thread.sleep(150);
      } catch (Exception e) {
        System.out.println(e);
      }
      cb2.revokeDone();
      if (consumer2.assignment().size() == 0)
        break;
    }

    cb1.clear();
    cb2.clear();

    subscribeList1 = consumer1.assignment();
    iter1 = subscribeList1.iterator();
    while(iter1.hasNext()) {
      System.out.println("consumer 2 unsubscribe regexArr, consumer 1: " + iter1.next());
    }
    subscribeList2 = consumer2.assignment();
    iter2 = subscribeList2.iterator();
    while(iter2.hasNext()) {
      System.out.println("consumer 2 unsubscribe regexArr, consumer 2: " + iter2.next());
    }
    assertTrue(subscribeList1.size() == 0);
    assertTrue(subscribeList2.size() == 0);

    consumer1.subscribe(regex, cb1);

    numTries = 5;
    while (numTries > 0) {
      try {
        Thread.sleep(150);
      } catch (Exception e) {
        System.out.println(e);
      }
      cb1.assignDone();
      if (consumer1.assignment().size() == 7*2)
        break;
    }

    cb1.clear();
    cb2.clear();

    subscribeList1 = consumer1.assignment();
    iter1 = subscribeList1.iterator();
    while(iter1.hasNext()) {
      System.out.println("consumer 1 re-subscribe regexArr, consumer 1: " + iter1.next());
    }
    subscribeList2 = consumer2.assignment();
    iter2 = subscribeList2.iterator();
    while(iter2.hasNext()) {
      System.out.println("consumer 1 re-subscribe regexArr, consumer 2: " + iter2.next());
    }
    assertTrue(subscribeList1.size() == 7*2);
    assertTrue(subscribeList2.size() == 0);

    consumer1.close();
    consumer2.close();
  }

  @Test
  public void testListenerGroupSubscribeFullPath() throws IOException {
    Properties props = new Properties();
    MarlinConfigDefaults cdef = MarlinConfigDefaults.getDefaultInstance();
    props.put("key.deserializer",
              "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    props.put("value.deserializer",
              "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    props.put("auto.offset.reset", "earliest");
    props.put("group.id", "fullpathgroup");
    props.put(cdef.getMetadataMaxAge(), 100);

    RebalanceCb cb = new RebalanceCb();
    ByteArrayDeserializer keyD = new ByteArrayDeserializer();
    ByteArrayDeserializer valueD = new ByteArrayDeserializer();

    KafkaConsumer consumer = new KafkaConsumer<byte[], byte[]>(props, keyD, valueD);
    runFullPathTest(consumer, STREAMFPLG, cb);
  }

  @Test
  public void testListenerSubscribeFullPath() throws IOException {
    Properties props = new Properties();
    MarlinConfigDefaults cdef = MarlinConfigDefaults.getDefaultInstance();
    props.put("key.deserializer",
              "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    props.put("value.deserializer",
              "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    props.put("auto.offset.reset", "earliest");
    props.put(cdef.getMetadataMaxAge(), 100);

    KafkaConsumer consumer = new KafkaConsumer<byte[], byte[]>(props);
    runFullPathTest(consumer, STREAMFP, null);
  }


  private void runFullPathTest(KafkaConsumer consumer, String streamName, RebalanceCb cb) throws IOException {
    madmin.createTopic(streamName, "topic", 1);

    List<String> topic = new ArrayList<String>();
    topic.add(streamName+":topic");
    topic.add("/mapr/my.cluster.com" + streamName + ":topic");

    if (cb != null)
      consumer.subscribe(topic, cb);
    else
      consumer.subscribe(topic);

    try {
      Thread.sleep(150);
    } catch (Exception e) {
      System.out.println(e);
    }

    if (cb != null) cb.assignDone();

    Set<TopicPartition> subscribeList = consumer.assignment();
    assertTrue(subscribeList.size() == 1);

    consumer.close();
  }

  private void runRegexTestWithSingleListener(KafkaConsumer consumer, String streamName, RebalanceCb cb) throws IOException {
    // anything that starts with "topic"
    Pattern regex = Pattern.compile(streamName+":topic.*$");
    consumer.subscribe(regex, cb);

    try {
      Thread.sleep(150);
    } catch (Exception e) {
      System.out.println(e);
    }

    Set<TopicPartition> subscribeList = consumer.assignment();
    Iterator<TopicPartition> iter = subscribeList.iterator();
    while(iter.hasNext()) {
      System.out.println("SHOULD NOT SEE ANY STREAMS " + iter.next());
    }
    assertTrue(subscribeList.size() == 0);

    madmin.createTopic(streamName, "topic1", 1);
    madmin.createTopic(streamName, "topic", 1);
    madmin.createTopic(streamName, "topi", 1);
    madmin.createTopic(streamName, "abcd", 1);
    madmin.createTopic(streamName, "ab", 1);
    madmin.createTopic(streamName, "007topic", 1);
    madmin.createTopic(streamName, "1234567", 1);
    madmin.createTopic(streamName, "234567", 1);

    try {
      Thread.sleep(150);
    } catch (Exception e) {
      System.out.println(e);
    }

    if (cb != null) cb.assignDone();

    subscribeList = consumer.assignment();
    iter = subscribeList.iterator();
    while(iter.hasNext()) {
      System.out.println("subcribe " + streamName + " :topic.*$ " + iter.next());
    }
    // topic1, topic
    assertTrue(subscribeList.size() == 2);

    // anything that is "ab", "abb", "abbb", etc.
    regex = Pattern.compile(streamName+":ab+$");
    consumer.subscribe(regex, cb);

    try {
      Thread.sleep(150);
    } catch (Exception e) {
      System.out.println(e);
    }

    if (cb != null) cb.revokeDone();

    try {
      Thread.sleep(150);
    } catch (Exception e) {
      System.out.println(e);
    }

    if (cb != null) cb.assignDone();

    subscribeList = consumer.assignment();
    iter = subscribeList.iterator();
    while(iter.hasNext()) {
      System.out.println("subscribe " + streamName + ":ab+$ " + iter.next());
    }
    // ab
    assertTrue(subscribeList.size() == 1);

    // anything starting with 0 or 1.
    regex = Pattern.compile(streamName+":[0,1]+.*");
    consumer.subscribe(regex, cb);

    try {
      Thread.sleep(150);
    } catch (Exception e) {
      System.out.println(e);
    }

    if (cb != null) cb.revokeDone();

    try {
      Thread.sleep(150);
    } catch (Exception e) {
      System.out.println(e);
    }

    if (cb != null) cb.assignDone();

    subscribeList = consumer.assignment();
    iter = subscribeList.iterator();
    while(iter.hasNext()) {
      System.out.println("subscribe " + streamName + ":[0,1]+ " + iter.next());
    }
    // 007topic, 1234567
    assertTrue(subscribeList.size() == 2);

    // delete topics and see if we update properly
    madmin.deleteTopic(streamName, "topic1");
    madmin.deleteTopic(streamName, "007topic");

    try {
      Thread.sleep(150);
    } catch (Exception e) {
      System.out.println(e);
    }

    if (cb != null) cb.revokeDone();

    subscribeList = consumer.assignment();
    iter = subscribeList.iterator();
    while(iter.hasNext()) {
      System.out.println("deleted topic1 and 007topic" + iter.next());
    }

    // topic, abcd, 1234567
    assertTrue(subscribeList.size() == 1);

    // unsubscribe from topics starting with 0 or 1
    consumer.unsubscribe();

    try {
      Thread.sleep(150);
    } catch (Exception e) {
      System.out.println(e);
    }

    if (cb != null) cb.revokeDone();

    subscribeList = consumer.assignment();
    iter = subscribeList.iterator();
    while(iter.hasNext()) {
      System.out.println("unsubscrbe " + iter.next());
    }
    assertTrue(subscribeList.size() == 0);

    consumer.close();
  }


  private static final class RebalanceCb implements ConsumerRebalanceListener { 

    private boolean revoked;
    private boolean assigned;
    public RebalanceCb() {
      revoked = false;
      assigned = false;
    }

    public synchronized void clear() {
      revoked = false;
      assigned = false;
    }

    public synchronized void revokeDone () {
      while (revoked == false) {
        try {
          this.wait();
        } catch (Exception e) {
        }
      }
      revoked = false;
    }

    public synchronized void assignDone() {
      while (assigned == false) {
        try {
          this.wait();
        } catch (Exception e) {
        }
      }
      assigned = false;
    }

    public void onPartitionsAssigned(Collection<TopicPartition> partitions) {
      synchronized(this) {
        //System.out.println("partition assigned first " + partitions + " " + partitions.size());
        assigned = true;
        this.notifyAll();
      }
    }

    public void onPartitionsRevoked(Collection<TopicPartition> partitions) {
      synchronized(this) {
        //System.out.println("partition revoke first " + partitions + " " + partitions.size());
        revoked = true;
        this.notifyAll();
      }
    }

  }

}
