/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.streams.tests.listener;

import static org.junit.Assert.assertTrue;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.Future;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.regex.Pattern;

import org.apache.hadoop.conf.Configuration;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRebalanceListener;
import org.apache.kafka.clients.consumer.OffsetCommitCallback;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;

import com.mapr.tests.BaseTest;
import com.mapr.tests.annotations.ClusterTest;

import com.mapr.streams.Admin;
import com.mapr.streams.Streams;
import com.mapr.streams.StreamDescriptor;
import com.mapr.fs.proto.Marlinserver.MarlinConfigDefaults;


@Category(ClusterTest.class)
public class ListenerCommitTest extends BaseTest {
  private static final Logger _logger = LoggerFactory.getLogger(ListenerCommitTest.class);
  private static final String STREAM = "/jtest-" + ListenerCommitTest.class.getSimpleName();
  private static Admin madmin;
  private static KafkaProducer producer;
  private static KafkaConsumer kc;
  private static KafkaConsumer kclg;
  private static final int numParts = 10;
  public static final byte[] value = new byte[200];
  public static final byte[] key = "abc".getBytes();

  @BeforeClass
  public static void setupTestClass() throws Exception {
    final Configuration conf = new Configuration();
    MarlinConfigDefaults cdef = MarlinConfigDefaults.getDefaultInstance();

    madmin = Streams.newAdmin(conf);

    Properties props = new Properties();
    props.put("key.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    props.put("value.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    props.put(cdef.getParallelFlushersPerPartition(), false);
    props.put(cdef.getMetadataMaxAge(), 100);  // want to exercise metadata refresher
    props.put(cdef.getBufferTime(), 5000);  // flush quickly for testing
    producer = new KafkaProducer<byte[], byte[]>(props);

    props = new Properties();
    props.put("key.deserializer",
              "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    props.put("value.deserializer",
              "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    props.put("auto.offset.reset", "earliest");
    props.put("enable.auto.commit", false);
    props.put(cdef.getMetadataMaxAge(), 5000);
    kc = new KafkaConsumer<byte[], byte[]>(props);

    props.put("group.id", "committest");
    kclg = new KafkaConsumer<byte[], byte[]>(props);

    //Cleanup all stale streams
    try {
      madmin.deleteStream(STREAM);
    } catch (Exception e) {}

    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(numParts);
    madmin.createStream(STREAM, sdesc);
  }

  @AfterClass
  public static void cleanupTestClass() throws Exception {
    producer.close();
    kc.close();
    kclg.close();
    madmin.deleteStream(STREAM);
  }

  // Test consumer.commitSync()
  @Test
  public void testCommitSyncNoParam() throws Exception {
    String topicname = STREAM+":CommitSyncNoParam";
    commitSyncNoParam(kc, topicname);
  }
  @Test
  public void testCommitSyncNoParamLG() throws Exception {
    String topicname = STREAM+":CommitSyncNoParamLG";
    commitSyncNoParam(kclg, topicname);
  }

  // Test consumer.commitSync(Map<TopicPartition, OffsetAndMetadata> offsets)
  @Test
  public void testCommitSyncWithOffsets() throws Exception {
    String topicname = STREAM+":CommitSyncWithOffsets";
    commitSyncNoParam(kc, topicname);
  }
  @Test
  public void testCommitSyncWithOffsetsLG() throws Exception {
    String topicname = STREAM+":CommitSyncWithOffsetsLG";
    commitSyncNoParam(kclg, topicname);
  }

  // Test consumer.commitAsync()
  @Test
  public void testCommitAsyncNoParam() throws Exception {
    String topicname = STREAM+":CommitAsyncNoParam";
    commitAsyncNoParam(kc, topicname);
  }
  @Test
  public void testCommitAsyncNoParamLG() throws Exception {
    String topicname = STREAM+":CommitAsyncNoParamLG";
    commitAsyncNoParam(kclg, topicname);
  }

  // Test consumer.commitAsync(OffsetCommitCallback callback)
  @Test
  public void testCommitAsyncCallback() throws Exception {
    String topicname = STREAM+":CommitAsyncCallback";
    commitAsyncCallback(kc, topicname);
  }
  @Test
  public void testCommitAsyncCallbackLG() throws Exception {
    String topicname = STREAM+":CommitAsyncCallbackLG";
    commitAsyncCallback(kclg, topicname);
  }

  // Test consumer.commitAsync(Map<TopicPartition, OffsetAndMetadata> offsets, OffsetCommitCallback callback)
  @Test
  public void testCommitAsyncCallbackOffsets() throws Exception {
    String topicname = STREAM+":CommitAsyncCallbackOffsets";
    commitAsyncCallbackOffsets(kc, topicname);
  }
  @Test
  public void testCommitAsyncCallbackOffsetsLG() throws Exception {
    String topicname = STREAM+":CommitAsyncCallbackOffsetsLG";
    commitAsyncCallbackOffsets(kclg, topicname);
  }


  public void testPhase1(KafkaConsumer consumer,
                                String topicname,
                                TopicPartition[] topicpartitions,
                                Future[] futures) throws Exception {

    for (int i = 0; i < numParts; ++i) {
      topicpartitions[i] = new TopicPartition(topicname, i);
    }

    // Producer producer messages
    for (int i = 0; i < numParts; i++) {
      ProducerRecord<byte[], byte[]> record =
        new ProducerRecord<byte[], byte[]>(topicname, i, key, value);
       futures[i] = producer.send(record);
    }
    producer.flush();

    for (Future<RecordMetadata> future : futures) {
      future.get();
    }

    RebalanceCb callback = new RebalanceCb();
    List<String> subscribeList = new ArrayList<String>();
    subscribeList.add(topicname);

    consumer.subscribe(subscribeList, callback);

    callback.assignDone();

    consumer.poll(100);
    consumer.poll(100);

    for (int i = 0; i < numParts; i++) {
      boolean committedException = false;
      OffsetAndMetadata offset = null;
      try {
        offset = consumer.committed(topicpartitions[i]);
      } catch (Exception e) {
        committedException = true;
        System.out.println("committed exception: " + e);
      }
      assertTrue(committedException || (offset == null) || offset.offset() == 0);
    }
  }

  public void testPhase2(KafkaConsumer consumer,
                                String topicname,
                                TopicPartition[] topicpartitions,
                                Future[] futuresOld,
                                Future[] futuresNew) throws Exception {
    // Check results from phase 1.
    for (int i = 0; i < numParts; i++) {
      OffsetAndMetadata offset = consumer.committed(topicpartitions[i]);
      assertTrue(offset.offset() == ((RecordMetadata) futuresOld[i].get()).offset() + 1);
    }

    // Producer more producer messages
    for (int i = 0; i < numParts; i++) {
      ProducerRecord<byte[], byte[]> record =
        new ProducerRecord<byte[], byte[]>(topicname, i, key, value);
       futuresNew[i] = producer.send(record);
    }
    producer.flush();

    for (Future<RecordMetadata> future : futuresNew) {
      future.get();
    }

    // Now, make sure that the commit offset hasn't moved forward
    for (int i = 0; i < numParts; i++) {
      OffsetAndMetadata offset = consumer.committed(topicpartitions[i]);
      assertTrue(offset.offset() == ((RecordMetadata) futuresOld[i].get()).offset() + 1);
    }

    consumer.poll(100);
    consumer.poll(100);

    // Now, make sure that the commit offset hasn't moved forward
    for (int i = 0; i < numParts; i++) {
      OffsetAndMetadata offset = consumer.committed(topicpartitions[i]);
      assertTrue(offset.offset() == ((RecordMetadata) futuresOld[i].get()).offset() + 1);
    }
  }

  public void testPhase3(KafkaConsumer consumer,
                                String topicname,
                                TopicPartition[] topicpartitions,
                                Future[] futures) throws Exception {
    // Check results from phase 2.
    for (int i = 0; i < numParts; i++) {
      OffsetAndMetadata offset = consumer.committed(topicpartitions[i]);
      // System.out.println(topicpartitions[i] + " " + offset.offset() + " == " +
      //                    ((RecordMetadata) futures[i].get()).offset());
      assertTrue(offset.offset() == ((RecordMetadata) futures[i].get()).offset() + 1);
    }
    consumer.unsubscribe();
  }

  public void commitSyncNoParam(KafkaConsumer consumer, String topicname) throws Exception {
    TopicPartition[] topicpartitions = new TopicPartition[numParts];
    Future[] futures = new Future[numParts];
    Future[] futures2 = new Future[numParts];

    testPhase1(consumer, topicname, topicpartitions, futures);
    consumer.commitSync();  // The test!
    testPhase2(consumer, topicname, topicpartitions, futures, futures2);
    consumer.commitSync();  // The test!
    testPhase3(consumer, topicname, topicpartitions, futures2);
  }

  public void commitSyncWithOffsets(KafkaConsumer consumer, String topicname) throws Exception {

    String[] tokens = topicname.split(":");
    madmin.createTopic(tokens[0], tokens[1], numParts);

    TopicPartition[] topicpartitions = new TopicPartition[numParts];
    for (int i = 0; i < numParts; ++i) {
      topicpartitions[i] = new TopicPartition(topicname, i);
    }

    Map<TopicPartition, OffsetAndMetadata> toCommit = new HashMap<TopicPartition, OffsetAndMetadata>();
    for (int i = 0; i < numParts; ) { // only commit for the even partition ids
      toCommit.put(topicpartitions[i], new OffsetAndMetadata(2^i));
      i += 2;
    }

    RebalanceCb callback = new RebalanceCb();
    List<String> subscribeList = new ArrayList<String>();
    subscribeList.add(topicname);
    consumer.subscribe(subscribeList, callback);

    callback.assignDone();

    consumer.commitSync(toCommit);

    for (int i = 0; i < numParts; i++) {
      OffsetAndMetadata offset = consumer.committed(topicpartitions[i]);
      if (i%2 == 0) {
        assertTrue(offset.offset() == (2^i));
      } else {
        assertTrue(offset.offset() == 0);
      }
    }

    toCommit = new HashMap<TopicPartition, OffsetAndMetadata>();
    for (int i = 0; i < numParts; ++i) { // only commit for the even partition ids
      toCommit.put(topicpartitions[i], new OffsetAndMetadata(i+1024));
    }

    consumer.commitSync(toCommit);

    for (int i = 0; i < numParts; i++) {
      OffsetAndMetadata offset = consumer.committed(topicpartitions[i]);
      assertTrue(offset.offset() == i+1024);
    }

    consumer.unsubscribe();
  }

  public void commitAsyncNoParam(KafkaConsumer consumer, String topicname) throws Exception {
    TopicPartition[] topicpartitions = new TopicPartition[numParts];
    Future[] futures = new Future[numParts];
    Future[] futures2 = new Future[numParts];

    testPhase1(consumer, topicname, topicpartitions, futures);
    consumer.commitAsync();  // The test!
    try {
      Thread.sleep(300);
    } catch (Exception e) {
      System.out.println(e);
    }
    testPhase2(consumer, topicname, topicpartitions, futures, futures2);
    consumer.commitAsync();  // The test!
    try {
      Thread.sleep(300);
    } catch (Exception e) {
      System.out.println(e);
    }
    testPhase3(consumer, topicname, topicpartitions, futures2);
  }

  public void commitAsyncCallback(KafkaConsumer consumer, String topicname) throws Exception {
    TopicPartition[] topicpartitions = new TopicPartition[numParts];
    Future[] futures = new Future[numParts];
    Future[] futures2 = new Future[numParts];

    testPhase1(consumer, topicname, topicpartitions, futures);

    CommitCb ccb = new CommitCb();
    consumer.commitAsync(ccb);  // The test!
    ccb.commitDone();

    assertTrue(ccb.getException() == null);
    Map<TopicPartition, OffsetAndMetadata> committedOffsets = ccb.getCommittedOffsets();
    for (Map.Entry<TopicPartition, OffsetAndMetadata> entry : committedOffsets.entrySet()) {
      assertTrue(entry.getValue().equals(consumer.committed(entry.getKey())));
    }

    testPhase2(consumer, topicname, topicpartitions, futures, futures2);

    ccb = new CommitCb();
    consumer.commitAsync(ccb);  // The test!
    ccb.commitDone();

    assertTrue(ccb.getException() == null);
    committedOffsets = ccb.getCommittedOffsets();
    for (Map.Entry<TopicPartition, OffsetAndMetadata> entry : committedOffsets.entrySet()) {
      assertTrue(entry.getValue().equals(consumer.committed(entry.getKey())));
    }

    testPhase3(consumer, topicname, topicpartitions, futures2);
  }

  public void commitAsyncCallbackOffsets(KafkaConsumer consumer, String topicname) throws Exception {

    String tokens[] = topicname.split(":");
    madmin.createTopic(tokens[0], tokens[1], numParts);

    TopicPartition[] topicpartitions = new TopicPartition[numParts];
    for (int i = 0; i < numParts; ++i) {
      topicpartitions[i] = new TopicPartition(topicname, i);
    }

    Map<TopicPartition, OffsetAndMetadata> toCommit = new HashMap<TopicPartition, OffsetAndMetadata>();
    for (int i = 0; i < numParts; ) { // only commit for the even partition ids
      toCommit.put(topicpartitions[i], new OffsetAndMetadata(2^i));
      i += 2;
    }

    RebalanceCb callback = new RebalanceCb();
    List<String> subscribeList = new ArrayList<String>();
    subscribeList.add(topicname);
    consumer.subscribe(subscribeList, callback);

    callback.assignDone();

    CommitCb ccb = new CommitCb();
    consumer.commitAsync(toCommit, ccb);
    ccb.commitDone();

    assertTrue(ccb.getException() == null);
    Map<TopicPartition, OffsetAndMetadata> committedOffsets = ccb.getCommittedOffsets();
    for (Map.Entry<TopicPartition, OffsetAndMetadata> entry : committedOffsets.entrySet()) {
      assertTrue(entry.getValue().equals(consumer.committed(entry.getKey())));
    }

    for (int i = 0; i < numParts; i++) {
      OffsetAndMetadata offset  = null;
      boolean exceptionCaught = false;
      try {
        offset = consumer.committed(topicpartitions[i]);
      } catch (Exception e) {
        exceptionCaught = true;
        System.out.println("commited exception: " + e);
      }
      if (i%2 == 0) {
        assertTrue(offset.offset() == (2^i));
      } else {
        assertTrue(exceptionCaught || (offset == null) || offset.offset() == 0);
      }
    }

    toCommit = new HashMap<TopicPartition, OffsetAndMetadata>();
    for (int i = 0; i < numParts; ++i) { // only commit for the even partition ids
      toCommit.put(topicpartitions[i], new OffsetAndMetadata(i+1024));
    }

    ccb = new CommitCb();
    consumer.commitAsync(toCommit, ccb);
    ccb.commitDone();

    assertTrue(ccb.getException() == null);
    committedOffsets = ccb.getCommittedOffsets();
    for (Map.Entry<TopicPartition, OffsetAndMetadata> entry : committedOffsets.entrySet()) {
      assertTrue(entry.getValue().equals(consumer.committed(entry.getKey())));
    }

    for (int i = 0; i < numParts; i++) {
      OffsetAndMetadata offset = consumer.committed(topicpartitions[i]);
      assertTrue(offset.offset() == i+1024);
    }

    consumer.unsubscribe();
  }



  public final class RebalanceCb implements ConsumerRebalanceListener { 
    private boolean revoked;
    private boolean assigned;
    public RebalanceCb() {
      revoked = false;
      assigned = false;
    }

    public synchronized void clear() {
      revoked = false;
      assigned = false;
    }

    public synchronized void revokeDone () {
      while (revoked == false) {
        try {
          this.wait();
        } catch (Exception e) {
        }
      }
      revoked = false;
    }

    public synchronized void assignDone() {
      while (assigned == false) {
        try {
          this.wait();
        } catch (Exception e) {
        }
      }
      assigned = false;
    }

    public void onPartitionsAssigned(Collection<TopicPartition> partitions) {
      synchronized(this) {
        //System.out.println(this + " partition assigned " + partitions + " " + partitions.size());
        assigned = true;
        this.notifyAll();
      }
    }

    public void onPartitionsRevoked(Collection<TopicPartition> partitions) {
      synchronized(this) {
        //System.out.println(this  + " partition revoke " + partitions + " " + partitions.size());
        revoked = true;
        this.notifyAll();
      }
    }
  }

  public final class CommitCb implements OffsetCommitCallback {
    private boolean committed;
    private Map<TopicPartition, OffsetAndMetadata> committedOffsets;
    private Exception committedException;

    public CommitCb() {
      committed = false;
      committedOffsets = null;
      committedException = null;
    }

    public void onComplete(Map<TopicPartition, OffsetAndMetadata> offsets, Exception exception) {
      synchronized(this) {
        committedOffsets = offsets;
        committedException = exception;
        committed = true;
        this.notifyAll();
      }
    }

    public synchronized void commitDone() {
      while (committed == false) {
        try {
          this.wait();
        } catch (Exception e) {
        }
      }
    }

    public Map<TopicPartition, OffsetAndMetadata> getCommittedOffsets() {
      return committedOffsets;
    }

    public Exception getException() {
      return committedException;
    }

}


}
