/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.streams.tests.listener;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.lang.RuntimeException;
import java.nio.ByteBuffer;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.ArrayList;
import java.util.Set;
import java.util.Arrays;
import java.util.Map;
import java.util.HashMap;
import java.util.Collection;
import java.util.concurrent.atomic.AtomicBoolean;

import org.apache.hadoop.conf.Configuration;
import org.apache.kafka.clients.consumer.NoOffsetForPartitionException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.RecordTooLargeException;
import org.apache.kafka.common.errors.UnknownTopicOrPartitionException;

import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.streams.Admin;
import com.mapr.streams.StreamDescriptor;
import com.mapr.streams.Streams;
import com.mapr.streams.producer.Producer;
import com.mapr.streams.listener.Listener;
import com.mapr.streams.listener.Listener.TED_ACTION;
import com.mapr.streams.tests.producer.ProducerMultiTest.CountCallback;
import com.mapr.streams.tests.producer.SendMessagesToProducer;


import com.mapr.tests.BaseTest;
import com.mapr.tests.annotations.ClusterTest;

import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.producer.Callback;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.clients.consumer.ConsumerRebalanceListener;

@Category(ClusterTest.class)
public class BasicListenerTest extends BaseTest {
  private static final Logger _logger = LoggerFactory.getLogger(BasicListenerTest.class);
  private static final String PREFIX = "/jtest-" + BasicListenerTest.class.getSimpleName() + "-";
  private static Admin madmin;
  private static final int numParts = 4;

  @BeforeClass
  public static void setupTest() throws Exception {
    final Configuration conf = new Configuration();
    madmin = Streams.newAdmin(conf);

    //Cleanup all stale streams
    String sname = PREFIX + "LGtopicSubscr";
    int numStreams = 2;
    for (int i = 0; i < numStreams; i++) {
      try {
        madmin.deleteStream(sname + i);
      } catch (Exception e) {}
    }

    String sname1 = PREFIX + "topicSubscr";
    numStreams = 2;
    for (int i = 0; i < numStreams; i++) {
      try {
        madmin.deleteStream(sname1 + i);
      } catch (Exception e) {}
    }

    String sname2 = PREFIX + "LGMultiListener";
    numStreams = 2;
    for (int i = 0; i < numStreams; i++) {
      try {
        madmin.deleteStream(sname2 + i);
      } catch (Exception e) {}
    }

    //testSingleStream
    sname = PREFIX + "singlestream";
    String snameFull = sname + 0;
    try {
      madmin.deleteStream(snameFull);
    } catch (Exception e) {}

    //testMultipleStream
    sname = PREFIX + "multistream";
    numStreams = 4;
    for (int i = 0; i < numStreams; i++) {
      try {
        madmin.deleteStream(sname + i);
      } catch (Exception e) {}
    }
    //testMultipleStreamErrorHandling
    sname = PREFIX + "multistreameh";
    numStreams = 4;
    for (int i = 0; i < numStreams; i++) {
      try {
        madmin.deleteStream(sname + i);
      } catch (Exception e) {}
    }

    //testPollWithVaryingFetchSize
    sname = PREFIX + "pollfetchsize";
    try {
      madmin.deleteStream(sname + 0);
    } catch (Exception e) {}

    //testTopicHundredSubscription
    sname = PREFIX + "topicSubscrHundred";
    try {
      numStreams = 2;
      for (int i = 0; i < numStreams; i++)
        madmin.deleteStream(sname + i);
    } catch (Exception e) {}

    sname = PREFIX + "LGtopicSubscrHundred";
    try {
      numStreams = 2;
      for (int i = 0; i < numStreams; i++)
        madmin.deleteStream(sname + i);
    } catch (Exception e) {}

    //testListenerBeforeMsgsAreFlushed
    sname = PREFIX + "listenerbeforemsgs";
    try {
      madmin.deleteStream(sname);
    } catch (Exception e) {}

    //testListenerWithSeekToEnd
    sname = PREFIX + "listenerseektoend";
    try {
      madmin.deleteStream(sname);
    } catch (Exception e) {}

    //testListenerFirstPoll
    sname = PREFIX + "listenerfirstpoll";
    try {
      madmin.deleteStream(sname);
    } catch (Exception e) {}

    //testListenerSeqOffset
    sname = PREFIX + "listenerseqoffset";
    try {
      madmin.deleteStream(sname);
    } catch (Exception e) {}

    //testListenerSeqOffsetWithSeek
    for (int i = 0; i < 10; i++) {
      sname = PREFIX + "listenerseqoffsetwithseek" + i;
      try {
        madmin.deleteStream(sname);
      } catch (Exception e) {}
    }

    //testListenerZeroOffset
    sname = PREFIX + "listenerzerooffset";
    try {
      madmin.deleteStream(sname);
    } catch (Exception e) {}

    //testListenerWithTabetSplit
    for (int i = 0; i < 5; i++) {
      sname = PREFIX + "listenerwithtabletsplit" + i;
      try {
        madmin.deleteStream(sname);
      } catch (Exception e) {}
    }

    //testListenerWithNullCommitted
    sname = PREFIX + "listenerwithnullcommit";
    try {
      madmin.deleteStream(sname);
    } catch (Exception e) {}
  }

  @Test
  public void testTopicSubscription() throws IOException {
    String sname = PREFIX + "topicSubscr";
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(numParts);
    Exception ex = null;
    int numStreams = 2;
    boolean messagesOrdered = false;

    for (int i = 0; i < numStreams; i++) madmin.createStream(sname + i, sdesc);
    assertTrue(Producer.runTest(sname, numStreams /*nstreams*/,2  /*ntopics*/, 0 /*nslowtopics*/,
                                numParts /*npart*/, 10000 /*nmsgs*/,
                                !messagesOrdered));
    assertTrue(Listener.runTest(sname, numStreams /*nstreams*/, 2 /*ntopics*/, 0 /*nslowtopics*/,
                                numParts /*npart*/, 10000 /*nmsgs*/, true,
                                messagesOrdered, null));

    for (int i = 0; i < numStreams; i++) madmin.deleteStream(sname + i);

  }

  @Test
  public void testTopicHundredSubscription() throws IOException {
    String sname = PREFIX + "topicSubscrHundred";
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(numParts);
    Exception ex = null;
    int numStreams = 2;
    boolean messagesOrdered = false;

    System.err.println("creating " + numStreams + " streams");
    for (int i = 0; i < numStreams; i++) madmin.createStream(sname + i, sdesc);
    System.err.println("starting producer on 111 topics per streams");
    assertTrue(Producer.runTest(sname, numStreams /*nstreams*/, 111/*ntopics*/, 0 /*nslowtopics*/,
                                numParts /*npart*/, 1000 /*nmsgs*/,
                                !messagesOrdered));
    System.err.println("starting listener on 111 topics per streams");
    assertTrue(Listener.runTest(sname, numStreams /*nstreams*/, 111 /*ntopics*/, 0 /*nslowtopics*/,
                                numParts /*npart*/, 1000 /*nmsgs*/, true,
                                messagesOrdered, null));

    for (int i = 0; i < numStreams; i++) madmin.deleteStream(sname + i);
  }

  @Test
  public void testLGTopicSubscription() throws IOException {
    String sname = PREFIX + "LGtopicSubscr";
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(4); //numParts);
    Exception ex = null;
    int numStreams = 2;
    boolean messagesOrdered = false;

    for (int i = 0; i < numStreams; i++) madmin.createStream(sname + i, sdesc);
    assertTrue(Producer.runTest(sname, numStreams /*nstreams*/, 2 /*ntopics*/, 0 /*nslowtopics*/,
                                4 /*npart*/, 1000 /*nmsgs*/, !messagesOrdered));
    assertTrue(Listener.runTest(sname, numStreams /*nstreams*/, 2 /*ntopics*/, 0 /*nslowtopics*/,
                                4 /*npart*/, 1000 /*nmsgs*/, true,
                                messagesOrdered, "LGTopicTest"));

    for (int i = 0; i < numStreams; i++) madmin.deleteStream(sname + i);

  }

  @Test
  public void testLGTopicHundredSubscription() throws IOException {
    String sname = PREFIX + "LGtopicSubscrHundred";
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(4); //numParts);
    Exception ex = null;
    int numStreams = 2;
    boolean messagesOrdered = false;

    System.err.println("creating " + numStreams + " streams");
    for (int i = 0; i < numStreams; i++) madmin.createStream(sname + i, sdesc);
    System.err.println("starting producer on 111 topics per streams");
    assertTrue(Producer.runTest(sname, numStreams /*nstreams*/, 111 /*ntopics*/, 0 /*nslowtopics*/,
                                4 /*npart*/, 1000 /*nmsgs*/, !messagesOrdered));
    System.err.println("starting listener group on 111 topics per streams");
    assertTrue(Listener.runTest(sname, numStreams /*nstreams*/, 111 /*ntopics*/, 0 /*nslowtopics*/,
                                4 /*npart*/, 1000 /*nmsgs*/, true,
                                messagesOrdered, "LGTopicTest"));

    for (int i = 0; i < numStreams; i++) madmin.deleteStream(sname + i);

  }

  @Test
  public void testLGMultipleListeners() throws IOException {
    String sname = PREFIX + "LGMultiListener";
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(7); //numParts);
    Exception ex = null;
    int numStreams = 2;
    boolean messagesOrdered = false;

    for (int i = 0; i < numStreams; i++) madmin.createStream(sname + i, sdesc);
    assertTrue(Listener.runLGTest(sname, numStreams /*nstreams*/, 2 /*ntopics*/,
                                  7 /* num partitions */, true /* tracing */, "LGTest"));

    for (int i = 0; i < numStreams; i++) madmin.deleteStream(sname + i);
  }

  @Test
  public void testSingleStream() throws IOException {
    String sname = PREFIX + "singlestream";
    String snameFull = sname + 0;
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(numParts);

    Exception ex = null;

    madmin.createStream(snameFull, sdesc);

    _logger.info("Populate stream and check listener");
    assertTrue(Producer.runTest(sname, 1 /*nstreams*/, 2 /*ntopics*/, 0 /*nslowtopics*/,
                                numParts /*npart*/, 10000 /*nmsgs*/, false /* multiflushers */));
    assertTrue(Listener.runTest(sname, 1 /*nstreams*/, 2 /*ntopics*/, 0 /*nslowtopics*/,
                                numParts /*npart*/, 10000 /*nmsgs*/, "TPGrp"));

    _logger.info("test listener incorrect expected nummsgs");
    try {
      ex = null;
      Listener.runTest(sname, 1 /*nstreams*/, 2 /*ntopics*/, 0 /*nslowtopics*/,
                       numParts /*npart*/, 20000 /*nmsgs*/, "TPGrp");
    } catch (Exception e) {
      ex = e;
    }
    assertTrue(ex instanceof IOException);

    _logger.info("test with topics where message are generated slowly");
    madmin.deleteStream(snameFull);
    //Because of bug 20495
    try {
      Thread.sleep(10 * 1000);
    } catch (Exception e) {
    }
    madmin.createStream(snameFull, sdesc);
    assertTrue(Producer.runTest(sname, 1 /*nstreams*/, 2 /*ntopics*/, 2 /*nslowtopics*/,
                                numParts /*npart*/, 100000 /*nmsgs*/, false /* multiflushers */));
    assertTrue(Listener.runTest(sname, 1 /*nstreams*/, 2 /*ntopics*/, 2 /*nslowtopics*/,
                                numParts /*npart*/, 100000 /*nmsgs*/, "TPGrp"));

    madmin.deleteStream(snameFull);
  }

  @Test
  public void testMultipleStream() throws IOException {
    String sname = PREFIX + "multistream";
    int numStreams = 4;
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(numParts);

    Exception ex = null;
    for (int i = 0; i < numStreams; i++) madmin.createStream(sname + i, sdesc);

    _logger.info("Populate stream and check listener");
    assertTrue(Producer.runTest(sname, numStreams, 2 /*ntopics*/, 0 /*nslowtopics*/,
                                numParts /*npart*/, 10000 /*nmsgs*/, false /* multiflushers */));
    assertTrue(Listener.runTest(sname, numStreams, 2 /*ntopics*/, 0 /*nslowtopics*/,
                                numParts /*npart*/, 10000 /*nmsgs*/));

    _logger.info("test with topics where message are generated slowly");
    for (int i = 0; i < numStreams; i++) madmin.deleteStream(sname + i);
    //Because of bug 20495
    try {
      Thread.sleep(10 * 1000);
    } catch (Exception e) {
    }

    for (int i = 0; i < numStreams; i++) madmin.createStream(sname + i, sdesc);
    assertTrue(Producer.runTest(sname, numStreams, 2 /*ntopics*/, 2 /*nslowtopics*/,
                                numParts /*npart*/, 100000 /*nmsgs*/, false /* multiflushers */));
    assertTrue(Listener.runTest(sname, numStreams, 2 /*ntopics*/, 2 /*nslowtopics*/,
                                numParts /*npart*/, 100000 /*nmsgs*/));

    for (int i = 0; i < numStreams; i++) madmin.deleteStream(sname + i);
  }

  @Test
  public void testMultipleStreamErrorHandling() throws IOException {
    String sname = PREFIX + "multistreameh";
    int numStreams = 4;
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(numParts);

    Exception ex = null;
    for (int i = 0; i < numStreams; i++) madmin.createStream(sname + i, sdesc);

    _logger.info("Populate stream");
    assertTrue(Producer.runTest(sname, numStreams, 2 /*ntopics*/, 0 /*nslowtopics*/,
                                numParts /*npart*/, 100000 /*nmsgs*/, false /* multiflushers */));

    _logger.info("test listener stream delete while polling");
    try {
      ex = null;
      Listener.runTest(sname, numStreams, 2 /*ntopics*/, 0 /*nslowtopics*/,
                       numParts /*npart*/, 100000 /*nmsgs*/, TED_ACTION.kDeleteStream);
      System.out.println("Test completed without errrosr !!");
    } catch (Exception e) {
      System.out.println("Got an exception" + e);
      ex = e;
    }
    assertTrue(ex instanceof NoOffsetForPartitionException ||
               ex instanceof IOException);
    for (int i = 0; i < numStreams; i++) {
      try {
        madmin.deleteStream(sname + i);
      } catch (Exception e) {
      }
    }

    ex = null;
    if (System.getProperty("user.name").equals("root")) {
      return;
    }

    //Below tests are only valid for non-root users
    for (int i = 0; i < numStreams; i++) madmin.createStream(sname + i, sdesc);

    _logger.info("Populate stream");
    assertTrue(Producer.runTest(sname, numStreams, 2 /*ntopics*/, 0 /*nslowtopics*/,
                                numParts /*npart*/, 100000 /*nmsgs*/, false /* multiflushers */));

    _logger.info("test listener stream changeperm while polling");
    try {
      ex = null;
      Listener.runTest(sname, numStreams, 2 /*ntopics*/, 0 /*nslowtopics*/,
                       numParts /*npart*/, 100000 /*nmsgs*/, TED_ACTION.kChangePerm);
    } catch (Exception e) {
      ex = e;
    }
    assertTrue(ex instanceof NoOffsetForPartitionException);
    assertTrue(ex.toString().contains("Permission denied"));
    for (int i = 0; i < numStreams; i++) {
      try {
        madmin.deleteStream(sname + i);
      } catch (Exception e) {
      }
    }
  }

  @Test
  public void testPollWithVaryingFetchSize() throws IOException {
    String sname = PREFIX + "pollfetchsize";
    String snameFull = sname + 0;
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(numParts);

    Exception ex = null;

    _logger.info("Populate stream and check listener");
    madmin.createStream(snameFull, sdesc);
    assertTrue(Producer.runTest(sname, 1 /*nstreams*/, 2 /*ntopics*/, 0 /*nslowtopics*/,
                                numParts /*npart*/, 10000 /*nmsgs*/, false /* multiflushers */));
    assertTrue(Listener.runTestWithPollOptions(sname, 1 /*nstreams*/, 2 /*ntopics*/, 0 /*nslowtopics*/,
                                               numParts /*npart*/, 10000 /*nmsgs*/, 1 * 1536));
    madmin.deleteStream(snameFull);
    //Because of bug 20495
    try {
      Thread.sleep(10 * 1000);
    } catch (Exception e) {
    }

    madmin.createStream(snameFull, sdesc);
    assertTrue(Producer.runTest(sname, 1 /*nstreams*/, 2 /*ntopics*/, 0 /*nslowtopics*/,
                                numParts /*npart*/, 10000 /*nmsgs*/, false /* multiflushers */));
    assertTrue(Listener.runTestWithPollOptions(sname, 1 /*nstreams*/, 2 /*ntopics*/, 0 /*nslowtopics*/,
                                               numParts /*npart*/, 10000 /*nmsgs*/, 10 * 1024));
    madmin.deleteStream(snameFull);
    //Because of bug 20495
    try {
      Thread.sleep(10 * 1000);
    } catch (Exception e) {
    }

    _logger.info("test with topics where message are generated slowly");
    madmin.createStream(snameFull, sdesc);
    assertTrue(Producer.runTest(sname, 1 /*nstreams*/, 2 /*ntopics*/, 2 /*nslowtopics*/,
                                numParts /*npart*/, 100000 /*nmsgs*/, false /* multiflushers */));
    assertTrue(Listener.runTestWithPollOptions(sname, 1 /*nstreams*/, 2 /*ntopics*/, 2 /*nslowtopics*/,
                                               numParts /*npart*/, 100000 /*nmsgs*/, 1 * 1536));
    madmin.deleteStream(snameFull);
    //Because of bug 20495
    try {
      Thread.sleep(10 * 1000);
    } catch (Exception e) {
    }

    madmin.createStream(snameFull, sdesc);
    assertTrue(Producer.runTest(sname, 1 /*nstreams*/, 2 /*ntopics*/, 2 /*nslowtopics*/,
                                numParts /*npart*/, 100000 /*nmsgs*/, false /* multiflushers */));
    assertTrue(Listener.runTestWithPollOptions(sname, 1 /*nstreams*/, 2 /*ntopics*/, 2 /*nslowtopics*/,
                                               numParts /*npart*/, 100000 /*nmsgs*/, 10 * 1024));
    madmin.deleteStream(snameFull);
    //Because of bug 20495
    try {
      Thread.sleep(10 * 1000);
    } catch (Exception e) {
    }

    _logger.info("test with max fetch size lesser than msg size");
    madmin.createStream(snameFull, sdesc);
    assertTrue(Producer.runTest(sname, 1 /*nstreams*/, 2 /*ntopics*/, 2 /*nslowtopics*/,
                                numParts /*npart*/, 100000 /*nmsgs*/, false /* multiflushers */));
    try {
      ex = null;
      assertTrue(Listener.runTestWithPollOptions(sname, 1 /*nstreams*/, 2 /*ntopics*/, 2 /*nslowtopics*/,
                                                 numParts /*npart*/, 100000 /*nmsgs*/, 108 /*maxPartitionFetchSize*/));
      _logger.info("Test completed without errors !! It was expected to throw exception");
    } catch (Exception e) {
      _logger.info("Hit exception " + e);
      ex = e;
    }
    assertTrue(ex instanceof RecordTooLargeException);
    madmin.deleteStream(snameFull);
  }

  @Test
  public void testListenerBeforeMsgsAreFlushed() throws Exception {
    String sname = PREFIX + "listenerbeforemsgs";
    Thread producer;
    Thread consumer;
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    madmin.createStream(sname, sdesc);

    for (int i = 1; i < 4; i++) {
      String topicName = ":t" + i;
      //Producer
      Properties props = new Properties();
      props.put("key.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
      props.put("value.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
      props.put("streams.buffer.max.time.ms", "10000");
      KafkaProducer kafkaproducer = new KafkaProducer<byte[], byte[]>(props);

      //Listener
      Properties listenerProps = new Properties();
      listenerProps.put("key.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
      listenerProps.put("value.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
      listenerProps.put("fetch.min.bytes", "1");
      listenerProps.put("auto.offset.reset", "earliest");
      KafkaConsumer kafkaconsumer = new KafkaConsumer<byte[], byte[]>(listenerProps);

      //Producer Thread
      int numParts = 1;
      int numMsgs = 100;
      CountCallback cb = new CountCallback(numMsgs * numParts);

      SendMessagesToProducer worker = new SendMessagesToProducer(kafkaproducer, cb,
                                                                 sname + topicName, numParts, numMsgs);

      //Listener Thread
      ConsumeMessages listenWorker = new ConsumeMessages(kafkaconsumer, sname + topicName, numParts, numMsgs,
                                                         20 /*numpolls*/, false /*seekToEnd*/);
      producer = new Thread(worker);
      producer.start();
      try {
        Thread.sleep(2000);
      } catch (Exception e) {
        System.out.println("Sleep interrupted " + e);
      }

      consumer = new Thread(listenWorker);
      consumer.start();
      producer.join();
      consumer.join();
      assertTrue(listenWorker.hasPassed);
    }
    madmin.deleteStream(sname);
  }

  @Test
  public void testListenerWithSeekToEnd() throws Exception {
    String sname = PREFIX + "listenerseektoend";
    Thread producer;
    Thread consumer;
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    madmin.createStream(sname, sdesc);

    String topicName = ":t";
    //Producer
    Properties props = new Properties();
    props.put("key.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    props.put("value.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    KafkaProducer kafkaproducer = new KafkaProducer<byte[], byte[]>(props);

    //Listener
    Properties listenerProps = new Properties();
    listenerProps.put("key.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    listenerProps.put("value.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    listenerProps.put("fetch.min.bytes", "1");
    listenerProps.put("auto.offset.reset", "earliest");
    KafkaConsumer kafkaconsumer = new KafkaConsumer<byte[], byte[]>(listenerProps);
    madmin.createTopic(sname, "t", numParts);

    //Producer Thread
    int numParts = 1;
    int numMsgs = 10;
    CountCallback cb = new CountCallback(numMsgs * numParts);

    SendMessagesToProducer worker = new SendMessagesToProducer(kafkaproducer, cb,
                                                               sname + topicName, numParts, numMsgs);

    //Listener Thread
    ConsumeMessages listenWorker = new ConsumeMessages(kafkaconsumer, sname + topicName, numParts, numMsgs,
                                                       20 /*numpolls*/, true /*seekToEnd*/);
    producer = new Thread(worker);
    consumer = new Thread(listenWorker);
    consumer.start();
    try {
      Thread.sleep(2000);
    } catch (Exception e) {
      System.out.println("Sleep interrupted " + e);
    }

    producer.start();
    producer.join();
    consumer.join();
    assertTrue(listenWorker.hasPassed);
    madmin.deleteStream(sname);
  }

  @Test
  public void testListenerFirstPoll() throws Exception {
    String sname = PREFIX + "listenerfirstpoll";
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    madmin.createStream(sname, sdesc);

    String topicName = ":t";
    //Producer
    Properties props = new Properties();
    props.put("key.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    props.put("value.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    KafkaProducer kafkaproducer = new KafkaProducer<byte[], byte[]>(props);

    //Listener
    Properties listenerProps = new Properties();
    listenerProps.put("key.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    listenerProps.put("value.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    listenerProps.put("fetch.min.bytes", "1");
    listenerProps.put("auto.offset.reset", "earliest");
    KafkaConsumer kafkaconsumer = new KafkaConsumer<byte[], byte[]>(listenerProps);
    madmin.createTopic(sname, "t", numParts);

    //Producer Thread
    int numMsgs = 10;
    CountCallback cb = new CountCallback(numMsgs * 1);

    SendMessagesToProducer producer = new SendMessagesToProducer(kafkaproducer, cb,
                                                               sname + topicName, 1, numMsgs);
    producer.run();

    //Consumer
    List<String> topics = new ArrayList<String>();
    ConsumerRecords<byte[], byte[]> recs;
    topics.add(sname + topicName);
    for (int i = 0; i < 3; i++) {
      // Subscribe to topic
      kafkaconsumer.subscribe(topics);
      try {
        Thread.sleep(2000);
      } catch (Exception e) {
        System.out.println("Sleep interrupted " + e);
      }

      //First poll should return 0 msg
      recs = kafkaconsumer.poll(0);
      assertTrue(recs.count() == 0);
      recs = kafkaconsumer.poll(0);
      assertTrue(recs.count() == numMsgs);
      kafkaconsumer.unsubscribe();
    }

    //Check poll with non-zero timeout
    kafkaconsumer.subscribe(topics);
    try {
      Thread.sleep(2000);
    } catch (Exception e) {
      System.out.println("Sleep interrupted " + e);
    }
    recs = kafkaconsumer.poll(1);
    assertTrue(recs.count() == numMsgs);
    kafkaconsumer.unsubscribe();

    kafkaconsumer.close();

    madmin.deleteStream(sname);
  }
  
  @Test
  public void testListenerWithNullKeyValue() throws Exception {
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setCompressionAlgo("off");
    String sname = PREFIX + "listenerwithnullkeyvalue";

    madmin.createStream(sname, sdesc);
    String topicName = ":t";
    //Producer
    Properties props = new Properties();
    props.put("key.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    props.put("value.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    KafkaProducer kafkaproducer = new KafkaProducer<byte[], byte[]>(props);

    //Listener
    Properties listenerProps = new Properties();
    listenerProps.put("key.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    listenerProps.put("value.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    listenerProps.put("fetch.min.bytes", "1");
    listenerProps.put("auto.offset.reset", "earliest");
    KafkaConsumer kafkaconsumer = new KafkaConsumer<byte[], byte[]>(listenerProps);
    madmin.createTopic(sname, "t", 1);

    //Producer Thread
    int numMsgs = 256;
    CountCallback cb = new CountCallback(numMsgs * 1);

    SendMessagesToProducer producer = new SendMessagesToProducer(kafkaproducer, cb,
                                                               sname + topicName, 1, numMsgs, 0 /*msgSize*/);
    producer.run();

    //Consumer
    List<String> topics = new ArrayList<String>();
    ConsumerRecords<byte[], byte[]> recs;
    topics.add(sname + topicName);
    // Subscribe to topic
    kafkaconsumer.subscribe(topics);

    long seekOffset = 100;
    long expectedOffset = seekOffset;
    int totalNumMsgs = 0;
    Set<TopicPartition> subscribed = kafkaconsumer.assignment();
    assertTrue(subscribed.size() == 1);
    for (TopicPartition p : subscribed) {
      kafkaconsumer.seek(p, seekOffset);
    }
    try {
      Thread.sleep(2000);
    } catch (Exception e) {
      System.out.println("Sleep interrupted " + e);
    }

    while (true) {
      recs = kafkaconsumer.poll(1000);
      totalNumMsgs += recs.count();
      if (recs.count() == 0) {
        assertTrue("totalNumMsgs " + totalNumMsgs + " numMsgs " + numMsgs + " seekOffset " + seekOffset,
                   totalNumMsgs == (numMsgs - seekOffset + 1));
        break;
      }
      //Check offsets are sequential
      Iterator<ConsumerRecord<byte[], byte[]>> iter = recs.iterator();
      while (iter.hasNext()) {
        ConsumerRecord<byte[], byte[]> rec = iter.next();
        assertTrue(rec.offset() == expectedOffset);
        assertTrue(rec.key() == null);
        assertTrue(rec.value() == null);
        ++expectedOffset;
      }
    }
    kafkaconsumer.unsubscribe();

    kafkaconsumer.close();
    kafkaproducer.close();

    madmin.deleteStream(sname);
  }

  @Test
  public void testListenerSeqOffset() throws Exception {
    String sname = PREFIX + "listenerseqoffset";
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    madmin.createStream(sname, sdesc);

    String topicName = ":t";
    //Producer
    Properties props = new Properties();
    props.put("key.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    props.put("value.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    KafkaProducer kafkaproducer = new KafkaProducer<byte[], byte[]>(props);

    //Listener
    Properties listenerProps = new Properties();
    listenerProps.put("key.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    listenerProps.put("value.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    listenerProps.put("fetch.min.bytes", "1");
    listenerProps.put("auto.offset.reset", "earliest");
    KafkaConsumer kafkaconsumer = new KafkaConsumer<byte[], byte[]>(listenerProps);
    madmin.createTopic(sname, "t", numParts);

    //Producer Thread
    int numMsgs = 256;
    CountCallback cb = new CountCallback(numMsgs * 1);

    SendMessagesToProducer producer = new SendMessagesToProducer(kafkaproducer, cb,
                                                               sname + topicName, 1, numMsgs);
    producer.run();

    //Consumer
    List<String> topics = new ArrayList<String>();
    ConsumerRecords<byte[], byte[]> recs;
    topics.add(sname + topicName);
    // Subscribe to topic
    kafkaconsumer.subscribe(topics);
    try {
      Thread.sleep(2000);
    } catch (Exception e) {
      System.out.println("Sleep interrupted " + e);
    }

    //First poll should return 0 msg
    recs = kafkaconsumer.poll(0);
    assertTrue(recs.count() == 0);
    recs = kafkaconsumer.poll(0);
    assertTrue(recs.count() == numMsgs);

    //Check offsets are sequential
    Iterator<ConsumerRecord<byte[], byte[]>> iter = recs.iterator();
    long expectedOffset = 1;
    while (iter.hasNext()) {
      ConsumerRecord<byte[], byte[]> rec = iter.next();
      assertTrue(rec.offset() == expectedOffset);
      ++expectedOffset;
    }
    kafkaconsumer.unsubscribe();

    kafkaconsumer.close();

    madmin.deleteStream(sname);
  }

  @Test
  public void testListenerSeqOffsetWithSeek() throws Exception {
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setCompressionAlgo("off");
    for (int i = 0; i < 10; i++) {
      String sname = PREFIX + "listenerseqoffsetwithseek" + i;
      int msgSize = 5;
      if (i > 0)
        msgSize = i * 512;

      madmin.createStream(sname, sdesc);
      String topicName = ":t";
      //Producer
      Properties props = new Properties();
      props.put("key.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
      props.put("value.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
      KafkaProducer kafkaproducer = new KafkaProducer<byte[], byte[]>(props);

      //Listener
      Properties listenerProps = new Properties();
      listenerProps.put("key.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
      listenerProps.put("value.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
      listenerProps.put("fetch.min.bytes", "1");
      listenerProps.put("auto.offset.reset", "earliest");
      KafkaConsumer kafkaconsumer = new KafkaConsumer<byte[], byte[]>(listenerProps);
      madmin.createTopic(sname, "t", 1);

      //Producer Thread
      int numMsgs = 256;
      CountCallback cb = new CountCallback(numMsgs * 1);

      SendMessagesToProducer producer = new SendMessagesToProducer(kafkaproducer, cb,
                                                                 sname + topicName, 1, numMsgs, msgSize);
      producer.run();

      //Consumer
      List<String> topics = new ArrayList<String>();
      ConsumerRecords<byte[], byte[]> recs;
      topics.add(sname + topicName);
      // Subscribe to topic
      kafkaconsumer.subscribe(topics);

      long seekOffset = 100;
      long expectedOffset = seekOffset;
      int totalNumMsgs = 0;
      Set<TopicPartition> subscribed = kafkaconsumer.assignment();
      assertTrue(subscribed.size() == 1);
      for (TopicPartition p : subscribed) {
        kafkaconsumer.seek(p, seekOffset);
      }
      try {
        Thread.sleep(2000);
      } catch (Exception e) {
        System.out.println("Sleep interrupted " + e);
      }

      while (true) {
        recs = kafkaconsumer.poll(1000);
        totalNumMsgs += recs.count();
        if (recs.count() == 0) {
          assertTrue(totalNumMsgs == (numMsgs - seekOffset + 1));
          break;
        }
        //Check offsets are sequential
        Iterator<ConsumerRecord<byte[], byte[]>> iter = recs.iterator();
        while (iter.hasNext()) {
          ConsumerRecord<byte[], byte[]> rec = iter.next();
          assertTrue(rec.offset() == expectedOffset);
          assertTrue(rec.key().length == msgSize);
          assertTrue(rec.value().length == msgSize);
          ++expectedOffset;
        }
      }
      kafkaconsumer.unsubscribe();

      kafkaconsumer.close();
      kafkaproducer.close();

      madmin.deleteStream(sname);
    }
  }

  @Test
  public void testListenerZeroOffset() throws Exception {
    String sname = PREFIX + "listenerzerooffset";
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setCompressionAlgo("off");
    madmin.createStream(sname, sdesc);

    String topicName = ":t";
    //Producer
    Properties props = new Properties();
    props.put("key.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    props.put("value.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");

    //Listener
    Properties listenerProps = new Properties();
    listenerProps.put("key.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    listenerProps.put("value.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    listenerProps.put("fetch.min.bytes", "1");
    listenerProps.put("auto.offset.reset", "earliest");
    listenerProps.put("streams.zerooffset.record.on.eof", "true");
    KafkaConsumer kafkaconsumer = new KafkaConsumer<byte[], byte[]>(listenerProps);
    madmin.createTopic(sname, "t", numParts);
    //Consumer
    List<String> topics = new ArrayList<String>();
    ConsumerRecords<byte[], byte[]> recs;
    Iterator<ConsumerRecord<byte[], byte[]>> iter;
    int numZeroOffsetMsgs = 0;
    topics.add(sname + topicName);
    // Subscribe to topic
    kafkaconsumer.subscribe(topics);

    //Check zero offset on empty topic
    try {
      Thread.sleep(2000);
    } catch (Exception e) {
      System.out.println("Sleep interrupted " + e);
    }
    numZeroOffsetMsgs = 0;
    recs = kafkaconsumer.poll(1000);
    assertTrue(recs.count() == numParts);
    iter = recs.iterator();
    while (iter.hasNext()) {
      ConsumerRecord<byte[], byte[]> rec = iter.next();
      if (rec.offset() == 0)
        numZeroOffsetMsgs++;
    }
    assertTrue(numZeroOffsetMsgs == numParts);

    for (int i = 0; i < 10; i++) {
      //Producer Thread
      int numMsgs = 28;
      KafkaProducer kafkaproducer = new KafkaProducer<byte[], byte[]>(props);
      CountCallback cb = new CountCallback(numMsgs * numParts);
      SendMessagesToProducer producer = new SendMessagesToProducer(kafkaproducer, cb,
                                                                 sname + topicName, numParts, numMsgs);
      producer.run();

      numZeroOffsetMsgs = 0;
      recs = kafkaconsumer.poll(1000);
      assertTrue(recs.count() == numMsgs * numParts);
      iter = recs.iterator();
      while (iter.hasNext()) {
        ConsumerRecord<byte[], byte[]> rec = iter.next();
        assertTrue(rec.offset() > 0);
      }

      try {
        Thread.sleep(1000);
      } catch (Exception e) {
        System.out.println("Sleep interrupted " + e);
      }

      int recCount = 0;
      while (true) {
        recs = kafkaconsumer.poll(1000);
        if (recs.count() == 0)
          break;
        recCount += recs.count();
        iter = recs.iterator();
        while (iter.hasNext()) {
          ConsumerRecord<byte[], byte[]> rec = iter.next();
          if (rec.offset() == 0)
            ++numZeroOffsetMsgs;
        }
      }
      assertTrue(recCount == numParts);
      assertTrue(numZeroOffsetMsgs == numParts);
    }
    kafkaconsumer.unsubscribe();

    kafkaconsumer.close();

    madmin.deleteStream(sname);
  }

  @Test
  public void testListenerWithTabetSplit() throws Exception {
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setCompressionAlgo("off");
    for (int i = 0; i < 1; i++) {
      String sname = PREFIX + "listenerwithtabletsplit" + i;
      int msgSize = (i + 1) * 1024 * 1024;

      madmin.createStream(sname, sdesc);
      String topicName = ":t";
      //Producer
      Properties props = new Properties();
      props.put("key.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
      props.put("value.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
      KafkaProducer kafkaproducer = new KafkaProducer<byte[], byte[]>(props);

      //Listener
      Properties listenerProps = new Properties();
      listenerProps.put("key.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
      listenerProps.put("value.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
      listenerProps.put("fetch.min.bytes", "1");
      listenerProps.put("auto.offset.reset", "earliest");
      listenerProps.put("max.partition.fetch.bytes", 10 * 1024 * 1024);
      KafkaConsumer kafkaconsumer = new KafkaConsumer<byte[], byte[]>(listenerProps);
      madmin.createTopic(sname, "t", 1);

      //Producer Thread
      int numMsgs = 1000;
      CountCallback cb = new CountCallback(numMsgs * 1);

      SendMessagesToProducer producer = new SendMessagesToProducer(kafkaproducer, cb,
                                                                 sname + topicName, 1, numMsgs, msgSize);
      producer.run();

      //Consumer
      List<String> topics = new ArrayList<String>();
      ConsumerRecords<byte[], byte[]> recs;
      topics.add(sname + topicName);
      // Subscribe to topic
      kafkaconsumer.subscribe(topics);

      int totalNumMsgs = 0;
      try {
        Thread.sleep(1000);
      } catch (Exception e) {
        System.out.println("Sleep interrupted " + e);
      }

      while (true) {
        recs = kafkaconsumer.poll(2000);
        totalNumMsgs += recs.count();
        if (recs.count() == 0) {
          assertTrue(totalNumMsgs == numMsgs);
          break;
        }
      }
      kafkaconsumer.unsubscribe();

      kafkaconsumer.close();
      kafkaproducer.close();

      madmin.deleteStream(sname);
    }
  }

  @Test
  public void testListenerWithNullCommitted() throws Exception {
    String sname = PREFIX + "listenerwithnullcommit";
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    madmin.createStream(sname, sdesc);

    String topicName = ":t";
    TopicPartition p = new TopicPartition(sname + topicName, 0);
    //Listener
    Properties listenerProps = new Properties();
    listenerProps.put("key.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    listenerProps.put("value.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    listenerProps.put("fetch.min.bytes", "1");
    listenerProps.put("enable.auto.commit", false);
    listenerProps.put("auto.offset.reset", "earliest");
    listenerProps.put("streams.zerooffset.record.on.eof", "true");
    listenerProps.put("group.id", "testnullcommitted");
    KafkaConsumer kafkaconsumer = new KafkaConsumer<byte[], byte[]>(listenerProps);

    //Consumer subscribing
    List<String> topics = new ArrayList<String>();
    topics.add(sname + topicName);
    kafkaconsumer.subscribe(topics);

    try {
      Thread.sleep(2000);
    } catch (Exception e) {
    }

    //Without topic create committed should throw exception
    Exception ex = null;
    try {
      kafkaconsumer.committed(p);
    } catch (Exception e) {
      ex = e;
    }
    assertTrue(ex instanceof UnknownTopicOrPartitionException);

    madmin.createTopic(sname, "t", 1);
    try {
      Thread.sleep(2000);
    } catch (Exception e) {
    }
    assertTrue(kafkaconsumer.committed(p) == null);
  }

  @Test
  public void testListenerPosition() throws Exception {
    String sname = PREFIX + "testListenerPosition";
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    final AtomicBoolean rebalanceFinished = new AtomicBoolean(false);
    madmin.createStream(sname, sdesc);
    madmin.createTopic(sname, "t", 1);

    //Listener
    Properties listenerProps = new Properties();
    listenerProps.put("key.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    listenerProps.put("value.deserializer", "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    listenerProps.put("group.id", "testListenerPosition");
    listenerProps.put("enable.auto.commit", false);
    listenerProps.put("auto.offset.reset", "earliest");
    KafkaConsumer kafkaconsumer = new KafkaConsumer<byte[], byte[]>(listenerProps);
    KafkaConsumer consumer =  new KafkaConsumer<byte[], byte[]>(listenerProps);

    String topicName = ":t";
    TopicPartition tp = new TopicPartition(sname + topicName, 0);
    try {
      Map<TopicPartition,OffsetAndMetadata> offsets = new HashMap<>();
      offsets.put(tp, new OffsetAndMetadata(100));
      kafkaconsumer.subscribe(Arrays.asList(sname + topicName));
      kafkaconsumer.poll(0);
      kafkaconsumer.commitSync(offsets);
      kafkaconsumer.unsubscribe();
      consumer.subscribe(Arrays.asList(sname + topicName), new ConsumerRebalanceListener() {

            @Override
            public void onPartitionsAssigned(Collection<TopicPartition> partitions) {
                System.err.println("onPartitionsAssigned");
                rebalanceFinished.set(true);
            }

            @Override
            public void onPartitionsRevoked(Collection<TopicPartition> partitions) {
            }
        });

        // Waiting for consumer rebalance to finish, otherwise task will be initialized with outdated context
        while(!rebalanceFinished.get()) {
            // waiting for rebalance to finish
        }
      System.err.println("consumer position " + consumer.position(tp));
      assertTrue(consumer.position(tp) == 100);
    } catch (Exception e) {
      System.err.println(e);
      assertTrue(false);
    } finally {
      kafkaconsumer.unsubscribe();
      kafkaconsumer.close();
      consumer.unsubscribe();
      consumer.close();
      madmin.deleteStream(sname);
    }
  }

  public class ConsumeMessages implements Runnable {
    private String streamTopicName;
    private int numPartitions;
    private int numMsgsPerPartition;
    private KafkaConsumer consumer;
    private byte[] key;
    private byte[] value;
    private int numPolls;
    private int expectedNumMsgs;
    public boolean hasPassed;
    private boolean seekToEnd;

    public ConsumeMessages(KafkaConsumer c, String topicName,
                           int numparts, int numMessagesPerPartition, int npolls, boolean seekToEnd) {
      streamTopicName = topicName;
      numPartitions = numparts;
      numMsgsPerPartition = numMessagesPerPartition;
      consumer = c;
      numPolls = npolls;
      expectedNumMsgs = numMsgsPerPartition * numPartitions;
      this.seekToEnd = seekToEnd;
    }

    public void run() {
      int numConsumedMsgs = 0;
      hasPassed = false;

      // SeekToEnd if required
      if (seekToEnd) {
        System.out.println("Seeking to end");
        List<TopicPartition> partitions = new ArrayList<TopicPartition>();
        for (int i = 0; i < numPartitions; i++) {
          partitions.add(new TopicPartition(streamTopicName, i));
        }
        consumer.assign(partitions);
        Set<TopicPartition> subscribed = consumer.assignment();
        for (TopicPartition p : subscribed) {
          consumer.seekToEnd(p);
          System.out.println("Subscribed to " + p.topic() + " partition:" +
                             p.partition() + " position:" + consumer.position(p));
        }
      } else {
        List<String> topics = new ArrayList<String>();
        topics.add(streamTopicName);
        // Subscribe to topic
        consumer.subscribe(topics);
      }

      int i = 0;
      while (i < numPolls) {
        ConsumerRecords<byte[], byte[]> recs = consumer.poll(1000);
        numConsumedMsgs += recs.count();
        if (numConsumedMsgs == expectedNumMsgs)
          break;
        i++;
      }
      _logger.info("Msgs recived " + numConsumedMsgs +", expected " + expectedNumMsgs);
      consumer.close();
      if (numConsumedMsgs != expectedNumMsgs) {
        throw new RuntimeException("Num msgs received " + numConsumedMsgs + ", expected " + expectedNumMsgs);
      } else {
        hasPassed = true;
      }
    }
  }
}

