/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.streams.tests.admin;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.hadoop.conf.Configuration;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.streams.Admin;
import com.mapr.streams.StreamDescriptor;
import com.mapr.streams.Streams;
import com.mapr.streams.impl.admin.MarlinAdminImpl;
import com.mapr.streams.impl.admin.TopicFeedInfo;
import com.mapr.fs.proto.Marlinserver.MarlinTopicMetaEntry;
import com.mapr.fs.proto.Marlinserver.MarlinConfigDefaults;

import com.mapr.tests.BaseTest;
import com.mapr.tests.annotations.ClusterTest;

import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.ProducerConfig;

@Category(ClusterTest.class)
public class TopicTest extends BaseTest {
  private static final Logger _logger = LoggerFactory.getLogger(TopicTest.class);
  private static final String STREAM = "/jtest-" + TopicTest.class.getSimpleName();
  private static Admin madmin;

  @BeforeClass
  public static void setupTest() throws Exception {
    final Configuration conf = new Configuration();
    madmin = Streams.newAdmin(conf);

    try {
      madmin.deleteStream(STREAM);
    } catch (Exception e) {}

    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    madmin.createStream(STREAM, sdesc);
  }

  @AfterClass
  public static void cleanupTest() throws Exception {
    madmin.deleteStream(STREAM);
  }

  @Test
  public void testCreate() throws IOException {
    String topicName = "testCreate";
    String topicFullName = STREAM + ":" + topicName;
    MarlinTopicMetaEntry mentry;
    Exception ex;

    _logger.info("Verify topic create");
    madmin.createTopic(STREAM, topicName, 4 /*numFeeds*/);

    MarlinAdminImpl iadmin = (MarlinAdminImpl)madmin;
    mentry = iadmin.getTopicMetaEntry(topicFullName);
    assertTrue(mentry.getFeedIdsList().size() == 4);
    assertTrue(mentry.getIsDeleted() == false);

    _logger.info("Test topic create with same name");
    ex = null;
    try {
      madmin.createTopic(STREAM, topicName);
    } catch (Exception e) {
      ex = e;
    }
    assertTrue(ex != null);

    ex = null;
    try {
      madmin.createTopic(STREAM, "test.DotTopic");
    } catch (Exception e) {
      ex = e;
    }
    assertTrue(ex == null);
  }

  @Test
  public void testEdit() throws IOException {
    String topicName = "testEdit";
    String topicFullName = STREAM + ":" + topicName;
    MarlinTopicMetaEntry mentry;
    Exception ex;

    madmin.createTopic(STREAM, topicName, 4 /*numFeeds*/);

    _logger.info("Verify topic edit");
    madmin.editTopic(STREAM, topicName, 8 /*numFeeds*/);

    MarlinAdminImpl iadmin = (MarlinAdminImpl)madmin;
    mentry = iadmin.getTopicMetaEntry(topicFullName);
    assertTrue(mentry.getFeedIdsList().size() == 8);
    assertTrue(mentry.getIsDeleted() == false);

    _logger.info("Test topic edit with reduced feeds");
    ex = null;
    try {
      madmin.editTopic(STREAM, topicName, 6);
    } catch (Exception e) {
      ex = e;
    }
    assertTrue(ex != null);
  }

  @Test
  public void testDelete() throws IOException {
    String topicName = "testDelete";
    String topicFullName = STREAM + ":" + topicName;
    MarlinTopicMetaEntry mentry;
    Exception ex;

    MarlinAdminImpl iadmin = (MarlinAdminImpl)madmin;

    madmin.createTopic(STREAM, topicName, 4 /*numFeeds*/);
    mentry = iadmin.getTopicMetaEntry(topicFullName);
    int uniq = mentry.getTopicUniq();
    assertTrue(uniq >= 1);

    _logger.info("Verify topic delete");
    madmin.deleteTopic(STREAM, topicName);

    mentry = iadmin.getTopicMetaEntry(topicFullName);
    assertTrue(mentry.getIsDeleted() == true);

    _logger.info("Test topic delete on deleted topic");
    ex = null;
    try {
      madmin.deleteTopic(STREAM, topicName);
    } catch (Exception e) {
      ex = e;
    }
    assertTrue(ex != null);

    _logger.info("Verify topic recreate");
    madmin.createTopic(STREAM, topicName, 1 /*numFeeds*/);
    mentry = iadmin.getTopicMetaEntry(topicFullName);
    assertTrue(mentry.getIsDeleted() == false);
    assertTrue(mentry.getFeedIdsList().size() == 1);
    assertTrue(mentry.getTopicUniq() > uniq);
  }

  @Test
  public void testAutoOptions() throws IOException {
    String topicName = "testAutoOptions";
    String topicFullName = STREAM + ":" + topicName;
    MarlinTopicMetaEntry mentry;
    Exception ex;

    _logger.info("Test #feeds is picked up from the default value on stream");
    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    sdesc.setDefaultPartitions(3);
    madmin.editStream(STREAM, sdesc);

    madmin.createTopic(STREAM, topicName);

    MarlinAdminImpl iadmin = (MarlinAdminImpl)madmin;
    mentry = iadmin.getTopicMetaEntry(topicFullName);
    assertTrue(mentry.getIsDeleted() == false);
    assertTrue(mentry.getFeedIdsList().size() == 3);
  }

  @Test
  public void testError() throws IOException {
    MarlinTopicMetaEntry mentry;
    Exception ex;

    _logger.info("Test create with bad topic name");
    ex = null;
    try {
      madmin.createTopic("/xyz", "");
    } catch (Exception e) {
      ex = e;
    }
    assertTrue(ex != null);

    _logger.info("Test create with bad stream name");
    ex = null;
    try {
      madmin.createTopic("", "xyz");
    } catch (Exception e) {
      ex = e;
    }
    assertTrue(ex != null);

    _logger.info("Test create with non-existent stream");
    ex = null;
    try {
      madmin.createTopic("/non-existent", "xyz");
    } catch (Exception e) {
      ex = e;
    }
    assertTrue(ex != null);

    _logger.info("Test edit with non-existent topic");
    ex = null;
    try {
      madmin.editTopic(STREAM, "non-existent", 4);
    } catch (Exception e) {
      ex = e;
    }
    assertTrue(ex != null);

    _logger.info("Test delete with non-existent topic");
    ex = null;
    try {
      madmin.deleteTopic(STREAM, "non-existent");
    } catch (Exception e) {
      ex = e;
    }
    assertTrue(ex != null);

    _logger.info("Test getTopicMetaEntry on non-existent topic");
    ex = null;
    try {
      MarlinAdminImpl iadmin = (MarlinAdminImpl)madmin;
      mentry = iadmin.getTopicMetaEntry(STREAM + ":non-existent");
    } catch (Exception e) {
      ex = e;
    }
    assertTrue(ex != null);
  }

  @Test
  public void testTestCount() throws IOException {
    String streamName = STREAM + "2";
    Map<String, List<TopicFeedInfo>> map;
    List<TopicFeedInfo> flist;
    String sampleTopic = "testCount" + 5;
    String sampleFullTopic = streamName + ":" + sampleTopic;
    Exception ex;

    StreamDescriptor sdesc = Streams.newStreamDescriptor();
    madmin.createStream(streamName, sdesc);

    _logger.info("test on empty stream");
    int count = madmin.countTopics(streamName);
    assertTrue(count == 0);

    MarlinAdminImpl iadmin = (MarlinAdminImpl)madmin;
    map = iadmin.listTopics(streamName);
    assertTrue(map.size() == 0);

    ex = null;
    try {
      flist = iadmin.infoTopic(sampleTopic);
    } catch (Exception e) {
      ex = e;
    }
    assertTrue(ex != null);

    _logger.info("Create 10 topics & test count");
    for (int i = 0; i < 10; ++i)
      madmin.createTopic(streamName, "testCount" + i);

    count = madmin.countTopics(streamName);
    assertTrue(count == 10);

    map = iadmin.listTopics(streamName);
    assertTrue(map.size() == 10);
    assertTrue(map.containsKey(sampleTopic));

    flist = iadmin.infoTopic(sampleFullTopic);
    assertTrue(flist.size() == 1);

    _logger.info("delete 5 topics & test count");
    for (int i = 0; i < 5; ++i)
      madmin.deleteTopic(streamName, "testCount" + i);

    count = madmin.countTopics(streamName);
    _logger.info("count after deletes : " + count);
    assertTrue(count == 5);

    map = iadmin.listTopics(streamName);
    assertTrue(map.size() == 5);
    assertTrue(map.containsKey(sampleTopic));
    assertTrue(!map.containsKey(streamName + ":testCount" + 4));

    flist = iadmin.infoTopic(sampleFullTopic);
    assertTrue(flist.size() == 1);

    ex = null;
    try {
      flist = iadmin.infoTopic(streamName + ":testCount" + 4);
    } catch (Exception e) {
      ex = e;
    }
    assertTrue(ex != null);

    _logger.info("recreate 2 topics & test count");
    for (int i = 0; i < 2; ++i)
      madmin.createTopic(streamName, "testCount" + i);

    count = madmin.countTopics(streamName);
    assertTrue(count == 7);

    map = iadmin.listTopics(streamName);
    assertTrue(map.size() == 7);

    madmin.deleteStream(streamName);
  }

  @Test
  public void testFiveThousandTopics() throws IOException {
    String sname = "/jtest-fivethousand";
    int numTopics = 5000;
    try {
      madmin.deleteStream(sname);
    }catch (Exception e) {}

    madmin.createStream(sname, Streams.newStreamDescriptor());

    Properties props = new Properties();
    MarlinConfigDefaults cdef = MarlinConfigDefaults.getDefaultInstance();
    props.put("bootstrap.servers", "127.0.0.1:7222");
    props.put("key.serializer",
              "org.apache.kafka.common.serialization.ByteArraySerializer");
    props.put("value.serializer",
              "org.apache.kafka.common.serialization.ByteArraySerializer");
    KafkaProducer producer = new KafkaProducer<byte[], byte[]>(props);

    byte[] value = new byte[200];
    for (int i = 0; i < numTopics; ++i) {
      String topicFullName = sname + ":" + i;

      ProducerRecord<byte[], byte[]> record =
        new ProducerRecord<byte[], byte[]>(topicFullName, value);
      producer.send(record);
      if (i % 1000 == 0) {
        System.out.println("sending " + i);
      }
    }
    producer.flush();
    producer.close();

    int count = madmin.countTopics(sname);
    assertTrue(count == numTopics);

    madmin.deleteStream(sname);
  }

  @Test
  @Ignore("This test takes very, very long")
  public void testMillionTopics() throws IOException {
    String sname = "/jtest-million";
    madmin.createStream(sname, Streams.newStreamDescriptor());

    Properties props = new Properties();
    MarlinConfigDefaults cdef = MarlinConfigDefaults.getDefaultInstance();
    props.put("bootstrap.servers", "127.0.0.1:7222");
    props.put("key.serializer",
              "org.apache.kafka.common.serialization.ByteArraySerializer");
    props.put("value.serializer",
              "org.apache.kafka.common.serialization.ByteArraySerializer");
    KafkaProducer producer = new KafkaProducer<byte[], byte[]>(props);

    byte[] value = new byte[200];
    int numTopics = 1000000;
    for (int i = 0; i < numTopics; ++i) {
      String topicFullName = sname + ":" + i;

      ProducerRecord<byte[], byte[]> record =
        new ProducerRecord<byte[], byte[]>(topicFullName, value);
      producer.send(record);
      if (i % 10000 == 0) {
        System.out.println("sending " + i);
        // Recreate producer to avoid highmem usage on client
        //producer.close();
        //producer = new KafkaProducer<byte[], byte[]>(props);
      }
    }
    producer.flush();
    producer.close();

    int count = madmin.countTopics(sname);
    assertTrue(count == numTopics);

    madmin.deleteStream(sname);
  }
}
