/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.streams.listener;

import com.mapr.streams.producer.ProducerPerformance;

import java.io.*;
import java.nio.ByteBuffer;
import java.util.*;

import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRebalanceListener;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;
import org.apache.kafka.common.TopicPartition;


public class ListenerTest {
 
  public static String streamName;
  public static int numTopics = 2;
  public static int numPartitions = 4;
  public static int numMsgsPerPartition = 1000000;
  public static int numBatches = 1;
  public static long batchSleepMs = 10*1000;

  public static boolean verifyKeys = true;
  public static boolean keysInOrder = false;
  public static boolean allowDuplicateKeys = false;
  public static boolean isTracingEnabled = false;
  public static boolean printProgress = false;

  public static void usage() {
    System.err.println("ListenerTest -path <stream-full-name> [-ntopics <#topics>");
    System.err.println(" -npart <#partitions/topic> -nmsgs <#msgs/topicfeed> -keysinorder <True/False>");
    System.err.println(" -verifykesy <True/False> -debug <True/False>] -progress <True/False>");
    System.exit(1);
  }

  public static void main(String[] args) throws Exception {
    for (int i = 0; i < args.length; ++i) {
      if (args[i].equals("-path")) {
        i++;
        if (i >= args.length) usage();
        streamName = args[i];
      } else if (args[i].equals("-ntopics")) {
        i++;
        if (i >= args.length) usage();
        numTopics = Integer.parseInt(args[i]);
      } else if (args[i].equals("-npart")) {
        i++;
        if (i >= args.length) usage();
        numPartitions = Integer.parseInt(args[i]);
      } else if (args[i].equals("-nmsgs")) {
        i++;
        if (i >= args.length) usage();
        numMsgsPerPartition = Integer.parseInt(args[i]);
      } else if (args[i].equals("-verifykeys")) {
        i++;
        if (i >= args.length) usage();
        verifyKeys = Boolean.parseBoolean(args[i]);
      } else if (args[i].equals("-keysinorder")) {
        i++;
        if (i >= args.length) usage();
        keysInOrder = Boolean.parseBoolean(args[i]);
      } else if (args[i].equals("-debug")) {
        i++;
        if (i >= args.length) usage();
        isTracingEnabled = Boolean.parseBoolean(args[i]);
      } else if (args[i].equals("-nbatches")) {
        i++;
        if (i >= args.length) usage();
        numBatches = Integer.parseInt(args[i]);
      } else if (args[i].equals("-progress")) {
        i++;
        if (i >= args.length) usage();
        printProgress = Boolean.parseBoolean(args[i]);
      } else {
        usage();
      }
    }

    runTest();
  }

  public static boolean runStressTest(String stream,
                                      int nmsgs,
                                      int ntopics,
                                      int npartitions,
                                      int nbatches,
                                      long batchsleepms,
                                      boolean keysOrdered) throws IOException {
    streamName = stream;
    numMsgsPerPartition = nmsgs;
    numTopics = ntopics;
    numPartitions = npartitions;
    keysInOrder = keysOrdered;
    numBatches = nbatches;
    batchSleepMs = batchsleepms;
    return runTest();
  }

  public static boolean runTest() throws IOException {
    try {
      ListenerPerformance lp = new ListenerPerformance(streamName,
                                                       numTopics,
                                                       numPartitions,
                                                       numMsgsPerPartition,
                                                       numBatches,
                                                       verifyKeys,
                                                       keysInOrder,
                                                       allowDuplicateKeys,
                                                       isTracingEnabled,
                                                       printProgress,
                                                       null /*groupId*/,
                                                       "ListenerTest",
                                                       false /*topicSubscriptions*/);

      Thread lt = new Thread(lp);
      lt.start();

      ProducerPerformanceThread pp = new ProducerPerformanceThread(streamName,
                                                       numTopics,
                                                       numPartitions,
                                                       numMsgsPerPartition,
                                                       numBatches,
                                                       batchSleepMs,
                                                       verifyKeys,
                                                       keysInOrder,
                                                       printProgress);
      Thread pt = new Thread(pp);
      pt.start();

      pt.join();
      lt.join();
      if ((lp.status == false) || (pp.status == false)) {
        System.out.println("ListenerPerformance test failed.");
        return false;
      }
    } catch (InterruptedException e) {
      System.out.println(e);
      return false;
    }
    return true;
  }

  private static class ProducerPerformanceThread implements Runnable {
    public String stream;
    public int nmsgs;
    public int ntopics;
    public int npartitions;
    public int nbatches;
    public long batchsleepms;
    public boolean status;
    public boolean mflushers;
    public boolean verifyKeys;
    public boolean printProgress;

    public ProducerPerformanceThread(String stream, int ntopics, int npartitions, int nmsgs,
                               int nbatches, long batchSleepMs, boolean verifyKeys, boolean ordered, boolean printProgress) {
      this.stream = stream;
      this.nmsgs = nmsgs;
      this.ntopics = ntopics;
      this.npartitions = npartitions;
      this.nbatches = nbatches;
      this.batchsleepms = batchSleepMs;
      this.status = false;
      this.verifyKeys = verifyKeys;
      this.mflushers = !ordered;
      this.printProgress = printProgress;
    }

    public void run() {
      try {
        status = ProducerPerformance.runStressTest(stream,
                                                   nmsgs,
                                                   ntopics,
                                                   npartitions,
                                                   nbatches,
                                                   batchsleepms,
                                                   verifyKeys,
                                                   mflushers,
                                                   printProgress,
                                                   false /* roundrobin*/,
                                                   false /*hashkey*/);
      } catch (Exception e) {
        System.out.println(e);
        status = false;
      }
    }
  }
}

