/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.streams.listener;

import java.io.*;
import java.nio.ByteBuffer;
import java.util.*;
import static org.junit.Assert.assertTrue;

import org.apache.hadoop.conf.Configuration;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRebalanceListener;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;
import org.apache.kafka.common.TopicPartition;
import com.mapr.streams.impl.admin.CursorInfo;

import com.mapr.streams.Admin;
import com.mapr.streams.impl.admin.MarlinAdminImpl;
import com.mapr.streams.Streams;
import com.mapr.streams.StreamDescriptor;
import com.mapr.fs.proto.Marlinserver.MarlinConfigDefaults;
import org.apache.kafka.common.KafkaException;

public class Listener implements Runnable {
  public static String topicIntial = new String("topic");
  private Configuration conf;
  private Admin madmin;
  private String streamName;
  private int numStreams = 2;
  private int numTopics = 2;
  private int numSlowTopics = 2;
  private int numPartitions = 4;
  private int numExpectedMsgs = 1000;
  private int maxPartitionFetchSize = 64 * 1024;
  private boolean verifyKeys = true;
  private boolean keysInOrder = false;
  private boolean allowDuplicateKeys = false;
  private boolean isTracingEnabled = false;
  private boolean printStats = true;
  private TED_ACTION action;
  private boolean topicSubscription = false;
  private String groupId = null;
  private Hashtable<PartitionInfo, Integer> partitionSeqMap;
  private Hashtable<PartitionInfo, boolean[]> partitionBArrayMap;
  public boolean status;

  public static void usage() {
    System.err.println("Listener -path <stream-intialname> [-nstreams <num streams> -ntopics <num topics> -nslowtopics <num topics> -npart <numpartitions per topic> -nmsgs <num messages per topicfeed>]");
    System.exit(1);
  }

  public static void main(String[] args) throws Exception {
    String streamName = null;
    int numStreams = 2;
    int numTopics = 2;
    int numSlowTopics = 2;
    int numPartitions = 4;
    int numExpectedMsgs = 1000;
    boolean verifyKeys = true;
    boolean keysInOrder = true;
    boolean allowDuplicateKeys = false;
    boolean isTracingEnabled = false;
    TED_ACTION action = TED_ACTION.kNone;
    boolean topicSubscription = false;
    String groupId = null;
    int maxPartitionFetchSize = 64 * 1024;

    for (int i = 0; i < args.length; ++i) {
      if (args[i].equals("-path")) {
        i++;
        if (i >= args.length) usage();
        streamName = args[i];
      } else if (args[i].equals("-nstreams")) {
        i++;
        if (i >= args.length) usage();
        numStreams = Integer.parseInt(args[i]);
      } else if (args[i].equals("-ntopics")) {
        i++;
        if (i >= args.length) usage();
        numTopics = Integer.parseInt(args[i]);
      } else if (args[i].equals("-nslowtopics")) {
        i++;
        if (i >= args.length) usage();
        numSlowTopics = Integer.parseInt(args[i]);
      } else if (args[i].equals("-npart")) {
        i++;
        if (i >= args.length) usage();
        numPartitions = Integer.parseInt(args[i]);
      } else if (args[i].equals("-nmsgs")) {
        i++;
        if (i >= args.length) usage();
        numExpectedMsgs = Integer.parseInt(args[i]);
      } else if (args[i].equals("-verifykeys")) {
        i++;
        if (i >= args.length) usage();
        verifyKeys = Boolean.parseBoolean(args[i]);
      } else if (args[i].equals("-keysinorder")) {
        i++;
        if (i >= args.length) usage();
        keysInOrder = Boolean.parseBoolean(args[i]);
      } else if (args[i].equals("-debug")) {
        i++;
        if (i >= args.length) usage();
        isTracingEnabled = Boolean.parseBoolean(args[i]);
      } else if (args[i].equals("-allowduplicates")) {
        i++;
        if (i >= args.length) usage();
        allowDuplicateKeys = Boolean.parseBoolean(args[i]);
      } else if (args[i].equals("-tedaction")) {
        i++;
        if (i >= args.length) usage();
        action = TED_ACTION.parseTedAction(Integer.parseInt(args[i]));
        System.out.println("Setting tedaction to " + action);
      } else if (args[i].equals("-topicsubscr")) {
        i++;
        if (i >= args.length) usage();
        topicSubscription = Boolean.parseBoolean(args[i]);
      } else if (args[i].equals("-groupid")) {
        i++;
        if (i >= args.length) usage();
        groupId = args[i];
      } else if (args[i].equals("-maxfetchsize")) {
        i++;
        if (i >= args.length) usage();
        maxPartitionFetchSize = Integer.parseInt(args[i]);
      } else {
        usage();
      }
    }
    
    Listener lp = new Listener(streamName, numStreams, numTopics, numSlowTopics, numPartitions, numExpectedMsgs, verifyKeys, keysInOrder, allowDuplicateKeys, isTracingEnabled,
                               true /*printStats*/, action, topicSubscription,
                               groupId, maxPartitionFetchSize);
    Thread lt = new Thread(lp);
    lt.start();
 
    lt.join();
    if (lp.status == false) {
      System.out.println("Listener test failed.");
    }
  }

  public static boolean runLGTest(String streamName, int numStreams, int numTopics,
                                  int numPartitions, boolean isTracingEnabled, String groupId) {

    KafkaConsumer l1 = null;
    KafkaConsumer l2 = null;
    KafkaConsumer l3 = null;
    List<TopicPartition> partitions = new ArrayList<TopicPartition>();
    List<String> topicList = new ArrayList<String>();
    try {
       Configuration conf;
       Admin madmin = null;
       conf = new Configuration();
       try {
         madmin = Streams.newAdmin(conf);
       } catch (Exception e) {
         e.printStackTrace();
       }

       int totalNumTopics = numTopics;

       for (int sIdx = 0; sIdx < numStreams; sIdx++) {
         for (int i = 0; i < totalNumTopics; i++) {
           String streamTopicName = streamName + sIdx + ":" + topicIntial + i;
           topicList.add(streamTopicName);
          for (int j = 0; j < numPartitions; j++) {
            partitions.add(new TopicPartition(streamTopicName, j));
          }
        }
      }

      Properties props = new Properties();
      props.put("key.deserializer",
          "org.apache.kafka.common.serialization.ByteArrayDeserializer");
      props.put("value.deserializer",
          "org.apache.kafka.common.serialization.ByteArrayDeserializer");
      props.put("auto.offset.reset", "earliest");
      props.put("max.partition.fetch.bytes", 64 * 1024);
      if (groupId != null) {
        props.put("group.id", groupId);
      }

      ByteArrayDeserializer keyD = new ByteArrayDeserializer();
      ByteArrayDeserializer valueD = new ByteArrayDeserializer();
      l1 = new KafkaConsumer<byte[], byte[]>(props, keyD, valueD);

      RebCb cb = new RebCb(partitions);
      cb.addConsumer(l1);

      l1.subscribe(topicList, cb);

      /* Now create these topics and then verify we actually subscribe to them
       * */
      System.out.println("Creating topics");
      for (String topic : topicList) {
        try {
          String[] tokens = topic.split(":");
          madmin.createTopic(tokens[0], tokens[1], numPartitions);
        } catch (Exception e) {
          System.out.println("Topic create failed");
        }
      }

      System.out.println("Waiting for topic subscriptions");
      cb.WaitForAllSubscribed();

      Set<TopicPartition> subscribedl1 = l1.assignment();
      Set<String> subscribedTopicsl1 = l1.subscription();
      System.out.println("All subscribed: L1, partitions "  + subscribedl1.size()
                         + ", topics " + subscribedTopicsl1.size());

      if (subscribedl1.size() != numStreams * totalNumTopics * numPartitions) {
        System.out.println("All subscribed through L1, expected " +
                           (numStreams * totalNumTopics * numPartitions) +
                           " but got " + subscribedl1.size());
        return false;
      }
      if (subscribedTopicsl1.size() != numStreams * totalNumTopics) {
        System.out.println("All subscribed through L1, expected " +
                           (numStreams * totalNumTopics) +
                           " but got " + subscribedTopicsl1.size());
        return false;
      }

      /* Now start the second listener and wait for the revoke and assign calls
       * to come through. */
      l2 = new KafkaConsumer<byte[], byte[]>(props, keyD, valueD);
      cb.addConsumer(l2);

      /* subscribe l2 to the same topics */
      l2.subscribe(topicList, cb);

      cb.WaitForRevoke();
      cb.WaitForAllSubscribed();

      subscribedl1 =  l1.assignment();
      subscribedTopicsl1 = l1.subscription();
      Set<TopicPartition> subscribedl2 = l2.assignment();
      Set<String> subscribedTopicsl2 = l2.subscription();
      System.out.println("Two listeners: L1 (partitions " + subscribedl1.size() 
                         + ", topics " + subscribedTopicsl1.size() + "), L2 (partitiosn "
                         + subscribedl2.size() + ", topics " + subscribedTopicsl2.size()+ ")");

      if ((subscribedl1.size() + subscribedl2.size()) != (numStreams *
            totalNumTopics * numPartitions)) {
        System.out.println("Expected total subscriptions " +   
            (numStreams * totalNumTopics * numPartitions) + " found:" +
            (subscribedl1.size() + subscribedl2.size()));
        return false;
      }
      Set<String> totalSubscription = new HashSet<String>();
      totalSubscription.addAll(subscribedTopicsl1);
      totalSubscription.addAll(subscribedTopicsl2);
      if (totalSubscription.size() != (numStreams * totalNumTopics)) {
        System.out.println("Expected total subscriptions " +   
                           (numStreams * totalNumTopics) + " found:" +
                           (totalSubscription.size()));
        return false;
      }

      int maxPartitions = ((numPartitions / 2) + 1) * (numStreams * numTopics);
      int minPartitions = ((numPartitions / 2)) * (numStreams * numTopics);

      if (subscribedl1.size() < minPartitions || subscribedl1.size() > maxPartitions ||
          subscribedl2.size() < minPartitions || subscribedl2.size() > maxPartitions) {

        System.out.println("l1 subscription size " + subscribedl1.size());
        System.out.println("l2 subscription size " + subscribedl2.size());
        return false;
      }

      /* Change the number of feeds now */
      numPartitions *= 2;
      List<TopicPartition> newPartitions = new ArrayList<TopicPartition>();

      for (int sIdx = 0; sIdx < numStreams; sIdx++) {
        for (int i = 0; i < totalNumTopics; i++) {
          String streamTopicName = streamName + sIdx + ":" + topicIntial + i;
          for (int j = 0; j < numPartitions; j++) {
            newPartitions.add(new TopicPartition(streamTopicName, j));
          }
        }
      }

      cb.Reallocate(newPartitions);

      /* change the actual partitions */
      for (String topic : topicList) {
        try {
          String[] tokens = topic.split(":");
          madmin.editTopic(tokens[0], tokens[1], numPartitions);
        } catch (Exception e) {
          System.out.println("Topic edit failed");
        }
      }

      /* Wait for the new partitions to get assigned */
      cb.WaitForAllSubscribed();

      System.out.println("Checking subscriptions after partition change");
      subscribedl1 =  l1.assignment();
      subscribedTopicsl1 = l1.subscription();
      subscribedl2 = l2.assignment();
      subscribedTopicsl2 = l2.subscription();

      System.out.println("Partition change: L1 (partitions " + subscribedl1.size() 
                         + ", topics " + subscribedTopicsl1.size() + "), L2 (partitiosn "
                         + subscribedl2.size() + ", topics " + subscribedTopicsl2.size()+ ")");

      if ((subscribedl1.size() + subscribedl2.size()) != (numStreams *
            totalNumTopics * numPartitions)) {
        System.out.println("Expected total subscriptions " +   
            (numStreams * totalNumTopics * numPartitions) + " found:" +
            (subscribedl1.size() + subscribedl2.size()));
        return false;
      }
      totalSubscription.clear();
      totalSubscription.addAll(subscribedTopicsl1);
      totalSubscription.addAll(subscribedTopicsl2);
      if (totalSubscription.size() != (numStreams * totalNumTopics)) {
        System.out.println("Expected total subscriptions " +   
                           (numStreams * totalNumTopics) + " found:" +
                           (totalSubscription.size()));
        return false;
      }

      maxPartitions = ((numPartitions / 2) + 1) * (numStreams * numTopics);
      minPartitions = ((numPartitions / 2)) * (numStreams * numTopics);

      if (subscribedl1.size() < minPartitions || subscribedl1.size() > maxPartitions ||
          subscribedl2.size() < minPartitions || subscribedl2.size() > maxPartitions) {

        System.out.println("l1 subscription size " + subscribedl1.size());
        System.out.println("l2 subscription size " + subscribedl2.size());
        return false;
      }

      /* Now bring in another listener */
      l3 = new KafkaConsumer<byte[], byte[]>(props, keyD, valueD);
      cb.addConsumer(l3);

      /* subscribe l3 to the same topics */
      l3.subscribe(topicList, cb);

      cb.WaitForRevoke();
      cb.WaitForAllSubscribed();

      subscribedl1 = l1.assignment(); 
      subscribedl2 = l2.assignment();
      Set<TopicPartition> subscribedl3 = l3.assignment();
      subscribedTopicsl1 = l1.subscription();
      subscribedTopicsl2 = l2.subscription();
      Set<String> subscribedTopicsl3 = l3.subscription();

      System.out.println("New listener L3 added: " +
                         "L1 (partitions " + subscribedl1.size() 
                         + ", topics " + subscribedTopicsl1.size() + "), L2 (partitiosn "
                         + subscribedl2.size() + ", topics " + subscribedTopicsl2.size()
                         + "), L3 (partitions " + subscribedl3.size() + ", topics " + 
                         subscribedTopicsl3.size() + ")");

      if ((subscribedl1.size() + subscribedl2.size() + subscribedl3.size()) != (numStreams *
            totalNumTopics * numPartitions)) {
        System.out.println("Expected total subscriptions " +   
            (numStreams * totalNumTopics * numPartitions) + " found:" +
            (subscribedl1.size() + subscribedl2.size() + subscribedl3.size()));
        return false;
      }
      totalSubscription.clear();
      totalSubscription.addAll(subscribedTopicsl1);
      totalSubscription.addAll(subscribedTopicsl2);
      totalSubscription.addAll(subscribedTopicsl3);
      if (totalSubscription.size() != (numStreams * totalNumTopics)) {
        System.out.println("Expected total subscriptions " +   
                           (numStreams * totalNumTopics) + " found:" +
                           (totalSubscription.size()));
        return false;
      }

      maxPartitions = ((numPartitions / 3) + 1) * (numStreams * numTopics);
      minPartitions = ((numPartitions / 3)) * (numStreams * numTopics);

      if (subscribedl1.size() < minPartitions || subscribedl1.size() > maxPartitions ||
          subscribedl2.size() < minPartitions || subscribedl2.size() > maxPartitions ||
          subscribedl3.size() < minPartitions || subscribedl3.size() > maxPartitions) {

        System.out.println("l1 subscription size " + subscribedl1.size());
        System.out.println("l2 subscription size " + subscribedl2.size());
        System.out.println("l3 subscription size " + subscribedl3.size());
        return false;
      }

      /* Now take down l2 and make sure l1 and l3 get all the subscriptions */
      l2.unsubscribe();

      cb.WaitForRevoke();
      cb.WaitForAllSubscribed();
      subscribedl1 =  l1.assignment();
      subscribedl2 = l2.assignment();
      subscribedl3 = l3.assignment();
      subscribedTopicsl1 = l1.subscription();
      subscribedTopicsl2 = l2.subscription();
      subscribedTopicsl3 = l3.subscription();

      System.out.println("L2 unsubscribed: " +
                         "L1 (partitions " + subscribedl1.size() 
                         + ", topics " + subscribedTopicsl1.size() + "), L2 (partitiosn "
                         + subscribedl2.size() + ", topics " + subscribedTopicsl2.size()
                         + "), L3 (partitions " + subscribedl3.size() + ", topics " + 
                         subscribedTopicsl3.size() + ")");

      if ((subscribedl1.size() + subscribedl3.size()) != (numStreams *
            totalNumTopics * numPartitions)) {
        System.out.println("Expected total subscriptions " +   
            (numStreams * totalNumTopics * numPartitions) + " found:" +
            (subscribedl1.size() + subscribedl3.size()));
        return false;
      }
      totalSubscription.clear();
      totalSubscription.addAll(subscribedTopicsl1);
      totalSubscription.addAll(subscribedTopicsl2);
      totalSubscription.addAll(subscribedTopicsl3);
      if (totalSubscription.size() != (numStreams * totalNumTopics)) {
        System.out.println("Expected total subscriptions " +   
                           (numStreams * totalNumTopics) + " found:" +
                           (totalSubscription.size()));
        return false;
      }

      maxPartitions = ((numPartitions / 2) + 1) * (numStreams * numTopics);
      minPartitions = ((numPartitions / 2)) * (numStreams * numTopics);

      if (subscribedl1.size() < minPartitions || subscribedl1.size() > maxPartitions ||
          subscribedl3.size() < minPartitions || subscribedl3.size() > maxPartitions) {

        System.out.println("l1 subscription size " + subscribedl1.size());
        System.out.println("l2 subscription size " + subscribedl3.size());
        return false;
      }

      /* Now delete all the topic - wait for unsububscribe and then subscribe */
      System.out.println("Delete all the topics and make sure all the partitions" +
                          " get unsubscribed");

      for (String topic: topicList) {
        try {
          String[] tokens = topic.split(":");
          madmin.deleteTopic(tokens[0], tokens[1]);
        } catch (Exception e) {
          System.out.println("Topic delete failed");
        }
      }

      cb.WaitForAllUnsubscribed();
      System.out.println("Recreating all topics");

      /* Now delete all the topic - wait for unsububscribe and then subscribe */
      for (String topic: topicList) {
        try {
          String[] tokens = topic.split(":");
          madmin.createTopic(tokens[0], tokens[1], numPartitions);
        } catch (Exception e) {
          System.out.println("Topic recreate failed");
        }
      }


      System.out.println("Waiting for all subscriptions to come back");

      cb.WaitForAllSubscribed();
       
      System.out.println("Unsubscribing all");
      l1.unsubscribe();
      l3.unsubscribe();
      cb.WaitForAllUnsubscribed();
    } finally {
      if (l1 != null) {
        l1.close();
      }
    }

    return true;
  }

  public static boolean runTestWithPollOptions(String streamName, int numStreams,
      int numTopics, int numSlowTopics, int numPartitions, int numExpectedMsgs,
      int maxPartitionFetchSize) throws IOException {

    Listener lp = new Listener(streamName, numStreams, numTopics, numSlowTopics, numPartitions, numExpectedMsgs, true /*verifyKeys*/,
                               true /*keysInOrder*/, false /*allowDuplicateKeys*/, false /*isTracingEnabled*/, false /*printStats*/,
                               TED_ACTION.kNone, false /* topic subscription */, null /*groupid*/, maxPartitionFetchSize);
    lp.runWithException();
    return lp.status;
  }

  public static boolean runTest(String streamName, int numStreams, int numTopics, int numSlowTopics, int numPartitions, int numExpectedMsgs) throws IOException {
    Listener lp = new Listener(streamName, numStreams, numTopics, numSlowTopics, numPartitions, numExpectedMsgs, true /*verifyKeys*/,
                               true /*keysInOrder*/, false /*allowDuplicateKeys*/, false /*isTracingEnabled*/, false /*printStats*/,
                               TED_ACTION.kNone, false /* topic subscription */,
                               null /* groupid */, 64 * 1024 /* maxPartitionFetchSize*/);
    lp.runWithException();
    return lp.status;
  }

  public static boolean runTest(String streamName, int numStreams, int
      numTopics, int numSlowTopics, int numPartitions, int
      numExpectedMsgs, String lgGrp) throws IOException {
    Listener lp = new Listener(streamName, numStreams, numTopics, numSlowTopics, numPartitions, numExpectedMsgs, true /*verifyKeys*/,
                               true /*keysInOrder*/, false /*allowDuplicateKeys*/, false /*isTracingEnabled*/, false /*printStats*/,
                               TED_ACTION.kNone, false /* topic subscription */,
                               lgGrp /* groupid */, 64 * 1024 /* maxPartitionFetchSize*/);
    lp.runWithException();
    return lp.status;
  }

  public static boolean runTest(String streamName, int numStreams, int numTopics, int numSlowTopics, int numPartitions, int numExpectedMsgs, TED_ACTION action) throws IOException {
    Listener lp = new Listener(streamName, numStreams, numTopics, numSlowTopics, numPartitions, numExpectedMsgs, true /*verifyKeys*/,
                               true /*keysInOrder*/, false /*allowDuplicateKeys*/, false /*isTracingEnabled*/, false /*printStats*/,
                               action, false /* topic subscription */, null /* groupid */, 64 * 1024 /* maxPartitionFetchSize*/);
    lp.runWithException();
    return lp.status;
  }
  
  public static boolean runTest(String streamName, int numStreams, int
                                numTopics, int numSlowTopics, int numPartitions, int numExpectedMsgs,
                                boolean topicSubscription, boolean keysInOrder,
                                String groupId) throws IOException {
    Listener lp = new Listener(streamName, numStreams, numTopics, numSlowTopics, numPartitions, numExpectedMsgs, true /*verifyKeys*/,
                               keysInOrder, false /*allowDuplicateKeys*/, false /*isTracingEnabled*/, false /*printStats*/,
                               TED_ACTION.kNone, topicSubscription, groupId, 64 * 1024 /* maxPartitionFetchSize*/);
    lp.runWithException();
    return lp.status;
  }

  public Listener(String streamName, int numStreams, int numTopics, int numSlowTopics, int numPartitions, int numExpectedMsgs,
                  boolean verifyKeys, boolean keysInOrder,
                  boolean allowDuplicateKeys, boolean isTracingEnabled, boolean
                  printStats, TED_ACTION action, boolean topicSubscription,
                  String groupId, int maxPartitionFetchSize) {
    this.streamName = streamName;
    this.numStreams = numStreams;
    this.numTopics = numTopics;
    this.numSlowTopics = numSlowTopics;
    this.numPartitions = numPartitions;
    this.numExpectedMsgs = numExpectedMsgs;
    this.verifyKeys = verifyKeys;
    this.keysInOrder = keysInOrder;
    this.allowDuplicateKeys = allowDuplicateKeys;
    this.isTracingEnabled = isTracingEnabled;
    this.printStats = printStats;
    this.action = action;
    this.topicSubscription = topicSubscription;
    this.groupId = groupId;
    this.maxPartitionFetchSize = maxPartitionFetchSize;

    conf = new Configuration();
    try {
      madmin = Streams.newAdmin(conf);
    } catch (Exception e) {
      e.printStackTrace();
    }
  }

  public void run() {
    try {
      runWithException();
    } catch (IOException e) {
      System.out.println(e);
    }
  }

  public void runWithException() throws IOException {
    KafkaConsumer listener = null;
    List<TopicPartition> partitions = new ArrayList<TopicPartition>();
    List<String> topicList = new ArrayList<String>();
    Thread tt = null;
    try {
      int pollsWithMissingMsgs = 0;
      status = false;
      if (streamName == null || streamName.length() == 0) {
        System.err.println("stream name cannot be empty.");
        Listener.usage();
      }
      
      if (keysInOrder) {
        partitionSeqMap = new Hashtable<PartitionInfo, Integer>();
      } else {
        partitionBArrayMap = new Hashtable<PartitionInfo, boolean[]>();
      }
      
      int totalNumTopics = numTopics + numSlowTopics;

      for (int sIdx = 0; sIdx < numStreams; sIdx++) {
        for (int i = 0; i < totalNumTopics; i++) {
          String streamTopicName = streamName + sIdx + ":" + topicIntial + i;
          topicList.add(streamTopicName);
          for (int j = 0; j < numPartitions; j++) {
            partitions.add(new TopicPartition(streamTopicName, j));
            if (isTracingEnabled)
              System.out.println("Creating a subscription for " + streamTopicName + " feed:" + j);
          }
        }
      }

      Properties props = new Properties();
      MarlinConfigDefaults cdef = MarlinConfigDefaults.getDefaultInstance();
      props.put("key.deserializer",
          "org.apache.kafka.common.serialization.ByteArrayDeserializer");
      props.put("value.deserializer",
          "org.apache.kafka.common.serialization.ByteArrayDeserializer");
      //props.put("auto.offset.reset", "earliest");
      props.put("max.partition.fetch.bytes", maxPartitionFetchSize);
      props.put(cdef.getMetadataMaxAge(), 100);
      if (groupId != null) {
        props.put("group.id", groupId);
      }

      ByteArrayDeserializer keyD = new ByteArrayDeserializer();
      ByteArrayDeserializer valueD = new ByteArrayDeserializer();
      listener = new KafkaConsumer<byte[], byte[]>(props, keyD, valueD);
      RebCb cb = new RebCb(partitions);
      cb.addConsumer(listener);

      if (topicSubscription) {
        if (groupId != null) {
          listener.subscribe(topicList, cb);
          cb.WaitForAllSubscribed();
        } else {
          listener.subscribe(topicList);
        }

        Set<String> subscribedTopics = listener.subscription();
        System.out.println("Using subscribe() api, got " + subscribedTopics.size() + " from subscription()");
        if (subscribedTopics.size() != numStreams * totalNumTopics) {
          throw new IOException("Subscription failed. Expected:" + 
                                (numStreams * totalNumTopics) + " found:" +
                                subscribedTopics.size());
        }
      } else {
        listener.assign(partitions);
        listener.assign(partitions);

        Set<String> subscribedTopics = listener.subscription();
        System.out.println("Using assign() api, got " + subscribedTopics.size() + " from subscription()");
        if (!subscribedTopics.isEmpty()) {
          throw new IOException("Subscription failed.  Should not have topic subscription " +
                                "when using assign() api");
        }
      }

      Set<TopicPartition> subscribed = listener.assignment();

      if (subscribed.size() != numStreams * totalNumTopics * numPartitions) {
        throw new IOException("Subscription failed. Expected:" + 
            (numStreams * totalNumTopics * numPartitions) + " found:" +
            subscribed.size());
      }

      System.out.println("Subscription successful:" + subscribed.size());
      for (TopicPartition p : subscribed) {
        long position = listener.position(p);
        // if (numSlowTopics == 0 && (position < numExpectedMsgs)) {
        //   throw new IOException("Position not as expected. numMsgs:"
        //                         + numExpectedMsgs + " position:" + position);
        // }
        if (isTracingEnabled) {
          System.out.println("Subscribed to " + p.topic() + " partition:" +
                             p.partition() + " position:" + position + "(" +
                             numExpectedMsgs + ")");
        }
        listener.seekToBeginning(p);
      }

      PerfStats stats = new PerfStats();
      if (action != TED_ACTION.kNone) {
        TedThread ttp = new TedThread(action, this);
        tt = new Thread(ttp);
        tt.start();
      }
      while (true) {
        if (isTracingEnabled)
          System.out.println("pollsWithMissingMsgs "+ pollsWithMissingMsgs);

        ConsumerRecords<byte[], byte[]> recs = listener.poll(1000);
        if (recs.count() == 0) {
          ++pollsWithMissingMsgs;
        } else {
          pollsWithMissingMsgs = 0;
        }
        VerifyAndAddStats(recs, stats);
        if (pollsWithMissingMsgs >= 2)
          break;
      }

      if (verifyKeys) {
        VerifyPollingEnd();
      }

      if (printStats)
        stats.printReport();
      
      status = true;

      System.out.println("committing offsets");
      listener.commitSync();
      System.out.println("returning from commit offset");

      if (groupId != null && totalNumTopics < 10) {
        System.out.println("verifying list cursors");
        /* verify cursors made it to the disk */
        for (int sIdx = 0; sIdx < numStreams; sIdx++) {
          for (int i = 0; i < totalNumTopics; i++) {
            MarlinAdminImpl admin = (MarlinAdminImpl) madmin;
            List<CursorInfo> ci = admin.listCursors(streamName + sIdx,
                                                    groupId, topicIntial + i,
                                                    -1);
            if (ci.size() != numPartitions) {
              throw new IOException("Expected "+ numPartitions+" cursorInfo structure returned " + ci.size());
            }

            for (CursorInfo c : ci) {
              TopicPartition p = partitions.get(sIdx*totalNumTopics*numPartitions +
                                                (i*numPartitions) + c.feedId());
              long committedOffset = listener.committed(p).offset();
              if (committedOffset != c.cursor()) {
                throw new IOException("read offset from disk:" +
                    c.cursor() + " queryied:" + committedOffset);
              }
            }
          }
        }
        System.out.println("verifying list cursors done");
      }

      /* Test Seek and position */
      // TODO integrate all these tests with poll
      long[] committed = new long[subscribed.size()];
      int i = 0;
      for (TopicPartition p : subscribed) {
        committed[i] = listener.committed(p).offset();
        long position = listener.position(p);
        if (position != committed[i]) {
          System.out.println("Postion is :" + position + " committed:" + committed[i]);
          throw new IOException("position is not what was expected");
        }

        listener.seekToBeginning(p);
        if (listener.position(p) != 0) {
          throw new IOException("seekToBeginning failed");
        }
        i++;
      }

      TopicPartition[] partsToSeek = new TopicPartition[partitions.size()];
      partitions.toArray(partsToSeek);
      listener.seekToEnd(partsToSeek);
      i = 0;

      for (TopicPartition p : subscribed) {
        long position = listener.position(p);
        if (position < committed[i++]) {
          System.out.println("Postion after seekToEnd is :" + position + " committed:" + committed[i]);
          throw new IOException("position is not what was expected, after seekToEnd");
        }
      }

      /* Delete topics and wait for all subscriptions to be gone then recreate
       * the topics and wait for all subscriptions to come back.
       */
      for (String topic: topicList) {
        String[] tokens = topic.split(":");
        madmin.deleteTopic(tokens[0], tokens[1]);
      }

      int sleepMaxTries = 5;
      while (sleepMaxTries > 0) {
        try {
          Thread.sleep(150);
        } catch (Exception e) {
        }
        subscribed = listener.assignment();
        if (subscribed.size() == 0) break;
        sleepMaxTries--;
      }

      subscribed = listener.assignment();
      if (subscribed.size() != 0) {
        throw new IOException("Found subscription after topic delete");
      }

      for (String topic : topicList) {
        String[] tokens = topic.split(":");
        madmin.createTopic(tokens[0], tokens[1], numPartitions);
      }

      sleepMaxTries = 5;
      while (sleepMaxTries > 0) {
        try {
          Thread.sleep(150);
        } catch (Exception e) {
        }
        subscribed = listener.assignment();
        if (subscribed.size() == numStreams * totalNumTopics * numPartitions) break;
        sleepMaxTries--;
      }

      subscribed = listener.assignment();
      if (subscribed.size() != numStreams * totalNumTopics * numPartitions) {
        throw new IOException("Found subscription after topic delete");
      }

      /* Now get cursors on these new partitions and they should all be zero */
      for (TopicPartition p : subscribed) {
        try {
          OffsetAndMetadata offset = listener.committed(p);
          long c = 0;
          if (offset != null)
            c = listener.committed(p).offset();
          if (c != 0) {
            throw new IOException ("Cursor found for a recreated topic");
          }
        } catch (KafkaException ke) {
        }
      }
     
      /* Now delete and recreate topic with different number of feeds and make
       * sure only those subsriptions come back. */

      for (String topic: topicList) {
        String[] tokens = topic.split(":");
        madmin.deleteTopic(tokens[0], tokens[1]);
      }

      if (topicSubscription && groupId != null) {
        cb.WaitForAllUnsubscribed();
      }

      for (String topic : topicList) {
        String[] tokens = topic.split(":");
        madmin.createTopic(tokens[0], tokens[1], numPartitions/2);
      }

      try {
        sleepMaxTries = 5;
        while (sleepMaxTries > 0) {
          try {
            Thread.sleep(150);
          } catch (Exception e) {
          }
          subscribed = listener.assignment();
          if (subscribed.size() == numStreams * totalNumTopics * (numPartitions/2)) break;
          sleepMaxTries--;
        }

      } catch (Exception e) {
      }

      subscribed = listener.assignment();
      if (subscribed.size() != numStreams * totalNumTopics * (numPartitions/2)) {
        throw new IOException("Found " + subscribed.size() + " subscription after topic delete"  
                              + " Expected:" + numStreams * totalNumTopics *
                              (numPartitions/2));
      }

      System.out.println("Unsubscribing");
      if (topicSubscription) {
        listener.unsubscribe();
        if (groupId != null) {
          cb.WaitForAllUnsubscribed();
        }
      } else {
        listener.unsubscribe();
        Set<TopicPartition> subscribedEnd = listener.assignment();
        if (subscribedEnd.size() != 0) {
          throw new IOException("Unsubscribe failed");
        }
      }
      System.out.println("The test completed successfully");
    } finally {
      if (listener != null) {
        listener.close();
      }
      try {
      if (tt != null)
        tt.join();
      } catch (Exception e) {
        e.printStackTrace();
      }
    }
  }

  public void DeleteStream() {
    try {
      madmin.deleteStream(streamName + 0);
    } catch (Exception e) {
      e.printStackTrace();
    }
  }

  public void ChangePerm() {
    try {
      StreamDescriptor sdesc = Streams.newStreamDescriptor();
      System.out.println("Change listen perms for " + streamName + 0);
      sdesc.setConsumePerms("u:0 | g:0");
      madmin.editStream(streamName + 0, sdesc);
      sdesc = Streams.newStreamDescriptor();
      System.out.println("Change admin perms for " + streamName + 0);
      sdesc.setAdminPerms("u:0 | g:0");
      madmin.editStream(streamName + 0, sdesc);
    } catch (Exception e) {
      e.printStackTrace();
    }
  }

  private void VerifyAndAddStats(ConsumerRecords<byte[], byte[]> recs, PerfStats stats) throws IOException {
    Iterator<ConsumerRecord<byte[], byte[]>> iter = recs.iterator();
    long numBytes = 0; 
    long recSize = 0;
    Hashtable<PartitionInfo, Long> partitionMsgBytesMap = null;
    if (verifyKeys)
      partitionMsgBytesMap = new Hashtable<PartitionInfo, Long>();

    while (iter.hasNext()) {
      ConsumerRecord<byte[], byte[]> rec = iter.next();
      if (isTracingEnabled)
        System.out.println(rec);
      byte[] key = null;
      byte[] value = null;
      try {
        key = rec.key();
        value = rec.value();
      } catch (Exception e) {
        throw new IOException("ConsumerRecord Exception");
      }
      recSize = key.length + value.length;
      numBytes += recSize;
      if (verifyKeys) {
        String keyStr = new String(key, "UTF-8");
        String[] tokens = keyStr.split(":");
        if (tokens.length != 4) {
          throw new IOException("Key " + keyStr + " not of correct format");
        }

        int partition = Integer.parseInt(tokens[2]);
        int seq = Integer.parseInt(tokens[3]);
        if (isTracingEnabled)
          System.out.println("Key " + keyStr + " ntokens " + tokens.length);
        if ((tokens[0].length() != streamName.length() + 1) ||
            (!tokens[0].startsWith(streamName)) ||
            (Integer.parseInt(tokens[0].substring(streamName.length())) >= numStreams)) {
          throw new IOException("StreamName in key " + tokens[0] + " didnot match intial "
                                + streamName + " and numstreams " + numStreams);
        }
        String recTopic = rec.topic();
        String[] recTopicTokens = recTopic.split(":");
        if ((recTopicTokens.length != 2) ||
            !tokens[1].equals(recTopicTokens[1])) {
          throw new IOException("Topic in key " + tokens[1] + " mismatched. Expected " + recTopicTokens[1]);
        }
        if (partition != rec.partition()) {
          throw new IOException("Partition in key " + partition + " mismatched. Expected " + rec.partition());
        }
        if (keysInOrder) {
          VerifyPollingOrderedKey(tokens[0], tokens[1], partition, seq);
        } else {
          VerifyPollingUnorderedKey(tokens[0], tokens[1], partition, seq);
        }
        AddPerPartitonRecs(partitionMsgBytesMap, tokens[0], tokens[1], partition, recSize);
      }
      //if (isTracingEnabled)
        //System.out.println("Value " + new String(value, "UTF-8"));
    }
    if (verifyKeys) {
      for (Long size : partitionMsgBytesMap.values()) {
        if (size > maxPartitionFetchSize) {
          System.out.println("Total msgbytes fetched " + size + " is greater than " + maxPartitionFetchSize);
          throw new IOException("Total msgbytes fetched " + size + " is greater than " + maxPartitionFetchSize);
        }
      }
    }
    stats.report(numBytes, recs.count());
  }

  private void VerifyPollingOrderedKey(String stName, String tpName, int partition, int seq)
      throws IOException
  {
    PartitionInfo pinfo = new PartitionInfo(stName, tpName, partition);
    Integer mappedSeq = partitionSeqMap.get(pinfo);
    int expSeq = 0;
    if (mappedSeq != null) {
      expSeq = mappedSeq + 1;
    }
    if (seq != expSeq) {
      throw new IOException("Current Seq " + seq + " for Stream " + stName +
                            " Topic " + tpName + " partition " + partition +
                            " mismatched. Expected " + expSeq);
    }
    partitionSeqMap.put(pinfo, seq);
  }

  private void AddPerPartitonRecs(Hashtable<PartitionInfo, Long> pMap, String stName, String tpName,
      int partition, long recSize)
  {
    PartitionInfo pinfo = new PartitionInfo(stName, tpName, partition);
    Long oldSize = pMap.get(pinfo);
    long updatedSize = recSize;
    if (oldSize != null) {
      updatedSize += oldSize;
    }
    pMap.put(pinfo, updatedSize);
  }

  private void VerifyPollingUnorderedKey(String stName, String tpName, int partition, int seq)
      throws IOException
  {
    PartitionInfo pinfo = new PartitionInfo(stName, tpName, partition);
    boolean[] bitArray = partitionBArrayMap.get(pinfo);
    if (bitArray == null) {
      //TODO(NARENDRA): use BitSet instead.
      bitArray = new boolean[numExpectedMsgs];
      partitionBArrayMap.put(pinfo, bitArray);
    }
    if (bitArray[seq] && !allowDuplicateKeys) {
      throw new IOException("Duplicate key for Stream " + stName + " Topic " + tpName + " partition " + partition + ". seq is " + seq);
    }
    bitArray[seq] = true;
  }

  private int NumExpectedMsgs(String topicName) {
    if (Integer.parseInt(topicName.substring(topicIntial.length())) < numTopics)
      return numExpectedMsgs;
    return numExpectedMsgs / 1000;
  }

  private void VerifyPollingEnd() throws IOException
  {
    int totalNumTopics = numTopics + numSlowTopics;
    int totalPartitions = numStreams * totalNumTopics * numPartitions;
    if (keysInOrder) {
      if (partitionSeqMap.size() != totalPartitions) {
        throw new IOException("Total entries in hashmap " + partitionSeqMap.size() + ", expected " + totalPartitions);
      }
      Set<Map.Entry<PartitionInfo, Integer>> set = partitionSeqMap.entrySet();
      for (Map.Entry<PartitionInfo, Integer> entry : set) {
        int lastExpectedSeq = NumExpectedMsgs(entry.getKey().topicName()) - 1;
        if (entry.getValue() != lastExpectedSeq) {
          throw new IOException(entry.getKey() + ", Last seq received " + entry.getValue() + ", expected " + lastExpectedSeq);
        }
      }
    } else {
      if (partitionBArrayMap.size() != totalPartitions) {
        throw new IOException("Total entries in hashmap " + partitionBArrayMap.size() + ", expected " + totalPartitions);
      }
      Set<Map.Entry<PartitionInfo, boolean[]>> set = partitionBArrayMap.entrySet();
      for (Map.Entry<PartitionInfo, boolean[]> entry : set) {
        boolean[] barray = entry.getValue();
        for (int i = 0; i < numExpectedMsgs; i++) {
          if (!barray[i]) {
            throw new IOException("Message with seq " + i + " missing");
          }
        }
      }
    }
  }

  private static final class RebCb implements ConsumerRebalanceListener { 
    
    List<TopicPartition> subscribed;
    boolean[] actuallySubscribed;
    boolean allSubscribed = false;
    boolean allUnsubscribed = true;
    List<Consumer<?, ?>> consumers = new ArrayList<Consumer<?, ?>>();                      

    RebCb(List<TopicPartition> partitions) {
      subscribed = partitions;
      actuallySubscribed = new boolean[partitions.size()];
      for (int i = 0; i < actuallySubscribed.length; i++) {
        actuallySubscribed[i] = false;
      }
    }

    public void addConsumer(Consumer<?,?> consumer) {
      consumers.add(consumer);
    }

    public void Reallocate(List<TopicPartition> partitions) {
      boolean[] newArr = new boolean[partitions.size()];
      for (int i = 0; i < partitions.size(); i++) {
        for (int j = 0; j < subscribed.size(); j++) {
          if (partitions.get(i).equals(subscribed.get(j))) {
            newArr[i] = actuallySubscribed[j];
          }
        }
      }
    
      subscribed = partitions;
      actuallySubscribed = newArr;
      allSubscribed = false;
    }

    public void WaitForRevoke () {
      while (AllSubscribed()) {
        try {
          this.wait();
        } catch (Exception e) {
        }
      }
    }

    public void WaitForAllSubscribed() {
      // for (int i = 0; i < actuallySubscribed.length; i++) {
      //   System.out.println("subscribed " + i + "/" + actuallySubscribed.length + " " + actuallySubscribed[i]);
      // }
      while (AllSubscribed() != true) {
        try {
          this.wait();
        } catch (Exception e) {
        }
      }
    }

    public void WaitForAllUnsubscribed() {
      // for (int i = 0; i < actuallySubscribed.length; i++) {
      //   System.out.println("unsubscribed " + i + "/" + actuallySubscribed.length + " " + actuallySubscribed[i]);
      // }
      while (AllUnsubscribed() != true) {
        try {
          this.wait();
        } catch (Exception e) {
        }
      }
    }

    public synchronized boolean AllSubscribed() { return allSubscribed; }
    public synchronized boolean AllUnsubscribed() { return allUnsubscribed; }

    public synchronized void setAllSubscribed() {
      for (int i = 0; i < actuallySubscribed.length; i++) {
        // System.out.println("subscribed " + i + "/" + actuallySubscribed.length + " " + actuallySubscribed[i]);
        if (actuallySubscribed[i] == false) {
          return;
        }
      }

      System.out.println("setting subscribed to true");
      allSubscribed = true;
    }

    public synchronized void setAllUnsubscribed() {
      for (int i = 0; i < actuallySubscribed.length; i++) {
        if (actuallySubscribed[i] != false) {
          return;
        }
      }

      System.out.println("setting unsubscribed to true");
      allUnsubscribed = true;
    }


    public void
      onPartitionsAssigned(Collection<TopicPartition> partitions) 
      {
        Assigned(partitions);
      }

    public synchronized void Assigned(Collection<TopicPartition> partitions) {
        for (TopicPartition partition : partitions) {
        // System.out.println("assigned " + partition.topic() + ":" +
        //                    partition.partition());

          for (int i = 0; i <  subscribed.size(); i++) {
            if (partition.equals(subscribed.get(i))) {
              assertTrue(actuallySubscribed[i] == false);
              actuallySubscribed[i] = true;
              allUnsubscribed = false;
              break;
            }
          }
        }

        setAllSubscribed();
        this.notifyAll();
      }

    public void
      onPartitionsRevoked(Collection<TopicPartition> partitions) {
        Revoked(partitions);

        if (allUnsubscribed) {
          for (Consumer<?, ?> consumer : consumers) {
            ConsumerRecords<?, ?> recs = consumer.poll(10);
            if (recs.count() != 0) {
              System.out.println("Got messages after all revoked");
            }
          }
        }
      }

    public synchronized void Revoked(Collection<TopicPartition> partitions) {
      for (TopicPartition partition : partitions) {
         // System.out.println("revoked " + partition.topic() + ":" +
         //     partition.partition());
        for (int i = 0; i <  subscribed.size(); i++) {
          if (partition.equals(subscribed.get(i))) {
            assertTrue(actuallySubscribed[i] == true);
            actuallySubscribed[i] = false;
            allSubscribed = false;
            break;
          }
        }
      }

      setAllUnsubscribed();
      this.notifyAll();
    }
  }     

  private final class PerfStats {
    private long startTime;
    private long endTime;
    private long totalBytes;
    private long totalMsgs;

    public PerfStats() {
      this.endTime = -1;
      this.totalBytes = 0;
      this.totalMsgs = 0;
      this.startTime = System.currentTimeMillis();
    }

    public synchronized void report(long bytes, long numMsgs) {
      this.totalBytes += bytes;
      this.totalMsgs += numMsgs;
    }

    public synchronized void printReport() {
      this.endTime = System.currentTimeMillis();
      System.out.println("Total time (ms): " + (this.endTime - this.startTime));
      System.out.println("Total bytes received: " + this.totalBytes);
      System.out.println("Total messages received: " + this.totalMsgs);
      long bytesInKb = totalBytes / 1024;
      System.out.println("Average nKBs/sec: " + bytesInKb * 1.0/(this.endTime-this.startTime)*1000.0);
      System.out.println("Average nMsgs/sec: " + this.totalMsgs * 1.0/(this.endTime-this.startTime)*1000.0);
    }
  }

  private class PartitionInfo {
    private final String streamName;
    private final String topicName;
    private final int partitionId;

    public PartitionInfo(String streamName, String topicName, int partitionId) {
      this.streamName = streamName;
      this.topicName = topicName;
      this.partitionId = partitionId;
    }

    public boolean equals(Object anObject) {
      if (this == anObject) {
        return true;
      }
      if (anObject instanceof PartitionInfo) {
        PartitionInfo pinfo = (PartitionInfo)anObject;
        if (streamName.equals(pinfo.streamName()) && topicName.equals(pinfo.topicName()) && (partitionId == pinfo.partitionId()))
          return true;
      }
      return false;
    }

    public int hashCode() {
      return topicName.hashCode() + partitionId;
    }

    @Override
    public String toString() {
      return "Stream: " + streamName + " Topic: " + topicName + " Partition: " + partitionId;
    }

    public String streamName() { return streamName; }
    public String topicName() { return topicName; }
    public int partitionId() { return partitionId; }
  }

  public enum TED_ACTION {
    kNone(0),
    kDeleteStream(1),
    kChangePerm(2);
    private TED_ACTION(int val) {
      this.val = val;
    }

    public int getValue() {
      return val;
    }

    static TED_ACTION parseTedAction(int val) {
      switch (val) {
        case 0: return kNone;
        case 1: return kDeleteStream;
        case 2: return kChangePerm;
        default: {
          System.out.println("Invalid ted action " + val);
          return kNone;
        }
      }
    }

    private final int val;
  }

  private class TedThread implements Runnable {
    private TED_ACTION action;
    private Listener listener;
    public TedThread(TED_ACTION action, Listener listener) {
      this.action = action;
      this.listener = listener;
    }

    public void run() {
      try {
        Random randomGenerator = new Random();
        int numSecs = 5 +  randomGenerator.nextInt(10);
        try {
        Thread.sleep(numSecs * 100); // sleep for 
        } catch (Exception e) {
        }
        if (this.action == TED_ACTION.kDeleteStream) {
          System.out.println("Ted Action DeleteStream");
          listener.DeleteStream();
        } else if (this.action == TED_ACTION.kChangePerm) {
          System.out.println("Ted Action ChangePerm");
          listener.ChangePerm();
        }
      } catch (Exception e) {
        e.printStackTrace();
      }
    }
  }
}

