/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.streams.demo;

import java.lang.System;
import java.io.*;
import java.util.List;
import java.util.Random;
import java.nio.ByteBuffer;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Properties;
import org.apache.kafka.clients.producer.Callback;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.common.PartitionInfo;

public class DemoProducer {

  public static String[] streamNames;
  public static boolean multipleFlushers = false;
  public static int load = 500;
  public static KafkaProducer producer;
  public static String producerConfig;

  public static void usage() {
    System.err.println("DemoProducer -path <topic-full-name> [-load <Kmsg/sec>] [-producerConfig configFile]");
    System.exit(1);
  }

  public static void main(String[] args) throws IOException {
    for (int i = 0; i < args.length; ++i) {
      if (args[i].equals("-path")) {
        i++;
        if (i >= args.length) usage();
        streamNames = args[i].split(",");
        System.out.println("Starting producer on " + args[i]);
      } else if (args[i].equals("-load")) {
        i++;
        if (i >= args.length) usage();
        load = Integer.parseInt(args[i]);
        load *= 1000; // in k messages per second
        System.out.println("Setting load to " + load + "msgs/sec");
      } else if (args[i].equals("-producerConfig")) {
        i++;
        if (i >= args.length) usage();
        producerConfig = args[i];
      } else {
        usage();
      }
    }

    if (streamNames == null || streamNames.length == 0) {
      usage();
    }

    Properties props = new Properties();
    if (producerConfig != null) {
      props.load(new FileInputStream(producerConfig));
    }

    if (props.getProperty("bootstrap.servers") == null) {
      props.put("bootstrap.servers", "localhost:9092");
    }

    if (props.getProperty("key.serializer") == null) {
      props.put("key.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    }

    if (props.getProperty("value.serializer") == null) {
      props.put("value.serializer", "org.apache.kafka.common.serialization.ByteArraySerializer");
    }

    if (props.getProperty("streams.parallel.flushers.per.partition") != null) {
        props.put("streams.parallel.flushers.per.partition", multipleFlushers);
    }

    producer = new KafkaProducer<byte[], byte[]>(props);
    produceTransactions();
  }

  public static void produceTransactions()
  {
    String goodLog = "This is a valid transaction";
    byte[] goodTransaction = goodLog.getBytes();

    Random streamNameRand =  new Random();
    Random rand = new Random();
    int randomNum = rand.nextInt(3*load);
    randomNum++;

    Callback cb = new ProducerCallback();
    long i = 0;
    System.out.println("There are " + streamNames.length + " streams");
    try {
      while (true) {
        long startTime = System.currentTimeMillis();
        for (int j = 0; j < load; j++) {
          int streamId = streamNameRand.nextInt(streamNames.length);
          String keyStr = String.format("43%02d 3344 2341 %04d", streamId, (i++ % 1000));
          byte[] key = keyStr.getBytes();
          byte[] value = goodTransaction; 
          if ((i % (long)randomNum) == 0) {
            long currTime = System.currentTimeMillis();
            String badLog = currTime + ": Suspicious transaction";
            value = badLog.getBytes();
            randomNum = rand.nextInt(load);
            randomNum++;
            System.out.println(currTime + ":" + keyStr + ":Introduced a bad" +
                " transaction in " + streamNames[streamId]);
          }
          ProducerRecord<byte[], byte[]> record = new ProducerRecord<byte[],
            byte[]>(streamNames[streamId], null, key, value);
          producer.send(record, cb);
        }

        /* sleep for the rest of the second before sending the next load */
        long endTime = startTime + 1000;
        long currTime = System.currentTimeMillis();
        if (endTime > currTime) {
          Thread.sleep(endTime - currTime);
        }
      }
    } catch (Exception e) {
      e.printStackTrace();
    }
  }

  private static final class ProducerCallback implements Callback {

    public ProducerCallback() {
    }

    public void onCompletion(RecordMetadata metadata,
                             Exception exception) {
      if (exception != null) {
        exception.printStackTrace();
        System.exit(1);
      }
    }
  }
}
