/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.streams.demo;

import java.lang.System;
import java.util.*;
import java.io.*;
import java.nio.ByteBuffer;
import java.text.SimpleDateFormat;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRebalanceListener;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.common.serialization.ByteArrayDeserializer;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.PartitionInfo;

public class DemoConsumer {

  public static List<String> streamNames = new ArrayList<String>();
  public static KafkaConsumer consumer;
  public static String consumerConfig = null;
  public static String consumerType = null;
  public static int waitBetweenScr = 60 * 1000;

  public static void usage() {
    System.err.println("DemoConsumer -path <topic-full-name> -consumerConfig <configfile>");
    System.exit(1);
  }

  public static void main(String[] args) throws IOException {
    for (int i = 0; i < args.length; ++i) {
      if (args[i].equals("-path")) {
        i++;
        if (i >= args.length) usage();
        String[] tmp = args[i].split(",");
        for (String oneName : tmp) {
          streamNames.add(oneName);
        }
        System.out.println("Starting consumer on " + args[i]);
      } else if (args[i].equals("-type")) {
        i++;
        if (i >= args.length) usage();
        consumerType = args[i];
      } else if (args[i].equals("-consumerConfig")) {
        i++;
        if (i >= args.length) usage();
        consumerConfig = args[i];
      } else {
        usage();
      }
    }

    if (streamNames.size() == 0) {
      usage();
    }

    if (consumerType == null) {
      consumerType = new String("analyzer");
    }

    System.out.println("starting a " + consumerType + " consumer");

    Properties props = new Properties();
    if (consumerConfig != null) {
      props.load(new FileInputStream(consumerConfig));
    }

    if (props.getProperty("bootstrap.servers") == null) {
      props.put("bootstrap.servers", "localhost:9092");
    }

    if (props.getProperty("key.deserializer") == null) {
      props.put("key.deserializer",
          "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    }

    if (props.getProperty("value.deserializer") == null) {
      props.put("value.deserializer",
          "org.apache.kafka.common.serialization.ByteArrayDeserializer");
    }

    consumer = new KafkaConsumer<byte[], byte[]>(props);
    consumeTransactions();
  }

  public static void consumeTransactions() 
  {
    SimpleDateFormat sdf = new SimpleDateFormat("HH:mm:ss,SSS");
    int numStreams = streamNames.size();
    int currentStream = 0;
    try {
      consumer.subscribe(streamNames);
      currentStream = numStreams;

      long totalBad = 0, total = 0;
      long currTotal = 0, currBad = 0;
      long lastTime = System.currentTimeMillis();
      long lastSubscribedTime = lastTime;
      long lastReport = 0;
      long lapseTime = 0;
      long maxTime = 0;

      while (true) {
        ConsumerRecords<byte[], byte[]> crecs = consumer.poll(1000);
        Iterator<ConsumerRecord<byte[], byte[]>> iter = crecs.iterator();
        long currentTime = System.currentTimeMillis();
        while (iter.hasNext()) {
          ConsumerRecord<byte[], byte[]> crec = iter.next();
          byte[] key = crec.key();
          byte[] value = crec.value();

          String log = new String(value);
          if (log.contains("Suspicious")) {
            long currTime = System.currentTimeMillis();
            String parts[] = log.split(":");
            long transactionTime = Long.parseLong(parts[0]);
            String keyStr = new String(key);
            if (consumerType.equals("analyzer")) {
              lapseTime = (currTime - transactionTime);
              if (lapseTime > maxTime)
                maxTime = lapseTime;
              System.out.println(sdf.format(currTime) + " " + currTime + ":" + keyStr + ":" + "bad transaction @" 
                  + transactionTime + " found within:" + (currTime - transactionTime)
                  + "ms max lapse:" + maxTime +"ms");
            }
            totalBad++;
            currBad++;
          }
          total++;
          currTotal++;
        }

        if ((currentStream < numStreams) &&
            ((lastSubscribedTime + waitBetweenScr) < currentTime)) {
          System.out.println("Subscribing to stream " + streamNames.get(currentStream));
          consumer.subscribe(streamNames);
          ++currentStream;
          lastSubscribedTime = currentTime;
        }

        if (consumerType.equals("freq") && (total > (lastReport + 5000))) {
          double tP = (float) totalBad  * 100.0 / (float) total;
          double cP = (float) currBad * 100.0 / (float) currTotal;
          System.out.println("Total transactions processed: " + total);
          System.out.println("Total suspicious transactions flagged: " + totalBad);
          System.out.printf("Current bad transaction rate : %(,.3f\n", tP);
          currBad = currTotal = 0;
          long timelapse = System.currentTimeMillis() - lastTime;
          lastTime = System.currentTimeMillis();
          lastReport = total;

          if (cP > (tP + 0.01)) {
            System.out.println("Saw more than 0.01% rise in bad transactions in " +
                                "last " + timelapse + "ms"); 
          }
        }
      } 
    } catch (Exception e) {
      e.printStackTrace();
      System.out.println("Consumer failed");
    }
  }
}
