/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.streams;

import org.apache.hadoop.conf.Configuration;

import org.ojai.Document;
import org.ojai.DocumentStream;
import org.ojai.store.DocumentStore;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.regex.Pattern;
import java.util.StringTokenizer;

import com.mapr.streams.Streams;

public class StreamAnalyzer {

  public static String streamName = null;
  public static Pattern topicRegex = null;
  public static String[] topicNames = null;
  public static String[] projections = null;
  public static boolean countMessages = true;
  public static boolean printMessages = false;
  public static int     maxParallelScans = -1;
  public static long    maxCacheSize = -1;

  public static void usage() {
    System.err.println("StreamAnalyzer -path <stream-full-name>");
    System.err.println("     [ -topics <comma separated topic names> ]");
    System.err.println("     [ -regex  <regular expression representing topic names> ]");
    System.err.println("     [ -countMessages <true/false> (default: true) ]");
    System.err.println("     [ -printMessages <true/false> (default: false) ]");
    System.err.println("     [ -projectFields <comma separated field names> ");
    System.err.println("                      (Valid field names: "
                                               + Streams.KEY + ", "
                                               + Streams.VALUE + ", "
                                               + Streams.TOPIC + ", "
                                               + Streams.PARTITION + ", "
                                               + Streams.OFFSET + ", "
                                               + Streams.PRODUCER + ", "
                                               + Streams.ID + ".");
    System.err.println("                       Works only if printMessages is true.");
    System.err.println("                       Default: all fields.) ]");
    System.err.println("     [ -parallelScans <max number of parallel scans> (default: 16)]");
    System.err.println("     [ -maxCacheSize  <max size of Documents to cache> (default: 100MB)]");
    System.exit(1);
  }

  public static void main(String[] args) throws IOException {
    for (int i = 0; i < args.length; ++i) {
      if (args[i].equals("-path")) {
        i++;
        if (i >= args.length) usage();
        streamName = args[i];
      } else if (args[i].equals("-topics")) {
        i++;
        if (i >= args.length) usage();
        StringTokenizer st = new StringTokenizer(args[i],", ");
        List<String> tokens = new ArrayList<String>();
        while (st.hasMoreTokens()) {
          tokens.add(st.nextToken());
        }

        topicNames = tokens.toArray(new String[tokens.size()]);
      } else if (args[i].equals("-regex")) {
        i++;
        if (i >= args.length) usage();
        topicRegex = Pattern.compile(args[i]);
      } else if (args[i].equals("-countMessages")) {
        i++;
        if (i >= args.length) usage();
        countMessages = Boolean.parseBoolean(args[i]);
      } else if (args[i].equals("-printMessages")) {
        i++;
        if (i >= args.length) usage();
        printMessages = Boolean.parseBoolean(args[i]);
      } else if (args[i].equals("-projectFields")) {
        i++;
        if (i >= args.length) usage();
        StringTokenizer st = new StringTokenizer(args[i],", ");
        List<String> tokens = new ArrayList<String>();
        while (st.hasMoreTokens()) {
          tokens.add(st.nextToken());
        }

        projections = tokens.toArray(new String[tokens.size()]);
      } else if (args[i].equals("-parallelScans")) {
        i++;
        if (i >= args.length) usage();
        maxParallelScans = Integer.parseInt(args[i]);
        if (maxParallelScans <= 0) {
          System.err.println("-parallelScans cannot be <= 0");
          System.exit(1);
        }
      } else if (args[i].equals("-maxCacheSize")) {
        i++;
        if (i >= args.length) usage();
        maxCacheSize = Long.parseLong(args[i]);
        if (maxCacheSize <= 0) {
          System.err.println("-maxCacheSize cannot be <= 0");
          System.exit(1);
        }
      } else {
        usage();
      }
    }

    if (streamName == null) {
      System.err.println("Stream path is mandatory command-line option");
      usage();
    }

    if ((projections != null) && (printMessages == false)) {
      System.err.println("Projections are only used if '-printMessages' is set to true");
      usage();
    }

    if ((topicNames != null) && (topicRegex != null)) {
      System.err.println("Only '-topics' or '-regex' allowed");
      usage();
    }

    runTest();
  }

  public static void runTest() throws IOException {
    DocumentStore store = null;

    Configuration conf = new Configuration();
    if (maxParallelScans > 0 ||
        maxCacheSize > 0) {
      if (maxParallelScans > 0) {
        conf.setInt(Streams.MAX_SCANNER_THREADS, maxParallelScans);
      }

      if (maxCacheSize > 0) {
        conf.setLong(Streams.MAX_CACHE_MEMORY, maxCacheSize);
      }
    }

    if (topicNames != null) {
      store = Streams.getMessageStore(streamName, conf, topicNames);
    } else if (topicRegex != null) {
      store = Streams.getMessageStore(streamName, conf, topicRegex);
    } else {
      store = Streams.getMessageStore(streamName, conf);
    }

    DocumentStream rs = null;

    if (projections != null) {
      rs = store.find(projections);
    } else {
      rs = store.find();
    }

    Iterator<Document> iter = rs.iterator();
    int count = 0;
    while (iter.hasNext()) {
      Document d = iter.next();
      if (printMessages) {
        if (d.getString(Streams.ID) != null) {
          System.out.print(" ID: " + d.getString(Streams.ID));
        }

        if (d.getString(Streams.TOPIC) != null) {
          System.out.print(" topic: " + d.getString(Streams.TOPIC));
        }

        try {
          System.out.print(" partition: " + d.getInt(Streams.PARTITION));
        } catch (NoSuchElementException e) { }

        try {
          System.out.print(" offset: " + d.getLong(Streams.OFFSET));
        } catch (NoSuchElementException e) { }

        if (d.getString(Streams.PRODUCER) != null) {
          System.out.print(" producer: " + d.getString(Streams.PRODUCER));
        }

        if (d.getBinary(Streams.KEY) != null) {
          System.out.print(" key: <binary>");
        }

        if (d.getBinary(Streams.VALUE) != null) {
          System.out.print(" value: <binary>");
        }

        System.out.println();
      }

      count ++;
    }

    if (countMessages) {
      System.out.println("Total number of messages: " + count);
    }
  }
}
