/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.tests.objectinstrumentation;

import java.util.AbstractMap.SimpleEntry;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.log4j.Level;
import org.junit.After;
import org.junit.Before;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;
import com.google.monitoring.runtime.instrumentation.AllocationRecorder;
import com.google.monitoring.runtime.instrumentation.Sampler;
import com.mapr.tests.BaseTest;

public class ObjectInstrumentationTest extends BaseTest {
  private static Logger _logger = LoggerFactory.getLogger(ObjectInstrumentationTest.class);

  static {
    ROOT_LOGGER.setLevel(Level.OFF);
    MAPR_LOGGER.setLevel(Level.INFO);
    createLogger("com.mapr.tests.objectinstrumentation", LOG_FILE);
  }

  protected enum InstrumentationType {
    ALL(~0),
    COUNT_OF_OBJECT(1 << 0),
    SIZE_OF_OBJECTS(1 << 1),
    CLASS_HISTOGRAM(1 << 2);

    final private long bit;
    private InstrumentationType(long bit) {
      this.bit = bit;
    }
  }
  long instrumentations = 0;
  protected void setTypes(InstrumentationType...types) {
    instrumentations = 0;
    if (types != null) {
      for (InstrumentationType type : types) {
        instrumentations |= type.bit;
      }
    }
  }
  protected boolean hasType(InstrumentationType type) {
    return (instrumentations & type.bit) != 0;
  }
  int histogramThreshold = 10000;
  protected void setHistogramThreshold(int threshold) {
    histogramThreshold = threshold;
  }

  /*
   * Fixture for object allocation instrumentation
   */
  protected final AtomicLong totalObjectCount = new AtomicLong(0);
  protected final AtomicLong totalObjectSize = new AtomicLong(0);
  protected final Map<String, AtomicLong> classInstanceMap = new HashMap<String, AtomicLong>();
  protected final Map<String, AtomicLong> arrayInstanceMap = new HashMap<String, AtomicLong>();
  protected final LinkedList<AtomicLong> counters = new LinkedList<AtomicLong>();
  protected final Sampler objectSampler = new Sampler() {
    /**
     * Do not allocate new objects in this method.
     */
    public void sampleAllocation(int count, String clazz, Object newObj, long size) {
      if (hasType(InstrumentationType.COUNT_OF_OBJECT)) {
        totalObjectCount.incrementAndGet();
      }

      if (hasType(InstrumentationType.SIZE_OF_OBJECTS)) {
        totalObjectSize.addAndGet(size);
      }

      if (hasType(InstrumentationType.CLASS_HISTOGRAM)) {
        AtomicLong counter = null;
        if (count < 0) {
          counter = classInstanceMap.get(clazz);
          if (counter == null) {
            synchronized (classInstanceMap) {
              counter = classInstanceMap.get(clazz);
              if (counter == null && counters.size() > 0) {
                counter = counters.poll();
                classInstanceMap.put(clazz, counter);
              }
            }
          }
        } else {
          counter = arrayInstanceMap.get(clazz);
          if (counter == null) {
            synchronized (arrayInstanceMap) {
              counter = arrayInstanceMap.get(clazz);
              if (counter == null && counters.size() > 0) {
                counter = counters.poll();
                arrayInstanceMap.put(clazz, counter);
              }
            }
          }
        }
        counter.incrementAndGet();
      }
    }
  };

  /**
   * Reset all variables participating in object instrumentation.
   */
  @Before
  public void beforeTest() {
    if (hasType(InstrumentationType.CLASS_HISTOGRAM)) {
      counters.clear();
      for (int i = 0; i < 10000; i++) {
        counters.offer(new AtomicLong(0));
      }
      classInstanceMap.clear();
      arrayInstanceMap.clear();
    }
    if (hasType(InstrumentationType.COUNT_OF_OBJECT)) {
      totalObjectCount.set(0);
    }
    if (hasType(InstrumentationType.SIZE_OF_OBJECTS)) {
      totalObjectSize.set(0);
    }
  }

  @After
  public void afterTest() {
    if (hasType(InstrumentationType.SIZE_OF_OBJECTS)) {
      _logger.info("Total number of objects created for test {}: {}.",
          TEST_NAME.getMethodName(), totalObjectCount.longValue());
    }

    if (hasType(InstrumentationType.COUNT_OF_OBJECT)) {
      _logger.info("Total size of objects created for test {}: {} bytes.",
          TEST_NAME.getMethodName(), totalObjectSize.longValue());
    }

    if (hasType(InstrumentationType.CLASS_HISTOGRAM)) {
      List<Entry<String, AtomicLong>> sortedList = 
          Lists.newArrayListWithCapacity(classInstanceMap.size() + arrayInstanceMap.size());
      sortedList.addAll(classInstanceMap.entrySet());
      for (Entry<String, AtomicLong> entry : arrayInstanceMap.entrySet()) {
        sortedList.add(new SimpleEntry<String, AtomicLong>(entry.getKey()+"[]", entry.getValue()));
      }
      Collections.sort(sortedList, new Comparator<Entry<String, AtomicLong>>() {
        @Override
        public int compare(Entry<String, AtomicLong> o1, Entry<String, AtomicLong> o2) {
          return o2.getValue().intValue() - o1.getValue().intValue();
        }
      });

      Entry<String, AtomicLong> kv;
      StringBuilder sb = new StringBuilder();
      Iterator<Entry<String, AtomicLong>> it = sortedList.iterator();
      int limit = System.getProperty("print.all.allocations") == null ? histogramThreshold : 1;
      while (it.hasNext() && (kv = it.next()).getValue().longValue() >= limit) {
        sb.append(' ').append(kv.getKey()).append("\t").append(kv.getValue().longValue()).append('\n');
      }
      _logger.info("Allocation histogram (with at least {} instances) for test {}#{}()\n{}",
          limit, getClass().getSimpleName(), TEST_NAME.getMethodName(), sb);
    }
  }

  protected void startSampling() {
    AllocationRecorder.addSampler(objectSampler);
  }

  protected void stopSampling() {
    AllocationRecorder.removeSampler(objectSampler);
  }

}
