/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.tests;

import java.io.InputStream;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.Path;

import org.apache.log4j.Appender;
import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.FileAppender;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.apache.log4j.rolling.FixedWindowRollingPolicy;
import org.apache.log4j.rolling.RollingFileAppender;
import org.apache.log4j.rolling.TriggeringPolicy;
import org.apache.log4j.spi.LoggingEvent;
import org.junit.Before;
import org.junit.Rule;
import org.junit.rules.TestName;
import org.junit.rules.TestWatcher;
import org.junit.runner.Description;

import com.mapr.tests.utils.Tests;

public class BaseTest {

  long testDuration;
  private static org.apache.log4j.Logger _logger = org.apache.log4j.Logger.getLogger(BaseTest.class);

  // This code assumes the "repo-root-marker" file exists under the repo root.
  // If this directory changes its name or location, you need to change the getSourceRoot
  // function to return the correct repo root.
  static final String ROOT_MARKER_FILE = "repo-root-marker";

  @Rule public TestName TEST_NAME = new TestName();
  @Before
  public void printID() throws Exception {
    String beginMsg = String.format("%s######%s Running %s#%s ",
        Tests.yellow(), Tests.reset(),
        getClass().getSimpleName(), TEST_NAME.getMethodName());
    int numDots = 120 - beginMsg.length();
    System.err.printf(beginMsg + new String(new char[numDots]).replace('\0', '.'));
    testDuration = System.currentTimeMillis();
  }


  //Assume the current directory is somewhere inside our mercury tree, try to find the root
  //of the tree.
  public static Path getSourceRoot() {
    Path buildFsDirPath = searchPath(ROOT_MARKER_FILE);
    if (buildFsDirPath != null) {
      return buildFsDirPath.getParent();
    }
    return null;
  }

  // Searches from the current path to root, to check whether there is any directory
  // along the path contains the file or directory with the inputName.
  public static Path searchPath(String inputName) {

    FileSystem lfs  = FileSystems.getDefault();
    Path localPath = lfs.getPath(".").toAbsolutePath();

    Iterable<Path> dirs = lfs.getRootDirectories();
    boolean found = false;
    Path root = null;
    for (Path name : dirs) {
      if (localPath.getRoot() == name) {
        root = name;
        found = true;
        break;
      }
    }
    if (found) {
      _logger.info("search for directory "+inputName+" from current directory "+localPath+" to root directory "+root);
    } else {
      _logger.info("Could not find root dir for current directory "+localPath);
      return null;
    }

    Path retPath = null;
    while (localPath != root) {
      try {
        retPath = localPath.resolve(inputName);
        if (Files.exists(retPath)) {
          break;
        }
        localPath = localPath.getParent();
      } catch (InvalidPathException e) {
        localPath = localPath.getParent();
      }
    }
    return retPath;
  }

  protected static boolean isClusterTest() {
    return !(System.getProperty("tests") == null
          || System.getProperty("tests").equals("simple")
        );
  }

  final TestLogReporter LOG_OUTCOME = new TestLogReporter();
  private class TestLogReporter extends TestWatcher {
    @Override
    protected void failed(Throwable e, Description description) {
      System.err.printf("(%s) [%sFAILED%s]\n",
          getDuration(), Tests.red(), Tests.reset());
    }

    @Override
    public void succeeded(Description description) {
      System.err.printf("(%s) [%sPASSED%s]\n", getDuration(), Tests.green(), Tests.reset());
    }

    private String getDuration() {
      return String.format("%.3f s", (System.currentTimeMillis()-testDuration)/1000.0);
    }
  }

  @Rule public final TestLogReporter logOutcome = LOG_OUTCOME;

  /*
   * Logger configurations
   */
  protected static final Logger ROOT_LOGGER;
  protected static final Logger MAPR_LOGGER;
  protected static final ConsoleAppender STDOUT;
  protected static final FileAppender LOG_FILE;

  static {
    String pattern = System.getProperty("MAPR_TESTS_LOG_PATTERN", "%d %-5p [%t] %c - %m%n");

    LOG_FILE = createFileAppender(pattern);

    STDOUT = new ConsoleAppender();
    STDOUT.setLayout(new PatternLayout(pattern));
    STDOUT.activateOptions();
    STDOUT.setThreshold(Level.ERROR);
    ROOT_LOGGER = Logger.getRootLogger();
    ROOT_LOGGER.setLevel(Level.toLevel(System.getProperty("MAPR_TESTS_ROOT_LOGGER_LEVEL"), Level.INFO));
    ROOT_LOGGER.addAppender(STDOUT);
    ROOT_LOGGER.addAppender(LOG_FILE);

    MAPR_LOGGER = createLogger("com.mapr", STDOUT, LOG_FILE);
  }

  static volatile boolean not_rolled = true;
  protected static FileAppender createFileAppender(String pattern) {
    final String logDir = System.getProperty("MAPR_TESTS_LOG_DIR", "./target/test-logs");
    final String log4jLogFile = System.getProperty("MAPR_TESTS_LOG_FILE", 
        String.format("mapr-tests.log", System.currentTimeMillis()/1000));
    final String logFile = logDir+"/"+log4jLogFile;

    RollingFileAppender fa = new RollingFileAppender();
    fa.setName("FileLogger");
    fa.setFile(logFile);
    fa.setLayout(new PatternLayout(pattern));
    fa.setThreshold(Level.DEBUG);
    fa.setAppend(true);

    // this rolling policy rolls the log on startup
    fa.setTriggeringPolicy(new TriggeringPolicy() {
      @Override
      public void activateOptions() {}
      @Override
      public synchronized boolean isTriggeringEvent(
          Appender arg0, LoggingEvent arg1, String arg2, long arg3) {
        if (not_rolled) {
          not_rolled = false;
          return true;
        }
        return not_rolled;
      }
    });

    FixedWindowRollingPolicy rollingPolicy = new FixedWindowRollingPolicy();
    rollingPolicy.setFileNamePattern(logFile.toLowerCase().endsWith(".log")
        ? logFile.substring(0, logFile.length()-4) + ".%i.log"
        : logFile + ".%i");
    fa.setRollingPolicy(rollingPolicy);

    fa.activateOptions();
    return fa;
  }

  protected static Logger createLogger(String loggerId, Appender... appenders) {
    Logger logger = Logger.getLogger(loggerId);
    logger.setAdditivity(false);
    for (Appender appender : appenders) {
      logger.addAppender(appender);
    }
    logger.setLevel(Level.DEBUG);
    return logger;
  }

  protected InputStream getResourceStream(String resourceName) {
    return getClass().getClassLoader().getResourceAsStream(resourceName);
  }

}
