/*
 * Decompiled with CFR 0.152.
 */
package com.mapr.security.client;

import com.google.common.io.BaseEncoding;
import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.fs.proto.Security;
import com.mapr.security.client.MapRClientSecurityException;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.concurrent.ThreadLocalRandom;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClientSecurity {
    private static final Logger LOG = LoggerFactory.getLogger(ClientSecurity.class);
    public static final String WWW_AUTHENTICATE = "WWW-Authenticate";
    public static final String AUTHORIZATION = "Authorization";
    public static final String WWW_ERR_AUTHENTICATE = "WWW-MAPR-Err-Authenticate";
    public static final String NEGOTIATE = "MAPR-Negotiate";
    public static final String CLUSTER_CONFIG_LOCATION = "/conf/mapr-clusters.conf";
    public static final String DEFAULT_INSTALL_LOCATION = "/opt/mapr";
    private static int KEY_SIZE_IN_BYTES = 32;
    private static int TAG_SIZE_IN_BYTES = 16;
    private static int IV_SIZE_IN_BYTES = 16;
    private String currentClusterName;
    private long randomSecret = 0L;
    private Security.Key userKey;
    private String userTicketFile = null;
    private boolean isClusterSecure = false;
    private boolean isClusterValid = false;
    private String[] cldbs;

    public ClientSecurity() {
        this(null);
    }

    public ClientSecurity(String clusterName) {
        this.currentClusterName = clusterName;
        this.parseMaprClustersConf();
    }

    public boolean isSecurityEnabled() {
        return this.isClusterSecure;
    }

    public boolean isClusterNameValid() {
        return this.isClusterValid;
    }

    public String getUserTicketAndKeyFileLocation() throws MapRClientSecurityException {
        String mapRFileNameSuffix_;
        String mapRDefaultKeyFileLocation_;
        Object filePath = System.getenv("MAPR_TICKETFILE_LOCATION");
        if (filePath != null && !((String)filePath).isEmpty()) {
            return filePath;
        }
        String osName = System.getProperty("os.name");
        if (osName.equalsIgnoreCase("Windows")) {
            mapRDefaultKeyFileLocation_ = System.getenv("TEMP");
            mapRFileNameSuffix_ = System.getProperty("user.name");
        } else {
            String euid_;
            mapRDefaultKeyFileLocation_ = "/tmp";
            String userName = System.getProperty("user.name");
            ArrayList<String> command = new ArrayList<String>();
            command.add("id");
            command.add("-u");
            command.add(userName);
            try {
                euid_ = this.executeCommandAndReturnOutput(command);
            }
            catch (IOException e) {
                LOG.error("Unable to obtain effective UID for user " + userName + ":" + e.getMessage());
                throw new MapRClientSecurityException("Unable to obtain effective UID for user " + userName + ":" + e.getMessage());
            }
            catch (InterruptedException e) {
                LOG.error("Error executing command id -u " + userName + ": " + e.getMessage());
                throw new MapRClientSecurityException("Error execuring command id -u " + userName + ": " + e.getMessage());
            }
            mapRFileNameSuffix_ = euid_;
        }
        filePath = mapRDefaultKeyFileLocation_ + File.separator + "maprticket_" + mapRFileNameSuffix_;
        return filePath;
    }

    public String generateChallenge() throws MapRClientSecurityException {
        try {
            byte[] secretBytesEncrypted;
            LOG.debug("Generating challenge for cluster " + this.currentClusterName);
            Security.TicketAndKey ticketKey = null;
            ticketKey = this.authenticateIfNeeded();
            if (ticketKey == null) {
                LOG.error("No good client ticket found for cluster " + this.currentClusterName);
                throw new MapRClientSecurityException("No good client ticket found for cluster " + this.currentClusterName);
            }
            this.userKey = ticketKey.getUserKey();
            this.randomSecret = this.generateRandomNumber();
            byte[] writeBuffer = new byte[]{(byte)(this.randomSecret >>> 56), (byte)(this.randomSecret >>> 48), (byte)(this.randomSecret >>> 40), (byte)(this.randomSecret >>> 32), (byte)(this.randomSecret >>> 24), (byte)(this.randomSecret >>> 16), (byte)(this.randomSecret >>> 8), (byte)(this.randomSecret >>> 0)};
            Security.AuthenticationReqFull.Builder bld = Security.AuthenticationReqFull.newBuilder();
            try {
                secretBytesEncrypted = this.aesEncrypt(this.userKey.getKey().toByteArray(), writeBuffer);
            }
            catch (NoSuchAlgorithmException e) {
                LOG.error("AES-256 GCM not supported: " + e.getMessage());
                throw new MapRClientSecurityException("AES-256 GCM not supported: " + e.getMessage());
            }
            catch (NoSuchPaddingException e) {
                LOG.error("AES-256 GCM with no padding not supported: " + e.getMessage());
                throw new MapRClientSecurityException("AES-256 GCM with no padding not supported: " + e.getMessage());
            }
            catch (InvalidKeyException e) {
                LOG.error("Invalid AES-256 GCM user key: " + e.getMessage());
                throw new MapRClientSecurityException("Invalid AES-256 GCM user key: " + e.getMessage());
            }
            catch (InvalidAlgorithmParameterException e) {
                LOG.error("Invalid parameters for AES-256 GCM: " + e.getMessage());
                throw new MapRClientSecurityException("Invalid parameters for AES-256 GCM: " + e.getMessage());
            }
            catch (IllegalBlockSizeException e) {
                LOG.error("Illegal AES-256 GCM block size: " + e.getMessage());
                throw new MapRClientSecurityException("Illegal AES-256 GCM block size: " + e.getMessage());
            }
            catch (BadPaddingException e) {
                LOG.error("Bad padding for AES-256 GCM: " + e.getMessage());
                throw new MapRClientSecurityException("Bad padding for AES-256 GCM: " + e.getMessage());
            }
            bld.setEncryptedRandomSecret(ByteString.copyFrom((byte[])secretBytesEncrypted));
            bld.setEncryptedTicket(ticketKey.getEncryptedTicket());
            for (String cldb : this.cldbs) {
                bld.addCldb(cldb);
            }
            byte[] authRequestBytes = bld.build().toByteArray();
            String challengeString = null;
            try {
                BaseEncoding base64 = BaseEncoding.base64();
                challengeString = base64.encode(authRequestBytes);
            }
            catch (Exception e) {
                throw new MapRClientSecurityException("Unable to encode challenge: " + e.getMessage());
            }
            LOG.debug("Successfully obtained challenge");
            return challengeString;
        }
        catch (MapRClientSecurityException t) {
            LOG.error("Exception while processing ticket data: " + t.getMessage());
            throw new MapRClientSecurityException("Exception while processing ticket data", t);
        }
    }

    public boolean hasValidTicket() throws MapRClientSecurityException {
        Security.TicketAndKey tk = this.authenticateIfNeeded();
        return tk != null;
    }

    public String getClusterName() {
        return this.currentClusterName;
    }

    public boolean validateServerResponseToChallenge(String responseToChallenge) throws MapRClientSecurityException {
        byte[] base64Bytes = null;
        try {
            BaseEncoding base64 = BaseEncoding.base64();
            base64Bytes = base64.decode((CharSequence)responseToChallenge);
        }
        catch (Exception e) {
            throw new MapRClientSecurityException("Unable to decode Base64-encoded server challenge: " + e.getMessage());
        }
        Security.AuthenticationResp authResponse = null;
        try {
            byte[] decodedResponse = this.aesDecrypt(this.userKey.getKey().toByteArray(), base64Bytes);
            authResponse = Security.AuthenticationResp.parseFrom((byte[])decodedResponse);
            if (authResponse == null) {
                throw new MapRClientSecurityException("Response is null");
            }
        }
        catch (Exception e) {
            throw new MapRClientSecurityException("Error while decrypting response " + e.getMessage());
        }
        if (authResponse.hasChallengeResponse()) {
            LOG.debug("Response to challenge found");
            long responseSecret = authResponse.getChallengeResponse();
            if (responseSecret != this.randomSecret + 1L) {
                throw new MapRClientSecurityException("Incorrect challenge response");
            }
        } else {
            throw new MapRClientSecurityException("No response secret");
        }
        LOG.debug("Successfully validated server response");
        return true;
    }

    private byte[] aesEncrypt(byte[] key, byte[] plainText) throws NoSuchAlgorithmException, NoSuchPaddingException, InvalidKeyException, InvalidAlgorithmParameterException, IllegalBlockSizeException, BadPaddingException {
        Cipher cipher = Cipher.getInstance("AES/GCM/NoPadding");
        SecureRandom random = SecureRandom.getInstanceStrong();
        SecretKeySpec secretKey = new SecretKeySpec(key, 0, key.length, "AES");
        byte[] iv = new byte[IV_SIZE_IN_BYTES];
        random.nextBytes(iv);
        GCMParameterSpec spec = new GCMParameterSpec(TAG_SIZE_IN_BYTES * 8, iv);
        cipher.init(1, (Key)secretKey, spec);
        byte[] cipherText = cipher.doFinal(plainText);
        byte[] cipherTextWithIV = Arrays.copyOf(iv, iv.length + cipherText.length);
        System.arraycopy(cipherText, 0, cipherTextWithIV, iv.length, cipherText.length);
        return cipherTextWithIV;
    }

    private byte[] aesDecrypt(byte[] key, byte[] cipherText) throws NoSuchAlgorithmException, NoSuchPaddingException, InvalidKeyException, InvalidAlgorithmParameterException, IllegalBlockSizeException, BadPaddingException {
        Cipher cipher = Cipher.getInstance("AES/GCM/NoPadding");
        SecretKeySpec secretKey = new SecretKeySpec(key, 0, key.length, "AES");
        byte[] iv = new byte[IV_SIZE_IN_BYTES];
        for (int i = 0; i < IV_SIZE_IN_BYTES; ++i) {
            iv[i] = cipherText[i];
        }
        GCMParameterSpec spec = new GCMParameterSpec(TAG_SIZE_IN_BYTES * 8, iv);
        cipher.init(2, (Key)secretKey, spec);
        byte[] cipherTextWithoutIV = Arrays.copyOfRange(cipherText, IV_SIZE_IN_BYTES, cipherText.length);
        byte[] plainText = cipher.doFinal(cipherTextWithoutIV);
        return plainText;
    }

    private String executeCommandAndReturnOutput(ArrayList<String> command) throws IOException, InterruptedException {
        ProcessBuilder processBuilder = new ProcessBuilder(command);
        processBuilder.redirectErrorStream(true);
        Process process = processBuilder.start();
        StringBuilder processOutput = new StringBuilder();
        try (BufferedReader processOutputReader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
            String readLine;
            while ((readLine = processOutputReader.readLine()) != null) {
                processOutput.append(readLine + System.lineSeparator());
            }
            process.waitFor();
        }
        return processOutput.toString().trim();
    }

    private boolean isTicketAndKeyUsable(Security.TicketAndKey ticketAndKey) {
        long expiryTimeSec;
        long currentTimeSec = System.currentTimeMillis() / 1000L;
        return currentTimeSec < (expiryTimeSec = ticketAndKey.getExpiryTime());
    }

    private byte[] decodeDataFromKeyFile(String encodedData) {
        byte[] encryptedData = null;
        try {
            BaseEncoding encoding = BaseEncoding.base64();
            encryptedData = encoding.decode((CharSequence)encodedData);
        }
        catch (Exception e) {
            LOG.error("Unable to decode Base-64 encoded ticket: " + e.getMessage());
            return null;
        }
        byte[] key = this.getKeyForKeyFile();
        try {
            byte[] decryptedData = this.aesDecrypt(key, encryptedData);
            return decryptedData;
        }
        catch (Exception e) {
            LOG.error("Unable to decrypt data: " + e.getMessage());
            return null;
        }
    }

    private byte[] getKeyForKeyFile() {
        byte[] keybuf = new byte[KEY_SIZE_IN_BYTES];
        for (int i = 0; i < KEY_SIZE_IN_BYTES; ++i) {
            keybuf[i] = 65;
        }
        return keybuf;
    }

    private void parseMaprClustersConf() {
        String maprHomeDir = System.getenv("MAPR_HOME");
        String installDir = maprHomeDir != null ? (!maprHomeDir.isEmpty() ? maprHomeDir : DEFAULT_INSTALL_LOCATION) : DEFAULT_INSTALL_LOCATION;
        String clusterConfig = installDir + CLUSTER_CONFIG_LOCATION;
        try {
            String line;
            File file = new File(clusterConfig);
            FileReader fileReader = new FileReader(file);
            BufferedReader bufferedReader = new BufferedReader(fileReader);
            boolean firstLine = true;
            while ((line = bufferedReader.readLine()) != null) {
                int i;
                String[] elements = line.split(" ");
                String thisCluster = elements[0];
                if (firstLine) {
                    if (this.currentClusterName == null) {
                        this.currentClusterName = thisCluster;
                    }
                    firstLine = false;
                }
                if (!this.currentClusterName.equals(thisCluster)) continue;
                this.isClusterValid = true;
                int hostIdx = 1;
                for (i = 1; i < elements.length; ++i) {
                    if (!elements[i].startsWith("secure=")) continue;
                    String[] secureSetting = elements[i].split("=");
                    this.isClusterSecure = false;
                    if (secureSetting[1].equalsIgnoreCase("true")) {
                        this.isClusterSecure = true;
                    }
                    hostIdx = i + 1;
                    break;
                }
                this.cldbs = new String[elements.length - hostIdx];
                for (i = hostIdx; i < elements.length; ++i) {
                    String[] nics = elements[i].split(";");
                    this.cldbs[i - hostIdx] = nics[0];
                    ++i;
                }
            }
            bufferedReader.close();
        }
        catch (IOException e) {
            LOG.error("Failed to parse mapr-clusters.conf: " + e.getMessage());
        }
    }

    private long generateRandomNumber() {
        return ThreadLocalRandom.current().nextLong(1L, Long.MAX_VALUE);
    }

    private Security.TicketAndKey authenticateIfNeeded() throws MapRClientSecurityException {
        if (!this.isSecurityEnabled()) {
            LOG.debug("security appears to be off");
            return null;
        }
        Security.TicketAndKey tk = null;
        try {
            tk = this.getTicketAndKeyForCluster();
        }
        catch (MapRClientSecurityException e) {
            LOG.info("Unable to obtain user ticket for cluster " + this.currentClusterName);
            throw new MapRClientSecurityException(e);
        }
        if (this.isTicketAndKeyUsable(tk)) {
            LOG.debug("Already have good ticket, done");
            return tk;
        }
        return null;
    }

    private Security.TicketAndKey getTicketAndKeyForCluster() throws MapRClientSecurityException {
        Security.TicketAndKey clientTicketAndKey = null;
        byte[] decryptedTicketAndKeyStream = null;
        String encryptedClientTicketAndKey = null;
        try {
            String line;
            this.userTicketFile = this.getUserTicketAndKeyFileLocation();
            File ticketFile = new File(this.userTicketFile);
            boolean exists = ticketFile.exists();
            if (!exists) {
                throw new MapRClientSecurityException("Ticket file " + this.userTicketFile + " does not exist");
            }
            FileReader fileReader = new FileReader(ticketFile);
            BufferedReader bufferedReader = new BufferedReader(fileReader);
            boolean found = false;
            while ((line = bufferedReader.readLine()) != null) {
                String[] elements = line.split(" ");
                String thisClusterName = elements[0];
                if (!thisClusterName.equals(this.currentClusterName)) continue;
                encryptedClientTicketAndKey = elements[1];
                found = true;
                break;
            }
            bufferedReader.close();
            if (!found) {
                throw new MapRClientSecurityException("No user ticket found for cluster " + this.currentClusterName + " in " + this.userTicketFile);
            }
            if (encryptedClientTicketAndKey == null) {
                throw new MapRClientSecurityException("Unable to obtain encrypted user ticket");
            }
            decryptedTicketAndKeyStream = this.decodeDataFromKeyFile(encryptedClientTicketAndKey);
        }
        catch (IOException e) {
            LOG.error("IO Exception: " + e.getMessage());
            throw new MapRClientSecurityException("I/O Exception: " + e.getMessage());
        }
        try {
            clientTicketAndKey = Security.TicketAndKey.parseFrom((byte[])decryptedTicketAndKeyStream);
        }
        catch (InvalidProtocolBufferException e) {
            LOG.error("Failed to parse decrypted user ticket byte stream");
            e.printStackTrace();
        }
        return clientTicketAndKey;
    }
}

