/* Copyright (c) 2011 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.fs.jni;

import java.io.IOException;
import java.nio.ByteBuffer;

public class RowColParser {

  public RowColParser() {
  }

  public void Init(ByteBuffer byteBuf, int endOffset) {
    this.byteBuf = byteBuf;
    this.endOffset = endOffset;
    state = STATE.kInit;
  }

  public void RowColParserNext() throws IOException {
    switch (state) {
      case kInit:
        ParseValue();
        return;

      case kFoundBaseTS:
        ParseValue();
        return;
      
      case kFoundField:
        ParseValue();
        return;

      case kFoundValue:
        if (!isLast) {
          ParseValue();
          return;
        } else if (byteBuf.position() < endOffset) {
          ParseField();
          return;
        } else {
          state = STATE.kFinished;
          return;
        }

      case kFinished:
        //caller should have stopped
        throw new IOException("Parser is incorrectly used");
    }
    throw new IOException("Parser is incorrectly used");
  }

  public int getFieldOffset() { return fieldOffset; }
  public int getFieldLen() { return fieldLen; }
  public int getDataOffset() { return dataOffset; }
  public int getDataLen() { return dataLen; }
  public long getVersion() { return version; }
  public boolean getIsLast() { return isLast; }
  public STATE getState() { return state; }
  public ValType getValType() { return valType; }

  private void ParseField() {
    fieldLen = byteBuf.getShort() & 0x0000FFFF;
    fieldOffset = byteBuf.position();
    state = STATE.kFoundField;
    byteBuf.position(byteBuf.position() + fieldLen);
  }

  private void ParseValue() throws IOException {
    byte type = byteBuf.get();
    isLast = false;
    if ((type & ValType.kRowColLastValueFlag.getValue()) != 0) {
      isLast = true;
    }

    type &= ~ValType.kRowColLastValueFlag.getValue();
    valType = ValType.valtypeFromByte(type);
    if (valType != ValType.kRowColMap && 
        valType != ValType.kRowColBaseTS) {
      version = byteBuf.getLong();
    }

    dataOffset = byteBuf.position();
    switch (valType) {
      case kRowColBaseTS: {
        long baseTS = byteBuf.getLong();
        if (baseTS <= 0) {
          throw new IOException("Invalid baseTS"); // baseTS has to be > 0
        }
        dataLen = 0;
        break;
      }
      case kRowColMap: {
        dataLen = 0;
        break;
      }
      case kRowColDeleteAll:
      case kRowColDeleteExact: {
        dataLen = 0;
        break;
      }
      case kRowColBinary8: {
        int len = byteBuf.get() & 0x000000FF;
        byteBuf.position(byteBuf.position() + len);
        dataLen = byteBuf.position() - dataOffset;
        break;
      }
      case kRowColBinary16: {
        int len = byteBuf.getShort() & 0x0000FFFF;
        byteBuf.position(byteBuf.position() + len);
        dataLen = byteBuf.position() - dataOffset;
        break;
      }
      case kRowColBinary32: {
        int len = byteBuf.getInt();
        byteBuf.position(byteBuf.position() + len);
        dataLen = byteBuf.position() - dataOffset;
        break;
      }
      default: {
        throw new IOException("Invalid valtype still " + valType);
      }
    }

    if (valType == ValType.kRowColBaseTS) {
      state = STATE.kFoundBaseTS;
    } else {
      state = STATE.kFoundValue;
    }
  }

  private ByteBuffer byteBuf;
  private int endOffset;
  enum STATE { kInit, kFoundBaseTS, kFoundField, kFoundValue, kFinished };
  STATE state;
  //Related to field name
  int fieldOffset;
  int fieldLen;

  //Related to value
  boolean isLast;
  int dataOffset;
  int dataLen;
  long version;

  ValType valType;
  enum ValType {
    kRowColDeleteAll((byte)0x01),
    kRowColMap((byte)0x04),
    kRowColBinary8((byte)0x10),
    kRowColBinary16((byte)0x11),
    kRowColBinary32((byte)0x12),
    kRowColDeleteExact((byte)0x14),
    kRowColBaseTS((byte)0x16),
    kRowColLastValueFlag((byte)0x80);

    private ValType(byte inVal) {
      val = inVal;
    }

    public byte getValue() {
      return val;
    }

    static ValType valtypeFromByte(byte val) throws IOException {
      switch (val) {
        case 0x01: return kRowColDeleteAll;
        case 0x04: return kRowColMap;
        case 0x10: return kRowColBinary8;
        case 0x11: return kRowColBinary16;
        case 0x12: return kRowColBinary32;
        case 0x14: return kRowColDeleteExact;
        case 0x16: return kRowColBaseTS;
        case (byte)0x80: return kRowColLastValueFlag;
      }
      throw new IOException("Invalid valtype " + val);
    }

    private final byte val;
  }
}

