/* Copyright (c) 2011 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.fs.jni;
  
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

public class RowColDecoder {
  private RowColDecoderCallback cbs;

  RowColDecoder(RowColDecoderCallback cbs) {
    this.cbs = cbs;
  }

  public void Decode(ByteBuffer buf, int startOffset, int endOffset) throws IOException {
    if ((buf == null) || (buf.limit() == 0) || (buf.limit() <= startOffset) ||
        buf.limit() < endOffset) {
      throw new IOException("Invalid rowcol buffer");
    }

    RowColParser parser = new RowColParser();
    buf.order(ByteOrder.LITTLE_ENDIAN);
    buf.position(startOffset);
    parser.Init(buf, endOffset);

    while (true) {
      parser.RowColParserNext();

      switch (parser.getState()) {
        case kInit:
          throw new IOException("Invalid Parser State");

        case kFoundBaseTS:
          break;

        case kFoundField:
          cbs.foundField(parser.getFieldOffset(), parser.getFieldLen());
          break;

        case kFoundValue:
          switch (parser.getValType()) {
            case kRowColDeleteAll:
              cbs.foundValue(0 /*dataOffset*/, 0 /*dataLength*/,
                             parser.getVersion(), parser.getIsLast(),
                             1 /*isDelete*/, 0 /*isDeleteExact*/);
              break;

            case kRowColDeleteExact:
              cbs.foundValue(0 /*dataOffset*/, 0 /*dataLength*/,
                             parser.getVersion(), parser.getIsLast(),
                             0 /*isDelete*/, 1 /*isDeleteExact*/);
              break;

            case kRowColMap:
              continue;

            case kRowColBaseTS:
              continue;

            case kRowColBinary8:
              cbs.foundValue(parser.getDataOffset() + 1,
                             parser.getDataLen() - 1,
                             parser.getVersion(), parser.getIsLast(),
                             0 /*isDelete*/, 0 /*isDeleteExact*/);
              break;

            case kRowColBinary16:
              cbs.foundValue(parser.getDataOffset() + 2,
                             parser.getDataLen() - 2,
                             parser.getVersion(), parser.getIsLast(),
                             0 /*isDelete*/, 0 /*isDeleteExact*/);
              break;

            case kRowColBinary32:
              cbs.foundValue(parser.getDataOffset() + 4,
                             parser.getDataLen() - 4,
                             parser.getVersion(), parser.getIsLast(),
                             0 /*isDelete*/, 0 /*isDeleteExact*/);
              break;

            default:
              throw new IOException("Unrecognised data type");
          }
          break;
        case kFinished:
          return;
      }
    }
  }
}
