/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.fs.jni;

import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.Condition;
import java.nio.ByteBuffer;


// for now, all buffers are assumed to be fully occupied.
public class  Page {
  private enum CacheState { Invalid, Valid, Filling }

  public static final  boolean    ALLOC_IN_SHM = true;

  public boolean inlru;

  // basic state of cache page
  public long            pageId;   // in 8k units
  public InodeAttributes iattr;    // part of page's identity

  CacheState           state;
  public boolean       dirty;
  public boolean       isCompressed;
  public boolean       inShm;
  public boolean       allocatedForRA;
  public int           validStart, validLen;  // start and len of valid area within page
  public int           ref;

  public ByteBuffer    bbuf;
  public Condition     cv;

  public Page          prev, next;
  public Page          hnext;

  private native int    decompress();
  private native int    compress();
  private native int    releaseMemory(ByteBuffer b, boolean inShm, boolean allocatedForRA);
  private native boolean allocateByteBuf(int size, boolean inShm, boolean allocateForRA);

  public void     setValid() { state = CacheState.Valid; }
  public boolean  valid() { return state == CacheState.Valid;  }

  public boolean  filling() { return state == CacheState.Filling; }

  public void     markDirty() { dirty = true; }
  public boolean  invalid() { return state == CacheState.Invalid; }

  public boolean eq( Page p) { return iattr.eq( p.iattr) && pageId == p.pageId; }
  public boolean eq( InodeAttributes a, long pId) { return iattr.eq(a) && pageId == pId; }

  public String toString() { 
    return "(" + pageId
      +", state "+ state
      + ")"; 
  }

  public void     invalidate() {
    state = CacheState.Invalid;
    pageId = 0;
    validStart = validLen = 0;
    isCompressed = false;
  }

  public void     setFilling() {
    state = CacheState.Filling;
  }

  public void releaseStorage() {
    releaseMemory(bbuf, inShm, allocatedForRA);
    bbuf = null;
  }

  public Page(Lock lock, boolean inShm, int pageSize, boolean allocateForRA) {
    this.inShm = inShm;
    this.allocatedForRA = allocateForRA;
    pageId = 0;
    validStart = 0;
    validLen = pageSize;
    ref = 0;
    iattr = null;
    dirty = isCompressed = false;
    state = CacheState.Invalid;
    inShm = allocateByteBuf(pageSize, inShm, allocateForRA); // sets bbuf w/ shm
    if (lock != null) cv = lock.newCondition();
    inlru = false;
  }
}
