/* Copyright (c) 2011 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.fs.jni;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.HashMap;
import java.util.Map;

public class MapRResult {
  private ByteBuffer   bbuf;             // buffer pointing to data
  public byte[]        bufBytes;         // byte[] having data
  public int           bufSize;
  public int           keyLength;        // size of key ,key starts at off 0
  public boolean       isDelete;         // set to true if this row is being
                                         // deleted
  //TODO(nagrawal) : merge familyOffsets,cfIds in single array for performance
  //                 as this fields needs to be copied from jni
  public int[]         familyOffsets;
  public int[]         columnOffsets;
  public int[]         columnLengths;
  public long[]        timestamps;
  public int[]         valueOffsets;
  public int[]         valueLengths;
  public int[]     isColDelete;      // this field is used only when we
                                     // create a MapRResult from a bucket
                                     // row.
  public int[]     isColDeleteExact; // this field is currently ignored
  public int[]         cfIds;
  public int[]         cellsPerFamily;
  public int[]         versions;

  public MapRResult() {
    bbuf = null;
    bufBytes = null;
    bufSize = 0;
    columnOffsets = null;
    keyLength = 0;
    isDelete = false;
    columnLengths = null;
    timestamps = null;
    valueOffsets = null;
    valueLengths = null;
    isColDelete = null;
    isColDeleteExact = null;
    cfIds = null;
    cellsPerFamily = null;
    versions = null;
  }

  // copy the row key into 'keyBuf'
  public ByteBuffer getKey() {
    if (bbuf == null) {
      if (bufBytes != null)
        bbuf = ByteBuffer.wrap(bufBytes);

      if (bbuf == null) {
        return null;
      }

      bbuf.order(ByteOrder.LITTLE_ENDIAN);
    }

    bbuf.rewind();
    ByteBuffer keyBuf = ByteBuffer.allocate(getKeyLength());
    keyBuf.order(ByteOrder.LITTLE_ENDIAN);
    bbuf.get(keyBuf.array());

    return keyBuf;
  }

  // Split the read buffer into per CF buffer so it can encoded/merged
  public Map<Integer, ByteBuffer> getJsonByteBufs() throws IOException  {
    Map<Integer, ByteBuffer> idBufMap = new HashMap<Integer, ByteBuffer>();

    if (bbuf == null) {
      if (bufBytes != null)
        bbuf = ByteBuffer.wrap(bufBytes);

      if (bbuf == null) {
        return null;
      }
    }

    // Do not rewind as its expected to be at the position after key
    bbuf.order(ByteOrder.LITTLE_ENDIAN);

    int nfamilies = cfIds.length;
    if (nfamilies == 0) {
      return null;
    }

    if (nfamilies == 1) {
      ByteBuffer b = bbuf.slice();
      b.order(ByteOrder.LITTLE_ENDIAN);
      idBufMap.put(cfIds[0], b); //This is to satisfy expectation
                                 //of MapRPut to have BB beginning at 0
      return idBufMap;
    }

    // TODO BB - add these to sl4j logging
    // System.out.println("BBUF " + " " + bbuf.position() + " " + bbuf.limit());
    for (int f = 0; f < nfamilies; ++f) {
      int offset = familyOffsets[f] - getKeyLength();
      int length = 0;
      if (f != (nfamilies -1)) {
        length = familyOffsets[f+1] - familyOffsets[f];
      } else {
        length = bbuf.limit() - familyOffsets[f];
      }

      // System.out.println("FAM " + " " + offset + " " + length);

      ByteBuffer cfBuf = bbuf.slice();
      cfBuf.position(offset);
      cfBuf.limit(offset+length);
      cfBuf.order(ByteOrder.LITTLE_ENDIAN);

      idBufMap.put(cfIds[f], cfBuf);
    }

    return idBufMap;
  }

  public void DecodeByteBuf(ParsedRow prow) throws IOException {
    if (isEmpty() || isDelete) {
      return;
    }
    bbuf = ByteBuffer.wrap(bufBytes);

    prow.Clear();
    MapRResultDecoderCallback cbs = new MapRResultDecoderCallback(prow);
    RowColDecoder dec = new RowColDecoder(cbs);
    int nfamilies = cfIds.length;
    if (nfamilies == 0) {
      throw new IOException("Number of familes for non-empty Result is 0");
    }
    cellsPerFamily = new int[nfamilies];

    int lastNumCols = 0;
    for (int f = 0; f < nfamilies; ++f) {
      prow.parseColumnCalled = false;
      if (f != (nfamilies -1)) 
        dec.Decode(bbuf, familyOffsets[f], familyOffsets[f+1]);
      else
        dec.Decode(bbuf, familyOffsets[f], bbuf.limit());

      cellsPerFamily[f] = prow.getNumColumns() - lastNumCols;
      lastNumCols = prow.getNumColumns();
    }

    int numCols = prow.getNumColumns();
    int numVals = prow.getNumValues();
    columnOffsets = new int[numCols];
    columnLengths = new int[numCols];
    versions = new int[numCols];
    System.arraycopy(prow.coff, 0, columnOffsets, 0, numCols);
    System.arraycopy(prow.clen, 0, columnLengths, 0, numCols);
    System.arraycopy(prow.nval, 0, versions, 0, numCols);

    valueOffsets = new int[numVals];
    valueLengths = new int[numVals];
    timestamps = new long[numVals];
    isColDelete = new int[numVals];
    isColDeleteExact = new int[numVals];
    System.arraycopy(prow.valoff, 0, valueOffsets, 0, numVals);
    System.arraycopy(prow.vallen, 0, valueLengths, 0, numVals);
    System.arraycopy(prow.ts, 0, timestamps, 0, numVals);
    System.arraycopy(prow.isDelete, 0, isColDelete, 0, numVals);
    System.arraycopy(prow.isDeleteExact, 0, isColDeleteExact, 0, numVals);
  }

  public ByteBuffer getByteBuf() {
    return bbuf;
  }

  public int getBufSize() {
    return bufSize;
  }

  public int getKeyLength() {
    return keyLength;
  }

  public boolean getIsDelete() {
    return isDelete;
  }

  public int[] getColumnOffsets() {
    return columnOffsets;
  }

  public int[] getColumnLengths() {
    return columnLengths;
  }

  public int[] getValueOffsets() {
    return valueOffsets;
  }

  public int[] getValueLengths() {
    return valueLengths;
  }

  public int[] getIsColDelete() {
    return isColDelete;
  }

  public int[] getIsColDeleteExact() {
	return isColDeleteExact;
  }
  public long[] getTimeStamps() {
    return timestamps;
  }

  public int[] getCfids() {
    return cfIds;
  }

  public int[] getCellsPerFamily() {
    return cellsPerFamily;
  }

  public int[] versions() {
    return versions;
  }

  public boolean isEmpty() {
    // if there is no bytebuffer or no columns or no key
    if (bufSize == 0) {
      return true;
    }
    return false;
  }

  private class MapRResultDecoderCallback implements RowColDecoderCallback {
    private ParsedRow prow;

    public MapRResultDecoderCallback(ParsedRow prow) {
      this.prow = prow;
    }

    public void foundField(int fieldOffset, int fieldLen) {
      prow.curCoff = fieldOffset;
      prow.curClen = fieldLen;
      prow.curNVal = 0;
      prow.parseColumnCalled = true;
    }

    public void foundValue(int valueOffset, int valueLength,
                           long version, boolean isLast,
                           int isDelete, int isDeleteExact) {
      prow.addValue(valueOffset, valueLength, version, isDelete, isDeleteExact);
      ++prow.curNVal;
      // parseColumnCalled is only false for column family deletes.  We always
      // need to add a column for these operations, so ignore the isLast flag
      // in this case.  It is possible this code will need to change if we ever
      // support delete-exact for column families.
      if (!prow.parseColumnCalled) {
        prow.addColumn(0 /*colOff*/, 0 /*colLen*/, prow.curNVal);        
      } else if (isLast) {
        prow.addColumn(prow.curCoff, prow.curClen, prow.curNVal);        
      }
    }
  }
}
