/* Copyright (c) 2011 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.fs.jni;

import com.mapr.fs.jni.Errno;
import java.nio.ByteBuffer;
import java.util.LinkedList;
import java.util.Arrays;
import static com.mapr.fs.jni.MapRConstants.PutConstants.*;

/* MapRPut : convert's hbase put to mapr put format
 *   
 * Key will be embedded in one of key value's serialized buffer.
 */
public class MapRPut extends MapRAsyncRpc {
  private static final int SIZEOF_LONG = Long.SIZE / Byte.SIZE;

  public byte[]         key;            // extract key separately
  public int[]          families;       // sorted family ids
  public int[]          cellsPerFamily; // how many cells per family
  public MapRKeyValue[] keyvals;        // keyvalues sorted per family
  public long           rowTotalBytes;  // size of row
  public int            numFamilies;
  public int            numCells;
  public byte           type;
  public int            status;
  public long           rowTimeStamp;  // used only by delete row operation
  public byte[]         tableUuid;
  public MapRPut        next;          // ref to next object

  // keyvalues are empty if it is a rowcol encoded buffer from java layer
  public boolean        isEncoded;// Are values already encoded in rowcol format
                                  // (only for json table type).
  public ByteBuffer[]   encBuffers;
  public int[]          encPositions;  // start position of the corres encBuffer
  public int[]          encLimits;     // corres limit in encBuffer

  public MapRPut(Object request, MapRCallBackQueue cbq) {
    this.request = request;
    this.cbq = cbq;
  }

  public MapRPut() {
  }

  public MapRKeyValue[] getKeyValues() {
    return keyvals;
  }

  public boolean isEncoded()
  {
    return isEncoded;
  }

  public void setIsEncoded(boolean val)
  {
     isEncoded = val;
  }

  public static boolean isBufferedPut(MapRPut put) {

    if ((put.type == MapRConstants.PutConstants.TYPE_PUT_ROW_BUFFERED) ||
        (put.type == MapRConstants.PutConstants.TYPE_DELETE_ROW_BUFFERED) ||
        (put.type == MapRConstants.PutConstants.TYPE_DELETE_CELLS_BUFFERED) ||
        (put.type == MapRConstants.PutConstants.TYPE_DELETE_CELLS_EXACT_BUFFERED)) {
      return true;
    }
    return false;
  }
  
  // Used by async()
  private void MapRPutCommon(byte[] key, int[] reqFamilies,
                             byte [][][] qualifiers,  byte[][][] values,
                             long timestamp, long [][] timestamps, 
                             byte type) {    
    this.key = key;
    this.type = type;
    this.numCells = 0;
    this.isEncoded = false;
    this.encBuffers = null;
    this.encPositions = null;
    this.encLimits = null;

    if (reqFamilies == null) {
      numFamilies = 0;
    } else {
      numFamilies = reqFamilies.length;
    }

    families = new int[numFamilies];
    for (int i = 0; i < numFamilies; ++i) {
      families[i] = reqFamilies[i];
    }

    cellsPerFamily = new int[numFamilies];
    if (qualifiers == null) {
      for (int i = 0; i < numFamilies; ++i) {
        cellsPerFamily[i] = 0;
      }
    } else {
      for (int i = 0; i < numFamilies; ++i) {
        cellsPerFamily[i] = qualifiers[i].length;
        numCells += cellsPerFamily[i];
      }
    }

    rowTotalBytes = key.length;
    rowTimeStamp = timestamp;

    if (numCells > 0) {
      keyvals = new MapRKeyValue[numCells];
      int i = 0;
      for (int j = 0; j < numFamilies; ++j) {
        for (int k = 0; k < cellsPerFamily[j]; ++k) {
          boolean firstCell = false, lastCell = false;
          if ((k == 0) || 
              !Arrays.equals(qualifiers[j][k - 1], qualifiers[j][k])) 
            firstCell = true;
          if ((k == cellsPerFamily[j] - 1) || 
              !Arrays.equals(qualifiers[j][k], qualifiers[j][k + 1])) 
            lastCell = true;

          // delete operation will have just 1 timestamp
          long kvTimestamp = (timestamps == null) ? timestamp : 
                                                    timestamps[j][k];
          byte[] value = null;

          // values will be null if this is a delete operation
          if ((type & TYPE_ISDELETE_MASK) == 0) {
            value = values[j][k];
          }

          keyvals[i] = new MapRKeyValue(qualifiers[j][k], value, 
                                        kvTimestamp, firstCell, lastCell);
          ++i;

          int valLength = (value == null) ? 0 : values[j][k].length;
          rowTotalBytes += (qualifiers[j][k].length + valLength + 
                            SIZEOF_LONG);
        }
      }
    }
  }

  // v1.4.1-mapr async compareAndSet() constructor
  public MapRPut(byte[] key, int family, byte [][] qualifiers,
                 byte[][] values, long timestamp) {
    MapRPutCommon(key, new int[] {family}, new byte [][][] {qualifiers}, 
                  new byte [][][] {values}, timestamp, null, 
                  TYPE_PUT_ROW_ASYNC);
  }

  // v1.4.1-mapr aysnc put constructor
  public MapRPut( byte[] key, int family,
                  byte [][] qualifiers,  byte[][] values, long timestamp,
                  Object request, MapRCallBackQueue cbq) {
    this.request = request;
    this.cbq = cbq;
    MapRPutCommon(key, new int[] {family}, new byte [][][] {qualifiers},
                  new byte [][][] {values}, timestamp, null,
                  TYPE_PUT_ROW_ASYNC);
  }

  // aysnc compareAndSet() and delete constructor
  public MapRPut(byte[] key, int family, byte [][] qualifiers,
                  byte[][] values, long timestamp, byte type) {
    MapRPutCommon(key, new int[] {family}, new byte[][][] {qualifiers}, 
                  new byte[][][] {values}, timestamp, null, type);
  }

  // aysnc delete() constructor
  public MapRPut(byte[] key, int[] families, byte [][][] qualifiers,
                 byte[][][] values, long timestamp, long[][] timestamps, 
                 byte type) {
    MapRPutCommon(key, families, qualifiers, values, timestamp, timestamps, 
                  type);
  }

  // aysnc put constructor
  public MapRPut( byte[] key, int[] families,
                  byte [][][] qualifiers,  byte[][][] values, long timestamp,
                  long[][] timestamps, Object request, MapRCallBackQueue cbq) {
    this.request = request;
    this.cbq = cbq;

    // If cbq is null, it is AppendRequest
    if (cbq == null) {
      MapRPutCommon(key, families, qualifiers, values, timestamp, timestamps, 
                    TYPE_PUT_ROW);
    } else {
      // else, it is PutRequest
      MapRPutCommon(key, families, qualifiers, values, timestamp, timestamps, 
                    TYPE_PUT_ROW_ASYNC);
    }
  }

  public long rowTotalBytes() {
    return this.rowTotalBytes;
  }

  /* MapRAsyncRpc */
  public void done() {
    // walk the list of objects using next and add callback for each one
    LinkedList<Object> requests = new LinkedList<Object>();
    LinkedList<Object> responses = new LinkedList<Object>();

    Object error = "MultiActionSuccess";
    if (status != 0) {
      error = new Exception("Error in inserting row: " + 
                            Errno.toString(status) + "(" + status + ")"); 
    } else {
      //Hbase wants only errors back.
      if (isBufferedPut(this)) {
        //LOG.info("Buffered Put operation succeed. Skip notify listener");
        return;
      }
    }

    MapRPut req = this;
    while (req != null) {
      if (req.getAsyncRequest() != null) {
        requests.add(req.getAsyncRequest());
        responses.add(error);
      }
      MapRPut t = req.next;
      req.next = null;
      req = t;
    }
    this.callback(requests, responses);
  }
}
