/*
 *  Copyright (c) 2009 & onwards. MapR Technologies, Inc., All rights reserved
 */

package com.mapr.hadoop.mapred;
 
import java.io.IOException;
import java.util.Map;

import com.mapr.baseutils.BaseUtilsHelper;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.MapRFsDataOutputStream;
import com.mapr.fs.jni.IOExceptionWithErrorCode;

import org.junit.After;
import org.junit.Assert;
import org.junit.Before;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathId;
import org.apache.hadoop.mapred.JobID;
import org.apache.hadoop.yarn.api.records.ApplicationId;
import org.apache.hadoop.yarn.server.api.ApplicationInitializationContext;
import org.junit.Test;
import org.mockito.Mockito;
import org.mockito.invocation.InvocationOnMock;
import org.mockito.stubbing.Answer;

import static org.junit.Assert.*;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.reset;

public class LocalVolumeAuxServiceTest {
  private static final String TEST_HOST_NAME = "testhost";
  private static final String ROOT_FID = "2068.32.131304";
  private static final String ROOT_FID_ALT = "2070.32.131304";
  private static final String OUTPUT_FID = "2068.34.131308";
  private static final String OUTPUT_U_FID = "2068.36.131312";
  private static final String SPILL_FID = "2068.33.131306";
  private static final String SPILL_U_FID = "2068.35.131310";

  private static final String OUTPUT_FID_JOB = "2068.34.131309";
  private static final String OUTPUT_U_FID_JOB = "2068.36.131313";
  private static final String SPILL_FID_JOB = "2068.33.131307";
  private static final String SPILL_U_FID_JOB = "2068.35.131311";

  static {
    BaseUtilsHelper.setMapRHostName(TEST_HOST_NAME);
  }
  
  private static final Log LOG = LogFactory.getLog(LocalVolumeAuxServiceTest.class);
  
  private LocalVolumeAuxService lvService;
  private MapRFileSystem fs;
  private MapRFsDataOutputStream fileId;
  private Configuration conf;

  @Test
  public void testGetMapRedLocalVolumeMountPath() throws Exception {
    LocalVolumeAuxService service = new LocalVolumeAuxService();
    service.serviceInit(new Configuration());
    Assert.assertEquals("/var/mapr/local/testhost/mapred", service.getMapRedLocalVolumeMountPath());
  }
  
  @Before
  public void setup() throws Exception {
	  lvService = new LocalVolumeAuxService();

	  fs = mock(MapRFileSystem.class);
	  fileId = mock(MapRFsDataOutputStream.class);
	  
	  when(fileId.getFidServers()).thenReturn(new long[] {123,456,789});
	  when(fileId.getFidStr()).thenReturn(ROOT_FID);
	  
	  // let's define error conditions to return for each method
	  when(fs.mkdirsFid(new Path("/var/mapr/local/" + BaseUtilsHelper.getMapRHostName() + "/mapred/nodeManager"))).thenReturn(ROOT_FID);
	  when(fs.mkdirsFid(ROOT_FID, "output")).thenReturn(OUTPUT_FID);
	  when(fs.mkdirsFid(ROOT_FID, "output.U")).thenReturn(OUTPUT_U_FID);
	  when(fs.mkdirsFid(ROOT_FID, "spill")).thenReturn(SPILL_FID);
	  when(fs.mkdirsFid(ROOT_FID, "spill.U")).thenReturn(SPILL_U_FID);
	  
	  when(fs.createFid(ROOT_FID, "fidservers")).thenReturn(fileId);
	  when(fs.getFileStatus(new Path("/var/mapr/local/" + BaseUtilsHelper.getMapRHostName() + "/mapred"))).thenReturn(null);
	  when(fs.getFileStatus(new Path("/var/mapr/local/" + BaseUtilsHelper.getMapRHostName() + "/mapred/nodeManager"))).thenReturn(null);
	  lvService.setFS(fs);
	  conf = new Configuration();
	  conf.set("mapr.mapred.localvolume.root.dir.name", "nodeManager");
	  lvService.setConf(conf);
  }
  
  @After
  public void shutDown() throws Exception {
	  lvService = null;
	  reset(fs);
  }
  
  @Test(timeout=10000)
  public void testInitMapReduceDirsWithException() throws Exception {
    when(fs.createFid(ROOT_FID, "fidservers")).thenThrow(new IOExceptionWithErrorCode("IOExceptionWithErrorCode", com.mapr.fs.jni.Errno.EACCES));
	  
	  try {
	    lvService.initMapReduceDirs();
	    fail("Should throw exception");
	  } catch (Throwable t) {
	    if (t instanceof IOException) {
	      if (!"IOExceptionWithErrorCode".equalsIgnoreCase(t.getMessage())) {
	        fail("Unexpected exception is thrown");
	      }
	    } else {
	      fail("Unexpected exception is thrown");
	    }
	  }
  }
  
  @Test(timeout=10000)
  public void testInitMapReduceDirs() throws Exception {
    lvService.serviceInit(conf);
    lvService.initMapReduceDirs();
    
    MapRDirectShuffleMetaData metadata = lvService.metaData;
    Map<String, PathId> data = metadata.getMapReduceDirsPathIds();
    assertNotNull(data);
    assertEquals(5, data.size());
    PathId root = data.get(".");
    assertNotNull(root);
    assertEquals(ROOT_FID,root.getFid());
  }
  
  @Test(timeout=10000)
  public void testInitApplication() throws Exception {
    try {
      lvService.serviceInit(conf);
      lvService.initMapReduceDirs();      
    } catch (Throwable t) {
      fail("Should not throw");
    }

    // test init application w/o interference of volume check
    long appTimestamp = System.currentTimeMillis();
    ApplicationId appId = ApplicationId.newInstance(appTimestamp, 12345);
    ApplicationInitializationContext appContext = new ApplicationInitializationContext("myuser", appId, null);
    JobID jobId = new JobID(Long.toString(appId.getClusterTimestamp()), appId.getId());
    final String jobIdStr = jobId.toString();
    
    when(fs.mkdirsFid(OUTPUT_FID, jobIdStr)).thenReturn(OUTPUT_FID_JOB);
    when(fs.mkdirsFid(OUTPUT_U_FID, jobIdStr)).thenReturn(OUTPUT_U_FID_JOB);
    when(fs.mkdirsFid(SPILL_FID, jobIdStr)).thenReturn(SPILL_FID_JOB);
    when(fs.mkdirsFid(SPILL_U_FID, jobIdStr)).thenReturn(SPILL_U_FID_JOB);

    Mockito.doNothing().when(fs).setOwnerFid(OUTPUT_FID_JOB, "myuser", "myuser");
    Mockito.doNothing().when(fs).setOwnerFid(OUTPUT_U_FID_JOB, "myuser", "myuser");
    Mockito.doNothing().when(fs).setOwnerFid(SPILL_FID_JOB, "myuser", "myuser");
    Mockito.doNothing().when(fs).setOwnerFid(SPILL_U_FID_JOB, "myuser", "myuser");

    lvService.initializeApplication(appContext);
    
    MapRDirectShuffleMetaData data = lvService.jobMetaData.get(jobIdStr);
    assertNotNull(data);
    
    assertEquals(5,data.getMapReduceDirsPathIds().size());
    PathId root = data.getMapReduceDirsPathIds().get(".");
    assertNotNull(root);
    assertEquals(ROOT_FID,root.getFid());
    PathId output = data.getMapReduceDirsPathIds().get("output");
    assertNotNull(output);
    assertEquals(OUTPUT_FID_JOB,output.getFid());
  }
  
  @Test(timeout=30000)
  public void testLockingOnVolumeReInitNoException() throws Exception {
    final int [] switchFlag = new int[1];
    switchFlag[0] = 0;
    lvService.serviceInit(conf);
    when(fs.createFid(ROOT_FID_ALT, "fidservers")).thenAnswer(new Answer<MapRFsDataOutputStream>() {

      @Override
      public MapRFsDataOutputStream answer(InvocationOnMock invocation)
          throws Throwable {
        synchronized(switchFlag) {
          switchFlag[0] = 1;
          switchFlag.notifyAll();
        }
        Thread.sleep(6000l); 
        synchronized(switchFlag) {
          switchFlag[0] = 2;
          switchFlag.notifyAll();
        }
        return fileId;
      }});

    final Throwable[] internalT = new Throwable[2];

    Thread maprDirsInit = new Thread(new Runnable() {

      @Override
      public void run() {
        try {
          when(fs.mkdirsFid(new Path("/var/mapr/local/" + BaseUtilsHelper.getMapRHostName() + "/mapred/nodeManager"))).thenReturn(ROOT_FID_ALT);
          when(fs.mkdirsFid(ROOT_FID_ALT, "output")).thenReturn(OUTPUT_FID);
          when(fs.mkdirsFid(ROOT_FID_ALT, "output.U")).thenReturn(OUTPUT_U_FID);
          when(fs.mkdirsFid(ROOT_FID_ALT, "spill")).thenReturn(SPILL_FID);
          when(fs.mkdirsFid(ROOT_FID_ALT, "spill.U")).thenReturn(SPILL_U_FID);

          lvService.initMapReduceDirs();
        } catch (Throwable t) {
          LOG.error(t.getMessage(), t);
          internalT[0] = t;
          fail("Should not throw: " + t.getMessage());
        }
      }});
    
    Thread initApp = new Thread(new Runnable() {

      @Override
      public void run() {
        
        long appTimestamp = System.currentTimeMillis();
        ApplicationId appId = ApplicationId.newInstance(appTimestamp, 12345);
        ApplicationInitializationContext appContext = new ApplicationInitializationContext("myuser", appId, null);
        JobID jobId = new JobID(Long.toString(appId.getClusterTimestamp()), appId.getId());
        final String jobIdStr = jobId.toString();

        try {
          when(fs.mkdirsFid(OUTPUT_FID, jobIdStr)).thenReturn(OUTPUT_FID_JOB);
          when(fs.mkdirsFid(OUTPUT_U_FID, jobIdStr)).thenReturn(OUTPUT_U_FID_JOB);
          when(fs.mkdirsFid(SPILL_FID, jobIdStr)).thenReturn(SPILL_FID_JOB);
          when(fs.mkdirsFid(SPILL_U_FID, jobIdStr)).thenReturn(SPILL_U_FID_JOB);
  
          Mockito.doNothing().when(fs).setOwnerFid(OUTPUT_FID_JOB, "myuser", "myuser");
          Mockito.doNothing().when(fs).setOwnerFid(OUTPUT_U_FID_JOB, "myuser", "myuser");
          Mockito.doNothing().when(fs).setOwnerFid(SPILL_FID_JOB, "myuser", "myuser");
          Mockito.doNothing().when(fs).setOwnerFid(SPILL_U_FID_JOB, "myuser", "myuser");

          synchronized(switchFlag) {
            while (switchFlag[0] == 0) {
              try {
                switchFlag.wait();
              } catch (InterruptedException e) {
                break;
              }
            }
            assertEquals(1, switchFlag[0]);
          }
    
          long time = System.currentTimeMillis();
          lvService.initializeApplication(appContext);
          // should wait at least few secs.
          long timeDiff = (System.currentTimeMillis() - time);
          LOG.info("TimeDiff: " + timeDiff);
          
          assertTrue(timeDiff > 4000);
          MapRDirectShuffleMetaData data = lvService.jobMetaData.get(jobIdStr);
          assertNotNull(data);
          
          assertEquals(5,data.getMapReduceDirsPathIds().size());
          LOG.info(data.getMapReduceDirsPathIds());
          PathId root = data.getMapReduceDirsPathIds().get(".");
          assertNotNull(root);
          assertEquals(ROOT_FID_ALT,root.getFid());
          PathId output = data.getMapReduceDirsPathIds().get("output");
          assertNotNull(output);
          assertEquals(OUTPUT_FID_JOB,output.getFid());

        } catch (Throwable t) {
          LOG.error(t.getMessage(), t);
          internalT[1] = t;
          fail("Should not throw" + t.getMessage());
        }
      }});
    
    maprDirsInit.setDaemon(true);
    initApp.setDaemon(true);
    
    maprDirsInit.start();
    Thread.sleep(1000);
    initApp.start();
    
    maprDirsInit.join();
    initApp.join();
    
    if (internalT[0] != null) {
      fail("Exception Was thrown: " + internalT[0].getMessage());
    }
    if (internalT[1] != null) {
      fail("Exception Was thrown: " + internalT[1].getMessage());
    }
  }
  
  @Test(timeout=30000)
  public void testLockingOnVolumeReInitWithException() throws Exception {
    // init everything with original values
    testInitMapReduceDirs();
    
    // real test
    final int [] switchFlag = new int[1];
    switchFlag[0] = 0;
    lvService.serviceInit(conf);
    when(fs.createFid(ROOT_FID_ALT, "fidservers")).thenAnswer(new Answer<MapRFsDataOutputStream>() {

      @Override
      public MapRFsDataOutputStream answer(InvocationOnMock invocation)
          throws Throwable {
        synchronized(switchFlag) {
          switchFlag[0] = 1;
          switchFlag.notifyAll();
        }
        Thread.sleep(6000l); 
        synchronized(switchFlag) {
          switchFlag[0] = 2;
          switchFlag.notifyAll();
        }
        throw new IOExceptionWithErrorCode("IOExceptionWithErrorCode", com.mapr.fs.jni.Errno.EACCES);
      }});

    final Throwable[] internalT = new Throwable[2];
    Thread maprDirsInit = new Thread(new Runnable() {

      @Override
      public void run() {
        try {
          when(fs.mkdirsFid(new Path("/var/mapr/local/" + BaseUtilsHelper.getMapRHostName() + "/mapred/nodeManager"))).thenReturn(ROOT_FID_ALT);

          when(fs.mkdirsFid(ROOT_FID_ALT, "output")).thenReturn(OUTPUT_FID);
          when(fs.mkdirsFid(ROOT_FID_ALT, "output.U")).thenReturn(OUTPUT_U_FID);
          when(fs.mkdirsFid(ROOT_FID_ALT, "spill")).thenReturn(SPILL_FID);
          when(fs.mkdirsFid(ROOT_FID_ALT, "spill.U")).thenReturn(SPILL_U_FID);

          lvService.initMapReduceDirs();
          fail("Should not succeed");
        } catch (Throwable t) {
          if (t instanceof IOException) {
            if (!"IOExceptionWithErrorCode".equalsIgnoreCase(t.getMessage())) {
              internalT[0] = t;
              LOG.error(t.getMessage(), t);
              fail("Unexpected exception is thrown");
            }
          } else {
            internalT[0] = t;
            LOG.error(t.getMessage(), t);
            fail("Unexpected exception is thrown");
          }
        }
      }});
    
    Thread initApp = new Thread(new Runnable() {

      @Override
      public void run() {
        
        long appTimestamp = System.currentTimeMillis();
        ApplicationId appId = ApplicationId.newInstance(appTimestamp, 12345);
        ApplicationInitializationContext appContext = new ApplicationInitializationContext("myuser", appId, null);
        JobID jobId = new JobID(Long.toString(appId.getClusterTimestamp()), appId.getId());
        final String jobIdStr = jobId.toString();

        try {
          when(fs.mkdirsFid(OUTPUT_FID, jobIdStr)).thenReturn(OUTPUT_FID_JOB);
          when(fs.mkdirsFid(OUTPUT_U_FID, jobIdStr)).thenReturn(OUTPUT_U_FID_JOB);
          when(fs.mkdirsFid(SPILL_FID, jobIdStr)).thenReturn(SPILL_FID_JOB);
          when(fs.mkdirsFid(SPILL_U_FID, jobIdStr)).thenReturn(SPILL_U_FID_JOB);
  
          Mockito.doNothing().when(fs).setOwnerFid(OUTPUT_FID_JOB, "myuser", "myuser");
          Mockito.doNothing().when(fs).setOwnerFid(OUTPUT_U_FID_JOB, "myuser", "myuser");
          Mockito.doNothing().when(fs).setOwnerFid(SPILL_FID_JOB, "myuser", "myuser");
          Mockito.doNothing().when(fs).setOwnerFid(SPILL_U_FID_JOB, "myuser", "myuser");

          synchronized(switchFlag) {
            while (switchFlag[0] == 0) {
              try {
                switchFlag.wait();
              } catch (InterruptedException e) {
                break;
              }
            }
            assertEquals(1, switchFlag[0]);
          }
    
          long time = System.currentTimeMillis();
          lvService.initializeApplication(appContext);
          // should wait at least few secs.
          long timeDiff = (System.currentTimeMillis() - time);
          LOG.info("TimeDiff: " + timeDiff);
          
          assertTrue(timeDiff > 4000);
          MapRDirectShuffleMetaData data = lvService.jobMetaData.get(jobIdStr);
          assertNotNull(data);
          
          assertEquals(5,data.getMapReduceDirsPathIds().size());
          LOG.info(data.getMapReduceDirsPathIds());
          PathId root = data.getMapReduceDirsPathIds().get(".");
          assertNotNull(root);
          assertEquals(ROOT_FID,root.getFid());
          PathId output = data.getMapReduceDirsPathIds().get("output");
          assertNotNull(output);
          assertEquals(OUTPUT_FID_JOB,output.getFid());

        } catch (Throwable t) {
          LOG.error(t.getMessage(), t);
          internalT[1] = t;
          fail("Should not throw" + t.getMessage());
        }
      }});
    
    maprDirsInit.setDaemon(true);
    initApp.setDaemon(true);
    
    maprDirsInit.start();
    Thread.sleep(1000);
    initApp.start();
    
    maprDirsInit.join();
    initApp.join();
    
    if (internalT[0] != null) {
      fail("Exception Was thrown: " + internalT[0].getMessage());
    }
    if (internalT[1] != null) {
      fail("Exception Was thrown: " + internalT[1].getMessage());
    }
  }
}

