/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.yarn.client;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.yarn.conf.YarnConfiguration;

import com.mapr.fs.proto.Common.ServiceData;
import com.mapr.util.zookeeper.ZKDataRetrieval;

/**
 * Helper Utils class to get ServiceData from Zookeeper based on the configuration
 * so it can be reused in multiple places
 */
public class MapRZKRMFinderUtils {

  private static final Log LOG = LogFactory.getLog(MapRZKRMFinderUtils.class);
  
  private static final Map<String, ZKDataRetrieval> zkConnectionMap = new HashMap<String, ZKDataRetrieval>();

  public static final Map<String,String> propertyMapping = new HashMap<String, String>();
  
  static {
	  propertyMapping.put("SCHEDULER_PORT", YarnConfiguration.RM_SCHEDULER_ADDRESS);
	  propertyMapping.put("WEBAPP_PORT", YarnConfiguration.RM_WEBAPP_ADDRESS);
	  propertyMapping.put("WEBAPP_HTTPS_PORT", YarnConfiguration.RM_WEBAPP_HTTPS_ADDRESS);
	  propertyMapping.put("RESOURCETRACKER_PORT", YarnConfiguration.RM_RESOURCE_TRACKER_ADDRESS);
	  propertyMapping.put("ADMIN_PORT", YarnConfiguration.RM_ADMIN_ADDRESS);
  }
  
  private static volatile String backupZKAddress;

  /**
   * Helper method to get ServiceData from Zookeeper 
   * @param conf
   * @param serviceName
   * @return info about service
   */
  public static ServiceData mapRZkBasedRMFinder(final Configuration conf, String serviceName) {
    String zkAddress = null;
    
    if ( conf == null ) {
      LOG.error("Can not get RM address since passed config object is null");
      throw new RuntimeException("Can not get RM address since passed config object is null");
    }
    
    try {
      FileSystem fs = FileSystem.get(conf);
      final String scheme = fs.getScheme();
      if (!"maprfs".equals(scheme)) {
        // no other FS implements getZkConnectString
        logH("FileSystem object is not maprfs, but: " + scheme + " trying to getZkConnectString from MapRFileSystem impl");
        final Configuration other = new Configuration(conf);
        other.set("fs.default.name", "maprfs:///");
        fs = FileSystem.get(other);
      }
      zkAddress = fs.getZkConnectString();
    } catch (IOException e) {
      logH("Zookeeper address not found from MapRFilesystem. Will try the configuration from yarn-site.xml" + e);
    }

    if (zkAddress == null) {
      logH("Zookeeper address not found from MapRFilesystem. Will try the configuration from yarn-site.xml");
    }
    
    if ( zkAddress == null ) {
      logH("Zookeeper address can not be retrieved. Trying backup zk address");
      synchronized(MapRZKRMFinderUtils.class) {
        zkAddress = backupZKAddress;
      }
    }

    if(zkAddress == null) {
        LOG.error("Zookeeper address not configured in Yarn configuration. Please check yarn-site.xml.");
        LOG.error("Unable to determine ResourceManager service address from Zookeeper.");
        throw new RuntimeException("Zookeeper address not found from MapR Filesystem and is also not configured in Yarn configuration.");
    }
    synchronized(MapRZKRMFinderUtils.class) {
     backupZKAddress = zkAddress;
    }

    // Retrieve an already created zk connection or create one if needed.
    // A map is used to support the case where a client is used with multiple MapR clusters.
    // (eg Difftable in MapR Hbase is one such client.)
    ZKDataRetrieval zkConnection;
    synchronized (zkConnectionMap) {
        zkConnection = zkConnectionMap.get(zkAddress);
        if (zkConnection == null) {
            zkConnection = new ZKDataRetrieval(zkAddress);
            zkConnectionMap.put(zkAddress, zkConnection);
        }
    }

    ServiceData hostInfo = zkConnection.getServiceMasterData(serviceName);

    if(hostInfo == null) {
        LOG.error("Unable to determine ResourceManager service address from Zookeeper at " + zkAddress);
        throw new RuntimeException("Unable to determine ResourceManager service address from Zookeeper at " + zkAddress);
    }
    return hostInfo;
   }
  
  /**
   * If Security is enabled user should be known and logging error is valid
   * If Security is not enabled UI doe snot really have a valid user and we can end up with tons of 
   * benign error messages in the log. Moving them to DEBUG level messages
   */
  private static void logH(String logString) {
	if ( UserGroupInformation.isSecurityEnabled()) {
	  LOG.error(logString);
	} else {
	  LOG.debug(logString);
	}
  }
}
