package org.apache.hadoop.yarn.client;

import com.mapr.fs.proto.Common;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.CommonConfigurationKeysPublic;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.ipc.RPC;
import org.apache.hadoop.yarn.conf.YarnConfiguration;

import com.mapr.util.zookeeper.ZKDataRetrieval;
import com.mapr.fs.proto.Common.ServiceData;

import java.io.Closeable;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.HashMap;
import java.util.Map;

/**
 * Class implementing the MapR Zookeeper based RM Failover reconnect.
 *
 * When Warden starts ResourceManager, it publishes the host:port info
 * in Zookeeper /services/resourcemanager/master znode. We use that info
 * here to connect to RM instead of trying all nodes.
 *
 * When MapR's RM HA is configured, this class needs to be provided as
 * the value for the property yarn.client.failover-proxy-provider
 * in yarn-site.xml
 * @param <T>
 */
public class MapRZKBasedRMFailoverProxyProvider<T> implements RMFailoverProxyProvider<T> {

    private static final Log LOG = LogFactory.getLog(MapRZKBasedRMFailoverProxyProvider.class);

    private RMProxy<T> rmProxy;
    private YarnConfiguration conf;
    private Class<T> protocol;

    private T currentProxy;
    private InetSocketAddress currentRMAddress;
    private boolean isFailover = true;

    @Override
    public void init(Configuration conf, RMProxy<T> proxy, Class<T> protocol) {
        this.rmProxy = proxy;
        this.protocol = protocol;
        this.rmProxy.checkAllowedProtocols(this.protocol);
        this.conf = new YarnConfiguration(conf);

        conf.setInt(CommonConfigurationKeysPublic.IPC_CLIENT_CONNECT_MAX_RETRIES_KEY,
                conf.getInt(YarnConfiguration.CLIENT_FAILOVER_RETRIES,
                        YarnConfiguration.DEFAULT_CLIENT_FAILOVER_RETRIES));

        conf.setInt(CommonConfigurationKeysPublic.
                        IPC_CLIENT_CONNECT_MAX_RETRIES_ON_SOCKET_TIMEOUTS_KEY,
                conf.getInt(YarnConfiguration.CLIENT_FAILOVER_RETRIES_ON_SOCKET_TIMEOUTS,
                        YarnConfiguration.DEFAULT_CLIENT_FAILOVER_RETRIES_ON_SOCKET_TIMEOUTS));

    }

    private void updateCurrentRMAddress() {
      
        final ServiceData hostInfo = MapRZKRMFinderUtils.mapRZkBasedRMFinder(conf, "resourcemanager");

        String hostName = hostInfo.getHost();
        int rmAddressPort = hostInfo.getPort();
        // Set the hostname in the configuration.
        conf.set(YarnConfiguration.RM_HOSTNAME, hostName);

        // First lets set the RM Address from the usual port info in our Zookeeper.
        // we need to always set it as application may have set it to wrong value
        // for example Oozie always has to set it, but the value may not be correct
        conf.set(YarnConfiguration.RM_ADDRESS, hostName + ":" + rmAddressPort);

        // Go over the extended info put in by warden for RM from its config files.
        for(Common.ExtendedInfo extInfo : hostInfo.getExtinfoList()) {
            String portPropertyName = extInfo.getKey();
            int portNumber;
            try {
                portNumber = Integer.valueOf(extInfo.getValue());
            } catch (NumberFormatException e) {
                LOG.warn("Unable to get portNumber from RM Warden Configuration. Property: " + portPropertyName, e);
                continue;
            }
            int defaultYarnPort = -1;
            String addressProperty = null;

            switch (portPropertyName)
            {
                case "SCHEDULER_PORT":
                    defaultYarnPort = YarnConfiguration.DEFAULT_RM_SCHEDULER_PORT;
                    addressProperty = YarnConfiguration.RM_SCHEDULER_ADDRESS;
                    break;
                case "WEBAPP_PORT":
                    defaultYarnPort = YarnConfiguration.DEFAULT_RM_WEBAPP_PORT;
                    addressProperty = YarnConfiguration.RM_WEBAPP_ADDRESS;
                    break;
                case "WEBAPP_HTTPS_PORT":
                    defaultYarnPort = YarnConfiguration.DEFAULT_RM_WEBAPP_HTTPS_PORT;
                    addressProperty = YarnConfiguration.RM_WEBAPP_HTTPS_ADDRESS;
                    break;
                case "RESOURCETRACKER_PORT":
                    defaultYarnPort = YarnConfiguration.DEFAULT_RM_RESOURCE_TRACKER_PORT;
                    addressProperty = YarnConfiguration.RM_RESOURCE_TRACKER_ADDRESS;
                    break;
                case "ADMIN_PORT":
                    defaultYarnPort = YarnConfiguration.DEFAULT_RM_ADMIN_PORT;
                    addressProperty = YarnConfiguration.RM_ADMIN_ADDRESS;
                    break;
            }

            if(defaultYarnPort != -1 && addressProperty != null && defaultYarnPort != portNumber) {
                conf.set(addressProperty, hostName + ":" + portNumber);
            }
        }

        try {
            currentRMAddress = rmProxy.getRMAddress(conf, protocol);
        } catch (IOException e) {
            LOG.error("Unable to get RM address for RM at " + hostName + " and protocol " + protocol);
            throw new RuntimeException("Unable to get RM address for RM at " + hostName + " and protocol " + protocol, e);
        }
        LOG.info("Updated RM address to " + currentRMAddress);
    }

    @Override
    public synchronized ProxyInfo<T> getProxy() {
        if (isFailover) {
            try {
                updateCurrentRMAddress();
                currentProxy = rmProxy.getProxy(conf, protocol, currentRMAddress);
                isFailover = false;
            } catch (Exception e) {
                LOG.error("Unable to create proxy to the ResourceManager " + currentRMAddress, e);
            }
        }
        return new ProxyInfo<T>(currentProxy, "ResourceManager at " + currentRMAddress);
    }

    @Override
    public synchronized void performFailover(T currentProxy) {
        isFailover = true;
    }

    @Override
    public Class<T> getInterface() {
        return protocol;
    }

    @Override
    public synchronized void close() throws IOException {
        if (currentProxy instanceof Closeable) {
            ((Closeable) currentProxy).close();
        } else {
            RPC.stopProxy(currentProxy);
        }

    }
}
