/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.yarn.client;


import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.yarn.conf.CustomRMAddressFinder;
import org.apache.hadoop.yarn.conf.YarnConfiguration;

import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Common.ServiceData;

/**
 * MapR ZK based implementation of finding RM address based on the configuration
 *
 */
public class MapRZKBasedRMAddressFinder extends CustomRMAddressFinder {

  private static final Log LOG = LogFactory.getLog(MapRZKBasedRMAddressFinder.class);
   
  @Override
  public String getRMAddress(Configuration conf, String address,
      String defaultAddr, int defaultPort) {
    
    final ServiceData hostInfo = MapRZKRMFinderUtils.mapRZkBasedRMFinder(conf, "resourcemanager");

    String hostName = hostInfo.getHost();
    int rmAddressPort = hostInfo.getPort();
    // Set the hostname in the configuration.
    conf.set(YarnConfiguration.RM_HOSTNAME, hostName);

    // First lets set the RM Address from the usual port info in our Zookeeper.
    String currentRMAddress = hostName + ":" + rmAddressPort;
    conf.set(YarnConfiguration.RM_ADDRESS, currentRMAddress);

    if ( LOG.isDebugEnabled() ) {
      LOG.debug("Updated RM address to " + currentRMAddress);
    }

    if ( YarnConfiguration.RM_ADDRESS.equals(address)) {
      // not much more to do
      return currentRMAddress;
    }
    
    // Go over the extended info put in by warden for RM from its config files.
    for(Common.ExtendedInfo extInfo : hostInfo.getExtinfoList()) {
        String portPropertyName = extInfo.getKey();
        
        String yarnBasedProperty = MapRZKRMFinderUtils.propertyMapping.get(portPropertyName);
        
        int portNumber;
        try {
            portNumber = Integer.valueOf(extInfo.getValue());
        } catch (NumberFormatException e) {
            LOG.warn("Unable to get portNumber from RM Warden Configuration. Property: " + portPropertyName + "will use default one", e);
            portNumber = defaultPort;
        }
        
        if ( yarnBasedProperty != null && yarnBasedProperty.equalsIgnoreCase(address)) {
          currentRMAddress = hostName + ":" + portNumber;
          conf.set(address, currentRMAddress);
          return currentRMAddress;
        }
    }
    // if we are here something got wrong - return YarnConfiguration.RM_ADDRESS with default port #
    LOG.warn("Could not find respective property: " + address + " in ZK. Returning YarnConfiguration.RM_ADDRESS");
    return hostName + ":" + defaultPort;
  }
}
