/* Copyright (c) 2014 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.hadoop.yarn.util;

import java.io.IOException;

import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.yarn.api.records.ApplicationId;
import org.apache.hadoop.yarn.api.records.ContainerId;
import org.apache.hadoop.yarn.conf.YarnConfiguration;
import org.apache.hadoop.yarn.exceptions.YarnRuntimeException;
import org.apache.hadoop.yarn.util.DFSLoggingHandler;
import org.apache.hadoop.yarn.util.TaskLogUtil;

import com.mapr.baseutils.BaseUtilsHelper;

/**
 * Supports redirecting stdout and stderr generated by YARN applications to
 * MapRFS using maprcp.
 *
 * It also overrides the APIs to get log directories since MapR uses local
 * volumes to store the logs generated on each node. So there isn't a single
 * base directory under which all the application logs are stored.
 * For e.g., if the tasks for app-1 ran on 2 nodes, then the logs would be stored
 * under: /var/mapr/local/host-1/... and /var/mapr/local/host-2/... So to retrieve
 * its logs, we need to query 2 different directory hierarchies.
 */
public class MapRFSLoggingHandler extends DFSLoggingHandler {
  private static final String MAPR_INSTALL_DIR = BaseUtilsHelper.getPathToMaprHome();

  private FileSystem fs;

  private Path localVolumeLogPath;

  private Path globPath;

  public MapRFSLoggingHandler() {
    try {
      this.fs = FileSystem.get(TaskLogUtil.getConf());
    } catch (IOException e) {
      throw new YarnRuntimeException(e);
    }

    String glob = TaskLogUtil.getPropertyValue(
        YarnConfiguration.DFS_LOGGING_DIR_GLOB);
    this.globPath = new Path(glob);

    final String hostRegex = "*";
    this.localVolumeLogPath = new Path(
        glob.replace(hostRegex, BaseUtilsHelper.getMapRHostName()));
  }

  @Override
  public String getStdOutCommand(String filePath) {
    final Path localLogDir = new Path(filePath).getParent();
    return MAPR_INSTALL_DIR
      + "/bin/maprcp - "
      + filePath
      + " -autoflush "
      // TODO Uncomment
      // Note: filePath is actually a glob <LOG_DIR>. This is done for cross platform cases.
      /*+ (withLogDirMove ?
        (   " -cpdirents "
        + localLogDir + " "
        + getCentralUriStr(localLogDir.toString()))
        : "" )*/
      +"1>/dev/null 2>/dev/null";
  }

  @Override
  public String getStdErrCommand(String stderrFile) {
    return "";
  }
 
  @Override
  public Path getLogDirForWrite(String relativeContainerLogDir)
    throws IOException {

    // Note: Cannot use new Path(localVolumeLogPath, relativeContainerLogDir)
    // as it does not create the //. This is required for maprcp.
    // For e.g., we need maprfs:///var/mapr/local and not maprfs:/var/mapr/local
    return new Path(localVolumeLogPath.toString()
        + Path.SEPARATOR + relativeContainerLogDir);
  }

  @Override
  public Path[] getLogDir(ApplicationId applicationId) throws IOException {
    return globSearch(applicationId.toString());
  }

  @Override
  public Path getLogDir(ContainerId containerId) throws IOException {
    StringBuilder sb = new StringBuilder();
    sb.append(containerId.getApplicationAttemptId().getApplicationId().toString())
      .append(Path.SEPARATOR)
      .append(containerId.toString());

    Path[] paths = globSearch(sb.toString());
    if (paths == null || paths.length == 0) {
      throw new YarnRuntimeException("Log dir not present for container: "
          + containerId);
    }

    return paths[0];
  }

  private Path[] globSearch(String suffix) throws IOException {
    Path finalGlobPath = new Path(globPath, suffix);
    return FileUtil.stat2Paths(fs.globStatus(finalGlobPath));
  }
}
