/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.fs.hbase.test;

import com.mapr.fs.hbase.HTableDescriptorProxy;

import java.net.URI;
import java.util.Random;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Append;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.HBaseAdmin;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;

public class TestMultithreadedTuncate extends Configured implements Tool {

  private static final byte[] F = "f".getBytes();
  private static final byte[] Q = "qual".getBytes();
  private static final byte[] ROW = "row".getBytes();
  private static final byte[] LONG = "long".getBytes();;
  private final Put PUT = new Put(ROW);
  private final byte[] VALUE = new byte[32];
  private final Random RAND = new Random(System.currentTimeMillis());

  private FileSystem fs;
  private HBaseAdmin admin;
  private HTable table;

  private TableName tablePath;
  private HTableDescriptor tableDesc;
  private ExecutorService executor;
  private int numThreads;
  private Path folder;
  private Path tableName;
  private int failureCount;

  public TestMultithreadedTuncate(Configuration conf) {
    super(conf);
    folder = new Path("/tmp/multi/level/dir");
    tableName = new Path(folder, "TestMultithreadedTuncate");
    tablePath = TableName.valueOf(tableName.toString());
    tableDesc = new HTableDescriptor(tablePath);
    //Call through proxy to support both hbase 0.98 and 1.1
    HTableDescriptorProxy.addFamily(tableDesc, new HColumnDescriptor(F));

    RAND.nextBytes(VALUE);
    PUT.add(F, Q, VALUE);
  }

  @Override
  public int run(String[] args) throws Exception {
    int code = 0;
    fs = FileSystem.get(new URI("maprfs:///"), getConf());
    fs.delete(folder, true);
    fs.mkdirs(folder);
    admin = new HBaseAdmin(getConf());
    admin.createTable(tableDesc);
    table = new HTable(getConf(), tablePath);
    numThreads = Math.max(Runtime.getRuntime().availableProcessors(), 10);
    executor = Executors.newFixedThreadPool(numThreads);

    executor.submit(new TruncateTable());

    for (int i = 0; i < numThreads-1; i++) {
      executor.submit(new DoDBOps());
    }

    int durationInSeconds = 60;
    Timer timer = new Timer();
    timer.schedule(new ShutdownTask(), durationInSeconds*1000);

    executor.awaitTermination(Long.MAX_VALUE, TimeUnit.SECONDS);
    if (failureCount > 0) {
      System.out.println(failureCount + " threads failed with error.");
    } else {
      System.out.println("No failures occurred.");
    }
    return code;
  }

  class DoDBOps implements Callable<Void> {

    @Override
    public Void call() throws Exception {
      final Append APPEND = new Append(ROW);
      APPEND.add(F, Q, VALUE);

      final Get GET = new Get(ROW);

      try {
        while(true) {
          switch ((int)(Math.random()*5)) {
          case 0:
            table.put(PUT);
            System.out.println("Put successful.");
            break;
          case 1:
            table.incrementColumnValue(ROW, F, LONG, 1);
            System.out.println("Increament successful.");
            break;
          case 2:
            table.append(APPEND);
            System.out.println("Append successful.");
            break;
          case 3:
            ResultScanner scanner = table.getScanner(F);
            Result res = scanner.next();
            scanner.close();
            System.out.println("Scan successful. Result = " + res);
            break;
          default:
            table.get(GET);
            System.out.println("Get successful.");
            break;
          }
          Thread.sleep((long) (50*Math.random())); // sleep up to 50 ms
        }
      } catch (InterruptedException e) {
        System.out.println("DoDBOps shutting down");
      } catch (Throwable e) {
        failureCount++;
        System.out.println("Uncaught exception " + e);
        e.printStackTrace();
        throw e;
      }
      return null;
    }

  }

  class TruncateTable implements Callable<Void> {
    @Override
    public Void call() throws Exception {
      try {
        while(true) {
          Thread.sleep(1000 + (int)(4000*Math.random())); // sleep for 1-5 seconds
          if ((int)(3*Math.random()) == 1) {
            admin.truncateTable(tablePath, true);
            table.put(PUT);
            System.out.println("Table truncated.");
          } else {
            fs.delete(folder, true);
            fs.mkdirs(folder);
            admin.createTable(tableDesc);
            table.put(PUT);
            System.out.println("Table recreated.");
          }
        }
      } catch (InterruptedException e) {
        System.out.println("TruncateTable shutting down");
      }
      return null;
    }
  }

  class ShutdownTask extends TimerTask {
    public void run() {
      executor.shutdownNow();
    }
  }

  public static void main(String[] args) throws Exception {
    int ret = ToolRunner.run(new TestMultithreadedTuncate(HBaseConfiguration.create()), args);
    System.exit(ret);
  }

}
