package com.mapr.fs.hbase.test;
import java.util.ArrayList;

import com.mapr.fs.hbase.HTableDescriptorProxy;
import com.mapr.fs.hbase.tools.mapreduce.SegKeyRangeUtil;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.DoNotRetryIOException;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.KeyValue.Type;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.HBaseAdmin;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.client.RowMutations;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.filter.CompareFilter.CompareOp;
import org.apache.hadoop.hbase.filter.NullComparator;
import org.apache.hadoop.hbase.filter.SingleColumnValueFilter;
import org.apache.hadoop.hbase.filter.SubstringComparator;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.util.Bytes;

/* Run the test with two table names, one for hbase table, one for mapr tables. 
 * This test compares the result of the two tables, and make sure our mapr table
 * results are consistent with hbase tables.
 */
public class TestAtomicOpsWithNullValue {

  // return table is created or not.
  public static boolean creatTable(String tableName, String[] familys, Configuration conf)
      throws Exception {

    HBaseAdmin admin = new HBaseAdmin(conf);
    if (admin.tableExists(tableName)) {
      System.out.println("table already exists!");
      return false;
    }
    HTableDescriptor tableDesc = new HTableDescriptor(tableName);
    for (int i = 0; i < familys.length; i++) {
      //Call through proxy to support both hbase 0.98 and 1.1
      HTableDescriptorProxy.addFamily(tableDesc, new HColumnDescriptor(familys[i]));
    }
    admin.createTable(tableDesc);
    System.out.println("create table " + tableName + " ok.");
    return true;
  }

  public static void printscan(HTable ht1, Scan scan)
      throws Exception {

    ArrayList<Result> ht1rs = new ArrayList<Result>();
  
    ResultScanner rs1;
    rs1=ht1.getScanner(scan);
    System.out.println("--------"+ht1.getName()+" Scan Results---------");
    for (Result result : rs1) {
      //ht1rsrowkey.add(result.getRow());
      System.out.println(SegKeyRangeUtil.resultToString(result));
      ht1rs.add(result);
    }
    rs1.close();
  }

  public static void main(String args[]) throws Exception{

    if (args.length < 1) {
      System.out.println("NULL Test for SingleColumnValueFilter. Usage:\n"
                       + "\t hbase com.mapr.fs.hbase.test.TestAtomicOpsWithNullValue tableName zkQuorumIP zkClientPort");
      return;
    }

    byte[] row0 = Bytes.toBytes("r0");
    byte[] row1 = Bytes.toBytes("r1");
    byte[] row2 = Bytes.toBytes("r2");
    byte[] row3 = Bytes.toBytes("r3");
    byte[] rowx = Bytes.toBytes("rx"); //non-exist column

    byte[] fam0 = Bytes.toBytes("f0");
    byte[] fam1 = Bytes.toBytes("f1");
    byte[] fam2 = Bytes.toBytes("f2");
    byte[] famx = Bytes.toBytes("fx"); //non-exist family

    byte[] qual0 = Bytes.toBytes("c0");
    byte[] qual1 = Bytes.toBytes("c1");
    byte[] qual2 = Bytes.toBytes("c2");
    byte[] qualx = Bytes.toBytes("cx"); //non-exist qualx

    //row0 has every family:qualifier with non-empty value
    byte[] r0val00 = Bytes.toBytes("r0-v00");
    byte[] r0val01 = Bytes.toBytes("r0-v01");
    byte[] r0val02 = Bytes.toBytes("r0-v02");
    byte[] r0val10 = Bytes.toBytes("r0-v10");
    byte[] r0val11 = Bytes.toBytes("r0-v11");
    byte[] r0val12 = Bytes.toBytes("r0-v12");
    byte[] r0val20 = Bytes.toBytes("r0-v20");
    byte[] r0val21 = Bytes.toBytes("r0-v21");
    byte[] r0val22 = Bytes.toBytes("r0-v22");  

    //row1 family0 has every qualifier, family1 has qualifier0 and 1, and family2 has qualifier0
    byte[] r1val00 = Bytes.toBytes("r1-v00");
    byte[] r1val01 = Bytes.toBytes("r1-v01");
    byte[] r1val02 = Bytes.toBytes("r1-v02");
    byte[] r1val10 = Bytes.toBytes("r1-v10");
    byte[] r1val11 = Bytes.toBytes("r1-v11");
    byte[] r1val20 = Bytes.toBytes("r1-v20");

    //row2 family0 has qualifier0 with binary value, family1 has qualifier1 with empty value, 
    //family2 has no qualifier.
    byte[] binval = new byte[] {10,20,30,40,50,60,70,80,90};
    byte[] valempty = Bytes.toBytes("");

    byte[] chgr1val00 = Bytes.toBytes("chg-r1-v00");
    byte[] chgr1val01 = Bytes.toBytes("chg-r1-v01");
    byte[] chgr1val02 = Bytes.toBytes("chg-r1-v02");

    String tableName1 = args[0];
    String quorum = "localhost";
    if (args.length > 1) {
      quorum = args[1];
    }
    String port = "5181";
    if (args.length > 2) {
      port = args[2];
    }

    // Instantiating configuration class
    Configuration conf = new Configuration();
    conf.set("hbase.zookeeper.quorum", quorum);
    conf.set("hbase.zookeeper.property.clientPort", port);

    System.out.println("hbase.zookeeper.quorum = " + conf.get("hbase.zookeeper.quorum")
                     + "hbase.zookeeper.property.clientPort = " + conf.get("hbase.zookeeper.property.clientPort"));

    boolean ret1;
    String[] familys = {new String(fam0), new String(fam1), new String(fam2)};
    boolean table1IsNew = creatTable(tableName1, familys, conf);
    HTable ht1 = new HTable(conf, tableName1);

    Scan simplescan = new Scan();
    ResultScanner rs1;
    if (!table1IsNew) {
      rs1=ht1.getScanner(simplescan);
      for (org.apache.hadoop.hbase.client.Result result : rs1) {
        ht1.delete(new Delete(result.getRow()));
      }
      rs1.close();
    }

    /* rows are like:
     */

    Put put0 = new Put(row0);
    put0.add(fam0, qual0, r0val00);
    put0.add(fam0, qual1, r0val01);
    put0.add(fam0, qual2, r0val02);
    put0.add(fam1, qual0, r0val10);
    put0.add(fam1, qual1, r0val11);
    put0.add(fam1, qual2, r0val12);
    put0.add(fam2, qual0, r0val20);
    put0.add(fam2, qual1, r0val21);
    put0.add(fam2, qual2, r0val22);

    Put put1 = new Put(row1);
    put1.add(fam0, qual0, r1val00);
    put1.add(fam0, qual1, r1val01);
    put1.add(fam0, qual2, r1val02);
    put1.add(fam1, qual0, r1val10);
    put1.add(fam1, qual1, r1val11);
    put1.add(fam2, qual0, r1val20);

    Put put2 = new Put(row2);
    put2.add(fam0, qual0, binval);
    put2.add(fam0, qual1, valempty);

    ht1.put(put0);
    ht1.put(put1);
    ht1.put(put2);
    ht1.flushCommits();

    printscan(ht1, simplescan);
    System.out.println("After populated data to table "+tableName1);

    byte[] chgr1val12 = Bytes.toBytes("chg-r1-v12: this should not be added");
    RowMutations rm112 = new RowMutations(row1);
    Put chgput112 = new Put(row1);
    chgput112.add(fam1, qual2, chgr1val12);
    rm112.add(chgput112);

    RowMutations rm100 = new RowMutations(row1);
    Put chgput100 = new Put(row1);
    chgput100.add(fam0, qual0, chgr1val00);
    rm100.add(chgput100);

    RowMutations rm101 = new RowMutations(row1);
    Put chgput101 = new Put(row1);
    chgput101.add(fam0, qual1, chgr1val01);
    rm101.add(chgput101);

    byte[] chgr1val01b = Bytes.toBytes("This should not be added for row1:fam0:qual1");
    RowMutations rm101b = new RowMutations(row1);
    Put chgput101b = new Put(row1);
    chgput101b.add(fam0, qual1, chgr1val01b);
    rm101b.add(chgput101b);

    RowMutations rm102 = new RowMutations(row1);
    Put chgput102 = new Put(row1);
    chgput102.add(fam0, qual2, chgr1val02);
    rm102.add(chgput102);

    byte[] chgrxval00 = Bytes.toBytes("This will be added then deleted");
    RowMutations rmx00 = new RowMutations(rowx);
    Put chgputx00 = new Put(rowx);
    chgputx00.add(fam0, qual0, chgrxval00);
    rmx00.add(chgputx00);
    
    RowMutations rmx00d = new RowMutations(rowx);
    Delete del00 = new Delete(rowx);
    rmx00d.add(del00);

    System.out.println("\n========Test1: checkandmutate with not-exist family ========");
    ret1 = ht1.checkAndMutate(row1, famx, qual1, CompareOp.EQUAL, null, rm100);
    System.out.println("checkandmutate for key("+Bytes.toStringBinary(row1)+") fam:col("+Bytes.toStringBinary(famx)+":"
                      +Bytes.toStringBinary(qual1)+") input family does not exist, this is an error case, should NOT apply mutate put("+
                      Bytes.toStringBinary(chgr1val00)+"), result=("+ret1+")");
    printscan(ht1, simplescan);
    assert(!ret1);

    System.out.println("\n========Test2: checkandmutate with exist family, qualifier and value ========");
    ret1 = ht1.checkAndMutate(row1, fam1, qual1, CompareOp.EQUAL, null, rm100);
    System.out.println("checkandmutate for key("+Bytes.toStringBinary(row1)+") fam:col("+Bytes.toStringBinary(fam1)+":"
                      +Bytes.toStringBinary(qual1)+") has value "+Bytes.toStringBinary(r1val11)+", input value is null,"
                      +" operator is EQUAL, should NOT apply mutate put("+
                      Bytes.toStringBinary(chgr1val00)+"), result=("+ret1+")");
    printscan(ht1, simplescan);
    assert(!ret1);

    System.out.println("\n========Test3: checkandmutate with exist family, qualifier and value ========");
    ret1 = ht1.checkAndMutate(row1, fam1, qual1, CompareOp.NOT_EQUAL, null, rm100);
    System.out.println("checkandmutate for key("+Bytes.toStringBinary(row1)+") fam:col("+Bytes.toStringBinary(fam1)+":"
                      +Bytes.toStringBinary(qual1)+") has value "+Bytes.toStringBinary(r1val11)+", input value is null,"
                      +" operator is NOT_EQUAL, should apply mutate put("+
                      Bytes.toStringBinary(chgr1val00)+"), result=("+ret1+")");
    printscan(ht1, simplescan);
    assert(ret1);

    System.out.println("\n========Test4: checkandmutate with not-exist qualifier ========");
    ret1 = ht1.checkAndMutate(row1, fam1, qualx, CompareOp.EQUAL, null, rm101);
    System.out.println("checkandmutate for key("+Bytes.toStringBinary(row1)+") fam:col("+Bytes.toStringBinary(fam1)+":"
                      +Bytes.toStringBinary(qualx)+") input qualifier does not exist, should apply mutate put("+
                      Bytes.toStringBinary(chgr1val01)+"), result=("+ret1+")");
    printscan(ht1, simplescan);
    assert(ret1);

    System.out.println("\n========Test5: checkandmutate with not-exist qualifier ========");
    ret1 = ht1.checkAndMutate(row1, fam1, qualx, CompareOp.NOT_EQUAL, null, rm101b);
    System.out.println("checkandmutate for key("+Bytes.toStringBinary(row1)+") fam:col("+Bytes.toStringBinary(fam1)+":"
                      +Bytes.toStringBinary(qualx)+") input qualifier does not exist, the comparator is NOT_EQUAL, should NOT apply mutate put("+
                      Bytes.toStringBinary(chgr1val01b)+"), result=("+ret1+")");
    printscan(ht1, simplescan);
    assert(!ret1);

    System.out.println("\n========Test6: checkandmutate with not-exist row ========");
    ret1 = ht1.checkAndMutate(rowx, fam0, qualx, CompareOp.EQUAL, null, rmx00);
    System.out.println("checkandmutate for key("+Bytes.toStringBinary(rowx)+") fam:col("+Bytes.toStringBinary(fam0)+":"
                      +Bytes.toStringBinary(qual0)+") input row does not exist, should apply mutate put("+
                      Bytes.toStringBinary(chgrxval00)+"), result=("+ret1+")");
    printscan(ht1, simplescan);
    assert(ret1);

    System.out.println("\n========Test7: checkandmutate with not-exist row ========");
    ret1 = ht1.checkAndMutate(rowx, fam0, qualx, CompareOp.EQUAL, null, rmx00d);
    System.out.println("checkandmutate for key("+Bytes.toStringBinary(rowx)+") fam:col("+Bytes.toStringBinary(fam0)+":"
                      +Bytes.toStringBinary(qual0)+") input qualifier does not exist, should apply mutate deletel), result=("+ret1+")");
    printscan(ht1, simplescan);
    assert(ret1);

    boolean throwerr = false;
    System.out.println("\n========Test8: checkandmutate on different row ========");
    try {
      ret1 = ht1.checkAndMutate(rowx, fam1, qualx, CompareOp.EQUAL, null, rm101b);
    } catch (DoNotRetryIOException e) {
      throwerr = true;
      System.out.println("checkandmutate for key("+Bytes.toStringBinary(rowx)+") fam:col("+Bytes.toStringBinary(fam1)+":"
                      +Bytes.toStringBinary(qualx)+") input qualifier does not exist, the comparator is EQUAL, "
                      +" however, the mutation is on key("+Bytes.toStringBinary(row1)+"should NOT apply mutate put("+
                      Bytes.toStringBinary(chgr1val01b)+"), result=("+ret1+")");
    }
    printscan(ht1, simplescan);
    assert(throwerr);
    ht1.close();
  }
}
