/**
 * Copyright The Apache Software Foundation
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.mapr.fs.hbase.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.io.IOException;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.client.mapr.BaseTableMappingRules;
import org.apache.hadoop.hbase.client.mapr.TableMappingRulesFactory;
import org.junit.BeforeClass;
import org.junit.Test;

import com.mapr.fs.hbase.MapRTableMappingRules;

public class TableMappingRuleTest {
  static Configuration conf_;
  static FileSystem fs_;
  private static String home_;

  @BeforeClass
  public static void setup() throws IOException {
      conf_ = HBaseConfiguration.create();
      fs_ = FileSystem.get(conf_);
      home_ = fs_.getHomeDirectory().toString();
  }

  @Test
  public void testNoPathMapping() throws IOException {
    conf_.set(MapRTableMappingRules.HBASE_TABLE_NAMESPACE_MAPPINGS, "");
    BaseTableMappingRules tableMappingRule = TableMappingRulesFactory.create(conf_);

    checkAlwaysMaprTables(tableMappingRule);

    assertFalse(tableMappingRule.isMapRTable("hbasetable"));
    assertNull(tableMappingRule.getMaprTablePath("hbasetable"));
  }
  
  @Test
  public void testAbsoluteMappings() throws IOException {
    conf_.set(MapRTableMappingRules.HBASE_TABLE_NAMESPACE_MAPPINGS,
        "table1:/root/t1," +
        "table2:/user/abc/r/," +
        "table1*:/p/q," +
        "table*:/x/y," +
        "aditya1:maprfs:/mapr/cluster1/user/aditya/default," +
        "aditya2:maprfs://mapr/cluster2/user/aditya/default," +
        "aditya*:maprfs:///mapr/cluster3/user/aditya");

    BaseTableMappingRules tableMappingRule = TableMappingRulesFactory.create(conf_);

    checkAlwaysMaprTables(tableMappingRule);

    assertTrue(tableMappingRule.isMapRTable("table1"));
    assertEquals(tableMappingRule.getMaprTablePath("table1"), new Path("/root/t1"));

    assertTrue(tableMappingRule.isMapRTable("table2"));
    assertEquals(tableMappingRule.getMaprTablePath("table2"), new Path("/user/abc/r/table2"));

    assertTrue(tableMappingRule.isMapRTable("table123"));
    assertEquals(tableMappingRule.getMaprTablePath("table123"), new Path("/p/q/table123"));

    assertTrue(tableMappingRule.isMapRTable("table23"));
    assertEquals(tableMappingRule.getMaprTablePath("table23"), new Path("/x/y/table23"));

    assertTrue(tableMappingRule.isMapRTable("aditya1"));
    assertEquals(tableMappingRule.getMaprTablePath("aditya1"), 
        new Path("maprfs:/mapr/cluster1/user/aditya/default"));

    assertTrue(tableMappingRule.isMapRTable("aditya2"));
    assertEquals(tableMappingRule.getMaprTablePath("aditya2"), 
        new Path("maprfs:/mapr/cluster2/user/aditya/default"));

    assertTrue(tableMappingRule.isMapRTable("aditya"));
    assertEquals(tableMappingRule.getMaprTablePath("aditya"), 
        new Path("maprfs:/mapr/cluster3/user/aditya/aditya"));

    assertFalse(tableMappingRule.isMapRTable("tabl1"));
    assertNull(tableMappingRule.getMaprTablePath("tabl1"));
  }

  @Test
  public void testDirectMappings() throws IOException {
    conf_.set(MapRTableMappingRules.HBASE_TABLE_NAMESPACE_MAPPINGS,
        "table1:a/b, table2:/user/abc/r/s, table1*:p/q, table*:/x/y");
    BaseTableMappingRules tableMappingRule = TableMappingRulesFactory.create(conf_);

    checkAlwaysMaprTables(tableMappingRule);

    assertTrue(tableMappingRule.isMapRTable("table1"));
    assertEquals(tableMappingRule.getMaprTablePath("table1"), new Path(home_, "a/b"));

    assertTrue(tableMappingRule.isMapRTable("table2"));
    assertEquals(tableMappingRule.getMaprTablePath("table2"), new Path("/user/abc/r/s"));

    assertTrue(tableMappingRule.isMapRTable("table123"));
    assertEquals(tableMappingRule.getMaprTablePath("table123"), new Path(home_, "p/q/table123"));

    assertTrue(tableMappingRule.isMapRTable("table23"));
    assertEquals(tableMappingRule.getMaprTablePath("table23"), new Path("/x/y/table23"));

    assertFalse(tableMappingRule.isMapRTable("tabl1"));
    assertNull(tableMappingRule.getMaprTablePath("tabl1"));
  }

  @Test
  public void testMappedRelativePaths() throws IOException {
    conf_.set(MapRTableMappingRules.HBASE_TABLE_NAMESPACE_MAPPINGS,
        "table1:a/b/v/, table1*:p/q/r/, table*:x/y/");
    BaseTableMappingRules tableMappingRule = TableMappingRulesFactory.create(conf_);

    checkAlwaysMaprTables(tableMappingRule);

    assertTrue(tableMappingRule.isMapRTable("table1"));
    assertEquals(tableMappingRule.getMaprTablePath("table1"), new Path(home_, "a/b/v/table1"));

    assertTrue(tableMappingRule.isMapRTable("table123"));
    assertEquals(tableMappingRule.getMaprTablePath("table123"), new Path(home_, "p/q/r/table123"));

    assertTrue(tableMappingRule.isMapRTable("table23"));
    assertEquals(tableMappingRule.getMaprTablePath("table23"), new Path(home_, "x/y/table23"));

    assertFalse(tableMappingRule.isMapRTable("atable1"));
    assertNull(tableMappingRule.getMaprTablePath("atable1"));
  }

  protected void checkAlwaysMaprTables(BaseTableMappingRules tableMappingRule) throws IOException {
    assertTrue(tableMappingRule.isMapRTable("/a/b/c/d"));
    assertEquals(tableMappingRule.getMaprTablePath("/a/b/c/d"), new Path("/a/b/c/d"));

    assertTrue(tableMappingRule.isMapRTable("b/c/d"));
    assertEquals(tableMappingRule.getMaprTablePath("b/c/d"), new Path(home_, "b/c/d"));

    assertTrue(tableMappingRule.isMapRTable("maprfs:///x/y/z"));
    assertEquals(tableMappingRule.getMaprTablePath("maprfs:///x/y/z"),
        new Path("maprfs:///x/y/z"));
  }
}
