/* Copyright (c) 2012 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.fs.hbase.test;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Random;

import org.apache.commons.codec.binary.Hex;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.client.Append;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.Increment;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.client.Row;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Pair;

import com.mapr.fs.MapRFileSystem;


public class DBTest {

   public static void usage() {
    System.err.println("DBTest -rows <1000> -keysize <10 bytes> -families <3> -columns <3> -valuesize <10 bytes> -threads <5> -keytype <random/mono> -batch <1> -debug -test <put/get/scan/mapred/increment/checkandput/append/delete/batch/rowmutations> -tablename maprfs:///myhtable -createtable -printrow");
    System.exit(1);
  }

  public static void formatkey(byte []key, int row) {
    int d = Math.abs(row);
    for (int i = key.length - 1; i >= 0; i--) {
      key[i] = (byte)((d % 10) + '0');
      d /= 10;
    }
  }

  public static int rows = 1000, keysize=10, columns=3, valuesize=10, batch=1, families=3;
  public static long ops;
  public static int keytype = 0; //0  random , 1 = mono
  public static int test = 0;
  public static String tablename = null;
  public static boolean createtable = false, printrow = false;

  public static  class StatsThread extends Thread {
    long time = 0;
    long lastops = 0;
    boolean shutdown = false;

    public StatsThread() {
    }

    @Override
    public void run() {
      while (!shutdown) {
        long diff = ops - lastops;
        System.out.println(time + ": #ops= " + diff);
        lastops = ops;
        ++time;
        try {
          Thread.sleep(1000);
        } catch (InterruptedException ie) {
        }
      }
    }

    private void shutdown() {
      shutdown = true;
    }
  }


  public static void main(String[] args) throws IOException {
    Configuration config = new Configuration();

    config.set("mapr.htable.impl","com.mapr.fs.MapRHTable");

    for (int i = 0; i < args.length; ++i) {
      if (args[i].equals("-rows")) {
        i++;
        if (i >= args.length) usage();
        rows = Integer.parseInt(args[i]);
      } else if (args[i].equals("-keysize")) {
        i++;
        if (i >= args.length) usage();
        keysize = Integer.parseInt(args[i]);
      } else if (args[i].equals("-tablename")) {
        i++;
        if (i >= args.length) usage();
        tablename = args[i];
      } else if (args[i].equals("-debug")) {
        config.set("fs.mapr.trace", "debug");
      } else if (args[i].equals("-createtable")) {
        createtable = true;
      } else if (args[i].equals("-printrow")) {
        printrow = true;
      } else if (args[i].equals("-columns")) {
        i++;
        if (i >= args.length) usage();
        columns = Integer.parseInt(args[i]);
      } else if (args[i].equals("-families")) {
        i++;
        if (i >= args.length) usage();
        families = Integer.parseInt(args[i]);
      } else if (args[i].equals("-valuesize")) {
        i++;
        if (i >= args.length) usage();
        valuesize = Integer.parseInt(args[i]);
      } else if (args[i].equals("-batch")) {
        i++;
        if (i >= args.length) usage();
        batch = Integer.parseInt(args[i]);
      } else if (args[i].equals("-keytype")) {
        i++;
        if (i >= args.length) usage();
        if (args[i].equals("random")) {
          keytype = 0;
        } else if (args[i].equals("mono")) {
          keytype = 1;
        }
      } else if (args[i].equals("-test")) {
        i++;
        if (i >= args.length) usage();
        if (args[i].equals("put")) {
          test = 0;
        } else if (args[i].equals("get")) {
          test = 1;
        } else if (args[i].equals("scan")) {
          test = 2;
        } else if (args[i].equals("mapred")) {
          test = 3;
        } else if (args[i].equals("increment")) {
          test = 4;        
        } else if (args[i].equals("append")) {
          test = 5;        
        } else if (args[i].equals("checkandput")) {
          test = 6;
        } else if (args[i].equals("checkanddelete")) {
          test = 7;
        } else if (args[i].equals("delete")) {
          test = 8;
        } else if(args[i].equals("batch")) {
          test = 9;
        } else if(args[i].equals("rowmutations")) {
          test = 10;
        }
      } else {
        usage();
      }
    }
    if (tablename == null) {
      tablename = new String("maprfs:///mytable");
    }
    if (createtable) {
      System.out.println("creating table " + tablename);
      try {
        Path tablePath = new Path(tablename);
        MapRFileSystem maprfs = new MapRFileSystem();
        maprfs.initialize(tablePath.toUri(), config);
        maprfs.createTable(tablePath);
        System.out.println("created table " + tablename);
        maprfs.close();
      } catch (IOException ioe) {
        System.out.println("failed to create table " + tablename);
        System.exit(1);
      }
    }
    switch (test) {
      case 0:
        puttest(config);
        break;
      case 1:
        gettest(config);
        break;
      case 2:
        scantest(config);
        break;
      case 3:
        testmapred(config);
        break;
      case 4:
        testincrement(config);
        break;
      case 5:
        testappend(config);
        break;
      case 6:
        testcheckandput(config);
        break;
      case 7:
        testcheckanddelete(config);
        break;
      case 8:
        testdelete(config);
        break;
      case 9:
        testbatch(config);
        break;
      case 10:
        //testrowmutations(config);
        break;
      default:
        break;
    }

  }

  public static void printstartend(String msg , byte [] start, byte[] end) {
    String startKey = new String(Hex.encodeHex(start));
    String endKey = new String(Hex.encodeHex(end));
    if (startKey.equals("")) {
      startKey = "-INFINITY";
    }
    if (endKey.equals("")) {
      endKey = "INFINITY";
    }
    System.out.println(msg  + " [" + startKey + "] -> [" + endKey + "]");
  }

  public static void testmapred(Configuration config) {
    HTable table = null;
     try {
       table = new HTable(config, tablename);       
       NavigableMap<HRegionInfo, ServerName> regions = table.getRegionLocations();
       Pair<byte[][], byte[][]> keys = table.getStartEndKeys();
       int i = 0;
       for (Map.Entry<HRegionInfo, ServerName> e: regions.entrySet()) {
         System.out.println("--------------------- [" + i + "] --------------------");
         HRegionInfo hri = e.getKey();
         printstartend("main ", hri.getStartKey(), hri.getEndKey());
         //ServerName sn = e.getValue();
         //System.out.println(" ServerName = " + sn.toString());
         // check 1
         if (!Arrays.equals(keys.getFirst()[i], hri.getStartKey())) {
           String k1 = new String(keys.getFirst()[i]);
           String k2 = new String(hri.getStartKey());
           System.out.println("ERROR1 start keys do not match k1=[" + k1 + "], k2=[" + k2 + "]");
         }
         // check 2
         if (!Arrays.equals(keys.getSecond()[i], hri.getEndKey())) {
           String k1 = new String(keys.getSecond()[i]);
           String k2 = new String(hri.getEndKey());
           System.out.println("ERROR2 start keys do not match k1=[" + k1 + "], k2=[" + k2 + "]");
         }

         HRegionLocation loc;

         // check 3
         loc = table.getRegionLocation(hri.getStartKey());
         //System.out.println(" Check 3 : HRegionLocation = " + loc.toString());
         printstartend("Check 3", loc.getRegionInfo().getStartKey(), loc.getRegionInfo().getEndKey());
         if (!loc.getRegionInfo().equals(hri)) {
           //System.out.println("ERROR3 HRegionInfo not same loc.getRegionInfo() = " + loc.getRegionInfo().toString());
         }

         // check 4
         byte [] key = new byte [hri.getStartKey().length];
         System.arraycopy(hri.getStartKey(), 0,  key, 0,  hri.getStartKey().length);
         if (key.length > 0) {
           key[key.length - 1]++;
         } else {
           key = new byte[1];
           key[0] = 1;
         }
         loc = table.getRegionLocation(key);
         printstartend("Check 4", loc.getRegionInfo().getStartKey(), loc.getRegionInfo().getEndKey());
         //System.out.println(" Check 4 : HRegionLocation = " + loc.toString());
         if (!loc.getRegionInfo().equals(hri)) {
           //System.out.println("ERROR4 HRegionInfo not same loc.getRegionInfo() = " + loc.getRegionInfo().toString()); 
         }

         if (hri.getEndKey().length != 0) {
           // check 5
           loc = table.getRegionLocation(hri.getEndKey());
           //System.out.println(" Check 5 : HRegionLocation = " + loc.toString());
           printstartend("Check 5", loc.getRegionInfo().getStartKey(), loc.getRegionInfo().getEndKey());
           if (loc.getRegionInfo().equals(hri)) {
             //System.out.println("ERROR5 HRegionInfo are same loc.getRegionInfo() = " + loc.getRegionInfo().toString());
           }
         }
         i++;
       }
       int j = 0;
       String nkey = "";
       while (j < 10) {
         HRegionLocation loc1 = table.getRegionLocation(nkey);
         if (loc1 != null) {
          printstartend("region:" + j, loc1.getRegionInfo().getStartKey(), loc1.getRegionInfo().getEndKey());
          nkey = Bytes.toString(loc1.getRegionInfo().getEndKey());
          if (loc1.getRegionInfo().getEndKey().length == 0) break;
         } else {
           break;
         }
         j++;
       }
     } catch (IOException ioe) {
       System.err.println("error " + ioe);
     } finally {
       if (table != null) {
         try {
          table.close();
        } catch (IOException e) {}
       }
     }
  }


  public static void puttest(Configuration config) {
    StatsThread statsThread = new StatsThread();
    statsThread.setDaemon(true);
    statsThread.start();

    byte [][] cfamily = new byte[families][];

    for (int f =0; f < families; ++f) {
      String fname =  "CF-" + Integer.toString(f);
      cfamily[f] = Bytes.toBytes(fname);
    }

    // column names
    byte [][]cnames = new byte[columns][];
    for (int j = 0; j < columns; j++) {
      String cname = "Column-" + Integer.toString(j);
      cnames[j] = Bytes.toBytes(cname);
    }

    // initialize random no
    Random randomSeed = new Random(System.currentTimeMillis());
    Random random = new Random(randomSeed.nextLong());

    // randomize values
    byte [][] values = new byte[1000][];
    for (int i =0 ; i < values.length; i++) {
      values[i] = new byte[valuesize];
      random.nextBytes(values[i]);
    }

    // init key
    byte [] key = new byte[keysize];
    Date date = new Date();

    System.out.println(date.toString() + " START [rows: " + rows +
        ", families: " + families +
        ", columns: " + columns +
        ", keysize: " + keysize +
        ", keytype: " + keytype +
        ", valuesize " + valuesize +
        "]");
    long stime = System.currentTimeMillis();
    try {
      HTable table = new HTable(config, tablename);
      // single row puts
      for (int i = 0; i < rows; i++) {
        if (keytype == 0) {
          random.nextBytes(key);
        } else {
          // +1 to key
          formatkey(key, i);
        }
        Put p = new Put(key);
        for (int f =0 ; f< families; ++f) {
          for (int j = 0; j < columns; ++j) {
            p.add(cfamily[f], cnames[j], values[Math.abs(random.nextInt()) % values.length]);
          }
        }
        ++ops;
        table.put(p);
      }
      System.err.println("closing the table");
      table.flushCommits();
      table.close();
      System.err.println("done closing the table");
    } catch (IOException ioe) {
      System.err.println("Error : " + ioe);
    } finally {
      long time = System.currentTimeMillis() - stime;
      System.out.println("END rows " + rows + " time taken = " + time);
      statsThread.shutdown();
      statsThread.interrupt();
    }
  }

  public static void gettest(Configuration config) {
    Configuration cc = HBaseConfiguration.create();
    byte [][] cfamily = new byte[families][];

    for (int f =0; f < families; ++f) {
      String fname =  "CF-" + Integer.toString(f);
      cfamily[f] = Bytes.toBytes(fname);
    }

    // column names
    byte [][]cnames = new byte[columns][];
    for (int j = 0; j < columns; j++) {
      String cname = "Column-" + Integer.toString(j);
      cnames[j] = Bytes.toBytes(cname);
    }

    // initialize random no
    Random randomSeed = new Random(System.currentTimeMillis());
    Random random = new Random(randomSeed.nextLong());

    // randomize values
    byte [][] values = new byte[1000][];
    for (int i =0 ; i < values.length; i++) {
      values[i] = new byte[valuesize];
      formatkey(values[i], i);
    }

    // init key
    byte [][] key = new byte[rows][];

    System.out.println("inserting [rows: " + rows +
        ", columns: " + columns +
        ", keysize: " + keysize +
        ", keytype: " + keytype +
        ", valuesize " + valuesize +
        "]");
    long stime = System.currentTimeMillis();
    try {
      HTable table = new HTable(config, tablename);
      // single row puts
      for (int i = 0; i < rows; i++) {
        key[i] = new byte[keysize];
        formatkey(key[i], i);
        Put p = new Put(key[i]);
        for (int f =0; f < families; ++f) {
          for (int j = 0; j < columns; j++) {
            p.add(cfamily[f], cnames[j], values[ (i+j) % values.length]);
          }
        }
        table.put(p);
        //table.flushCommits();
      }
      table.close();
    } catch (IOException ioe) {
      System.err.println("error : " + ioe);
    } finally {
      long time = System.currentTimeMillis() - stime;
      System.out.println("END rows " + rows + " time taken = " + time);
    }

    System.out.println("reading..keys back");
    try {
      HTable table = new HTable(config, tablename);
      // single row get
      for (int i = 0; i < rows;) {
        if (batch == 1 ) {
          Get g = new Get(key[i]);
          Result result = table.get(g);
          verify(result, key, values, i, cfamily, cnames);
          i++;
        } else {
          //int r = Math.abs(random.nextInt() % batch);
          //r++;
          int r = batch;
          if (i + r > rows) r = rows - i;
          List<Get> gets = new ArrayList<Get>(r);
          for (int j =0 ; j < r; j++) {
            Get g = new Get(key[i + j]);
            gets.add(g);
          }
          Result[] results =  table.get(gets);
          if (results.length != r) {
            System.out.println("Incorrect result size ased for "
                               + r + " got " + results.length);
            System.exit(1);
          }
          //AH expect it to come in sorted order since keys are mono
          for (int j =0 ; j < r; j++) {
            verify(results[j], key, values, i + j, cfamily, cnames);
          }
          i += r;
        }
      }
      System.out.println("Closing table");
      table.close();
      System.out.println("done closing table");
    } catch (IOException ioe) {
      System.err.println("error : " + ioe);
    } finally {
    }
  }

  public static void verify(Result result, byte[][] key, byte[][] values, int i,
              byte [][]cfamily, byte [][]cnames) {
    String orig = new String(key[i]);
    if (!result.isEmpty()){
      String got = new String(result.getRow());
      // compare keys
      if (Arrays.equals(key[i], result.getRow())) {
        for (int f = 0; f < families; ++f) {
          // compare values
          for (int j = 0; j < columns; j++) {
            byte[] val =  result.getValue(cfamily[f], cnames[j]);
            if (!Arrays.equals(val, values[(i + j) % values.length])) {
              String valstr = "NULL";
              if (val != null) {
                valstr =  new String(val);
              }
              String ovalstr = new String(values[(i+j) % values.length]);
              System.err.println(i + " : ERROR wrong VALUE @" + j +
                                 " key= " + got +
                                 " orig val =" + ovalstr + " got = " + valstr +
                                 " ROW: " + result.toString());
              System.exit(1);
            }
          }
        }
        System.out.println(i + " : SUCCESS ROW: " + result.toString());
      }  else {
        System.out.println(i + " : ERROR wrong key orig = " + orig + " got = " + got +
            " ROW: " + result.toString());
        System.exit(1);
      }
    } else {
      System.out.println(i+ " : ERROR empty result orig = " + orig + " got = EMPTY");
      System.exit(1);
    }
  }

  public static void scantest(Configuration config) {
    byte [][] cfamily = new byte[families][];

    for (int f =0; f < families; ++f) {
      String fname =  "CF-" + Integer.toString(f);
      cfamily[f] = Bytes.toBytes(fname);
    }

    // column names
    byte [][]cnames = new byte[columns][];
    for (int j = 0; j < columns; j++) {
      String cname = "Column-" + Integer.toString(j);
      cnames[j] = Bytes.toBytes(cname);
    }

    try {
      HTable table = new HTable(config, tablename);
      Scan scan = new Scan(HConstants.EMPTY_START_ROW);
      ResultScanner scanner = table.getScanner(scan);
      // single row scan
      for (int i = 0; i < rows;) {
        Result []results = scanner.next(batch);
        if (results == null) break;
        System.out.println("batch : got  " + results.length + " rows");
        for (int r =0; r < results.length; ++r) {
          Result result = results[r];
          int ir = i + r;
          System.out.println(ir + " : SUCCESS ROW: " + result.toString());
          if (printrow) {
            for (int f =0; f < families; ++f) {
              for (int j = 0; j < columns; j++) {
                String col = new String(cnames[j]);
                byte[] val =  result.getValue(cfamily[f], cnames[j]);
                if (val != null) {
                  String v = new String(val);
                  System.out.println("col: " + j + " " + col + " val " + v);
                } else {
                  System.out.println("col: " + j + " " + col + " val NULL *********");
                }
              }
            }
          }
        }
        i+= results.length;
      }
      scanner.close();
      table.close();
    } catch (IOException ioe) {
      System.err.println("error : " + ioe);
    }
  }
  
  // taken from hbase test
  public static void testincrement(Configuration config) {
    byte [][] cfamily = new byte[families][];
    try {
    HTable ht = new HTable(config, tablename);
    
    for (int f =0; f < families; ++f) {
      String fname =  "CF-" + Integer.toString(f);
      cfamily[f] = Bytes.toBytes(fname);
    }    

    byte [][] ROWS = new byte [][] {
        Bytes.toBytes("a"), Bytes.toBytes("b"), Bytes.toBytes("c"),
        Bytes.toBytes("d"), Bytes.toBytes("e"), Bytes.toBytes("f"),
        Bytes.toBytes("g"), Bytes.toBytes("h"), Bytes.toBytes("i")
    };
    byte [][] QUALIFIERS = new byte [][] {
        Bytes.toBytes("a"), Bytes.toBytes("b"), Bytes.toBytes("c"),
        Bytes.toBytes("d"), Bytes.toBytes("e"), Bytes.toBytes("f"),
        Bytes.toBytes("g"), Bytes.toBytes("h"), Bytes.toBytes("i")
    };
    
    byte [] FAMILY = cfamily[0];
    byte []ROW = ROWS[0];

    System.out.println("PASS 1");
    long newval;
    
    // Do some simple single-column increments    
    // First with old API
    newval = ht.incrementColumnValue(ROW, FAMILY, QUALIFIERS[0], 1);
    if (newval != 1) {
      System.out.println("bad ret value 1");
    }
    newval = ht.incrementColumnValue(ROW, FAMILY, QUALIFIERS[1], 2);
    if (newval != 2) {
      System.out.println("bad ret value 2");
    }
    newval = ht.incrementColumnValue(ROW, FAMILY, QUALIFIERS[2], 3);
    if (newval != 3) {
      System.out.println("bad ret value 3");
    }
    newval = ht.incrementColumnValue(ROW, FAMILY, QUALIFIERS[3], 4);
    if (newval != 4) {
      System.out.println("bad ret value 4");
    }
    // Verify expected results
    Result r = ht.get(new Get(ROW));
    KeyValue [] kvs = r.raw();
    if (kvs.length != 4) {
      System.err.println("wrong kvs length " + kvs.length + " expected 4");
     System.exit(1);
    }
    assertIncrementKey(kvs[0], ROW, FAMILY, QUALIFIERS[0], 1);
    assertIncrementKey(kvs[1], ROW, FAMILY, QUALIFIERS[1], 2);
    assertIncrementKey(kvs[2], ROW, FAMILY, QUALIFIERS[2], 3);
    assertIncrementKey(kvs[3], ROW, FAMILY, QUALIFIERS[3], 4);
    
    System.out.println("PASS 2");
    // Now increment things incremented with old and do some new
    Increment inc = new Increment(ROW);
    inc.addColumn(FAMILY, QUALIFIERS[1], 1);
    inc.addColumn(FAMILY, QUALIFIERS[3], 1);
    inc.addColumn(FAMILY, QUALIFIERS[4], 1);
    ht.increment(inc);
    System.out.println("PASS 3");
    
    // Verify expected results
    r = ht.get(new Get(ROW));
    kvs = r.raw();
    if (kvs.length != 5) {
      System.err.println("wrong kvs length " + kvs.length + " expected 5");
     System.exit(1);
    }
    System.out.println("PASS 4");
    assertIncrementKey(kvs[0], ROW, FAMILY, QUALIFIERS[0], 1);
    assertIncrementKey(kvs[1], ROW, FAMILY, QUALIFIERS[1], 3);
    assertIncrementKey(kvs[2], ROW, FAMILY, QUALIFIERS[2], 3);
    assertIncrementKey(kvs[3], ROW, FAMILY, QUALIFIERS[3], 5);
    assertIncrementKey(kvs[4], ROW, FAMILY, QUALIFIERS[4], 1);
    System.out.println("PASS 5");

    // Now try multiple columns by different amounts
    inc = new Increment(ROWS[1]);
    for (int i=0;i<QUALIFIERS.length;i++) {
      inc.addColumn(FAMILY, QUALIFIERS[i], i+1);
    }
    ht.increment(inc);
    System.out.println("PASS 6");
    // Verify
    r = ht.get(new Get(ROWS[1]));
    kvs = r.raw();
    if (QUALIFIERS.length != kvs.length) {
      System.err.println("wrong kvs length 1 " + kvs.length + " expected " + QUALIFIERS.length);
    }
    for (int i=0;i<QUALIFIERS.length;i++) {
      assertIncrementKey(kvs[i], ROWS[1], FAMILY, QUALIFIERS[i], i+1);
    }
    
    System.out.println("PASS 7");
    // Re-increment them
    inc = new Increment(ROWS[1]);
    for (int i=0;i<QUALIFIERS.length;i++) {
      inc.addColumn(FAMILY, QUALIFIERS[i], i+1);
    }
    ht.increment(inc);
    System.out.println("PASS 8");
    // Verify
    r = ht.get(new Get(ROWS[1]));
    kvs = r.raw();
    if (QUALIFIERS.length != kvs.length) {
      System.err.println("wrong kvs length 2 " + kvs.length + " expected " + QUALIFIERS.length);
    }
    for (int i=0;i<QUALIFIERS.length;i++) {
      assertIncrementKey(kvs[i], ROWS[1], FAMILY, QUALIFIERS[i], 2*(i+1));
    }
    System.out.println("PASS 9");
    } catch (Exception ioe) {
       System.err.println(" error ioe "+  ioe);
    }
  }

  public static void assertIncrementKey(KeyValue key, byte [] row, byte [] family,
      byte [] qualifier, long value)
  throws Exception {
    if(!Arrays.equals(row, key.getRow())) {
      System.err.println("Expected row [" + Bytes.toString(row) + "] " +
          "Got row [" + Bytes.toString(key.getRow()) +"]");
     System.exit(1);
    }
    if (!Arrays.equals(family, key.getFamily())) {
      System.err.println("Expected family [" + Bytes.toString(family) + "] " +
          "Got family [" + Bytes.toString(key.getFamily()) + "]");
     System.exit(1);
    }
    if (!Arrays.equals(qualifier, key.getQualifier())) {
      System.err.println("Expected qualifier [" + Bytes.toString(qualifier) + "] " +
          "Got qualifier [" + Bytes.toString(key.getQualifier()) + "]");
     System.exit(1);
    }
    if (Bytes.toLong(key.getValue()) != value) {
      System.err.println("Expected value [" + value + "] " + "Got value [" + 
          Bytes.toLong(key.getValue()) + "]");
     System.exit(1);
    }
  }
  
  // taken from hbase test
  public static void testcheckandput(Configuration config) {
    byte [][] cfamily = new byte[families][];
    try {
    HTable ht = new HTable(config, tablename);
    
    for (int f =0; f < families; ++f) {
      String fname =  "CF-" + Integer.toString(f);
      cfamily[f] = Bytes.toBytes(fname);
    }    

    byte [][] ROWS = new byte [][] {
        Bytes.toBytes("a"), Bytes.toBytes("b"), Bytes.toBytes("c"),
        Bytes.toBytes("d"), Bytes.toBytes("e"), Bytes.toBytes("f"),
        Bytes.toBytes("g"), Bytes.toBytes("h"), Bytes.toBytes("i")
    };
    byte [][] QUALIFIERS = new byte [][] {
        Bytes.toBytes("a"), Bytes.toBytes("b"), Bytes.toBytes("c"),
        Bytes.toBytes("d"), Bytes.toBytes("e"), Bytes.toBytes("f"),
        Bytes.toBytes("g"), Bytes.toBytes("h"), Bytes.toBytes("i")
    };
    
    byte [] FAMILY = cfamily[0];
    byte []ROW = ROWS[0];


    final byte [] anotherrow = Bytes.toBytes("anotherrow");
    final byte [] value1 = Bytes.toBytes("abcd1");
    final byte [] value2 = Bytes.toBytes("abcd2");
    
    Put put1 = new Put(ROW);
    put1.add(FAMILY, QUALIFIERS[0], value1);
    // row doesn't exist, so using non-null value should be considered "not match".
    boolean ok = ht.checkAndPut(ROW, FAMILY, QUALIFIERS[0], value1, put1);
    if (ok != false) {
       System.err.println("Failed PASS 1");
       System.exit(1);
    } else {
      System.out.println("PASS 1");
    }

    // row doesn't exist, so using "null" to check for existence should be considered "match".
    ok = ht.checkAndPut(ROW, FAMILY, QUALIFIERS[0], null, put1);
    if (ok == false) {
       System.err.println("Failed PASS 2");
       System.exit(1);
    } else {
      System.out.println("PASS 2");
    }    
    
    // row now exists, so using "null" to check for existence should be considered "not match".
    ok = ht.checkAndPut(ROW, FAMILY, QUALIFIERS[0], null, put1);
    if (ok != false) {
       System.err.println("Failed PASS 3");
       System.exit(1);
    } else {
      System.out.println("PASS 3");
    }
    
    Put put2 = new Put(ROW);
    put2.add(FAMILY, QUALIFIERS[0], value2);
    // row now exists, use the matching value to check
    ok = ht.checkAndPut(ROW, FAMILY, QUALIFIERS[0], value1, put2);
    if (ok == false) {
       System.err.println("Failed PASS 4");
       System.exit(1);
    } else {
      System.out.println("PASS 4");
    }
        
    Put put3 = new Put(anotherrow);
    put3.add(FAMILY, QUALIFIERS[0], value1);
    
    // try to do CheckAndPut on different rows
    try {
      ok = ht.checkAndPut(ROW, FAMILY, QUALIFIERS[0], value2, put3);
      System.err.println("trying to check and modify different rows should have failed.");
      System.exit(1);
    } catch(Exception e) {}
    System.out.println("DONE");
    } catch (Exception e) {
      System.err.println("error " + e);
    }
  }

  // taken from hbase test
  public static void testcheckanddelete(Configuration config) {
  }
 
  public static byte [][] makeNAscii(byte [] base, int n) {
    if(n > 256) {
      return makeNBig(base, n);
    }
    byte [][] ret = new byte[n][];
    for(int i=0;i<n;i++) {
      byte [] tail = Bytes.toBytes(Integer.toString(i));
      ret[i] = Bytes.add(base, tail);
    }
    return ret;
  }
  
  public static byte [][] makeN(byte [] base, int n) {
    if (n > 256) {
      return makeNBig(base, n);
    }
    byte [][] ret = new byte[n][];
    for(int i=0;i<n;i++) {
      ret[i] = Bytes.add(base, new byte[]{(byte)i});
    }
    return ret;
  }
  
  public static byte [][] makeNBig(byte [] base, int n) {
    byte [][] ret = new byte[n][];
    for(int i=0;i<n;i++) {
      int byteA = (i % 256);
      int byteB = (i >> 8);
      ret[i] = Bytes.add(base, new byte[]{(byte)byteB,(byte)byteA});
    }
    return ret;
  }

  public static long [] makeStamps(int n) {
    long [] stamps = new long[n];
    for(int i=0;i<n;i++) stamps[i] = i+1;
    return stamps;
  }

  public static boolean equals(byte [] left, byte [] right) {
    if (left == null && right == null) return true;
    if (left == null && right.length == 0) return true;
    if (right == null && left.length == 0) return true;
    return Bytes.equals(left, right);
  }

  public static void assertTrue(String arg, boolean condition) throws IOException {
    if (condition != true) {
      System.err.println("assert failed msg = " + arg);
      throw new IOException(arg);
    }
  }

  public static void assertTrue(boolean condition) throws IOException {
    assertTrue("empty msg", condition);
  }

  public static  void assertNumKeys(Result result, int n) throws Exception {
    assertTrue("Expected " + n + " keys but got " + result.size(),
              result.size() == n);
  }

  public static void assertNResult(Result result, byte [] row,
                             byte [][] families, byte [][] qualifiers, byte [][] values,
                             int [][] idxs)
  throws Exception {
    assertTrue("Expected row [" + Bytes.toString(row) + "] " +
        "Got row [" + Bytes.toString(result.getRow()) +"]",
        equals(row, result.getRow()));
    assertTrue("Expected " + idxs.length + " keys but result contains "
        + result.size(), result.size() == idxs.length);
    
    KeyValue [] keys = result.raw();
    
    for(int i=0;i<keys.length;i++) {
      byte [] family = families[idxs[i][0]];
      byte [] qualifier = qualifiers[idxs[i][1]];
      byte [] value = values[idxs[i][2]];
      KeyValue key = keys[i];
      
      assertTrue("(" + i + ") Expected family [" + Bytes.toString(family)
          + "] " + "Got family [" + Bytes.toString(key.getFamily()) + "]",
          equals(family, key.getFamily()));
      assertTrue("(" + i + ") Expected qualifier [" + Bytes.toString(qualifier)
          + "] " + "Got qualifier [" + Bytes.toString(key.getQualifier()) + "]",
          equals(qualifier, key.getQualifier()));
      assertTrue("(" + i + ") Expected value [" + Bytes.toString(value) + "] "
          + "Got value [" + Bytes.toString(key.getValue()) + "]",
          equals(value, key.getValue()));
    }
  }

 public static  void assertNResult(Result result, byte [] row,
     byte [] family, byte [] qualifier, long [] stamps, byte [][] values,
     int start, int end)
 throws IOException {
   assertTrue("Expected row [" + Bytes.toString(row) + "] " +
       "Got row [" + Bytes.toString(result.getRow()) +"]",
       equals(row, result.getRow()));
   int expectedResults = end - start + 1;
   assertEquals(expectedResults, result.size());
   
   KeyValue [] keys = result.raw();
   
   for (int i=0; i<keys.length; i++) {
     byte [] value = values[end-i];                                          
     long ts = stamps[end-i];
     KeyValue key = keys[i];
     
     assertTrue("(" + i + ") Expected family [" + Bytes.toString(family)
         + "] " + "Got family [" + Bytes.toString(key.getFamily()) + "]",
         equals(family, key.getFamily()));
     assertTrue("(" + i + ") Expected qualifier [" + Bytes.toString(qualifier)
         + "] " + "Got qualifier [" + Bytes.toString(key.getQualifier()) + "]",
         equals(qualifier, key.getQualifier()));
     assertTrue("Expected ts [" + ts + "] " +
         "Got ts [" + key.getTimestamp() + "]", ts == key.getTimestamp());
     assertTrue("(" + i + ") Expected value [" + Bytes.toString(value) + "] "
         + "Got value [" + Bytes.toString(key.getValue()) + "]",
         equals(value, key.getValue()));
   }
 }

 public static void assertEquals(int i, int j) throws IOException {
   if (i != j) {
     throw new IOException("not equal arg1" + i + " arg2 " + j);
   }
 }

 public static void assertEquals(byte[] expected,
     byte[] actual) throws IOException {
   if (Bytes.compareTo(expected, actual) != 0) {
     throw new IOException("expected:<" +
         Bytes.toStringBinary(expected) + "> but was:<" +
         Bytes.toStringBinary(actual) + ">");
   }
 }

  
  public static Result getSingleScanResult(HTable ht, Scan scan) throws IOException {
    ResultScanner scanner = ht.getScanner(scan);
    Result result = scanner.next();
    scanner.close();
    return result;
  }

  public static void testdelete(Configuration config) {
    byte [][] cfamily = new byte[families][];
    try {
      HTable ht = new HTable(config, tablename);
      for (int f =0; f < families; ++f) {
        String fname =  "CF-" + Integer.toString(f);
        cfamily[f] = Bytes.toBytes(fname);
      }

      byte [] ROW = Bytes.toBytes("testRow");
      byte [] FAMILY = Bytes.toBytes("CF-");
      byte [] QUALIFIER = Bytes.toBytes("testQualifier");
      byte [] VALUE = Bytes.toBytes("testValue");

      byte [][] ROWS = makeNAscii(ROW, 6);
      byte [][] FAMILIES = makeNAscii(FAMILY, 3);
      byte [][] VALUES = makeN(VALUE, 5);
      long [] ts = {1000, 2000, 3000, 4000, 5000};

      Put put = new Put(ROW);
      put.add(FAMILIES[0], QUALIFIER, ts[0], VALUES[0]);
      put.add(FAMILIES[0], QUALIFIER, ts[1], VALUES[1]);
      ht.put(put);
      
      System.out.println("PASS 1");
      Delete delete = new Delete(ROW);
      delete.deleteFamily(FAMILIES[0], ts[0]);
      ht.delete(delete);
      
      System.out.println("PASS 2");
      Get get = new Get(ROW);
      get.addFamily(FAMILIES[0]);
      get.setMaxVersions(Integer.MAX_VALUE);
      Result result = ht.get(get);
      assertNResult(result, ROW, FAMILIES[0], QUALIFIER,
                    new long [] {ts[1]},
                    new byte[][] {VALUES[1]},
                    0, 0);
      System.out.println("PASS 3");
      
      Scan scan = new Scan(ROW);
      scan.addFamily(FAMILIES[0]);
      scan.setMaxVersions(Integer.MAX_VALUE);
      result = getSingleScanResult(ht, scan);
      assertNResult(result, ROW, FAMILIES[0], QUALIFIER,
                    new long [] {ts[1]},
                    new byte[][] {VALUES[1]},
                    0, 0);
      System.out.println("PASS 4");

      
      // Test delete latest version
      put = new Put(ROW);
      put.add(FAMILIES[0], QUALIFIER, ts[4], VALUES[4]);
      put.add(FAMILIES[0], QUALIFIER, ts[2], VALUES[2]);
      put.add(FAMILIES[0], QUALIFIER, ts[3], VALUES[3]);
      put.add(FAMILIES[0], null, ts[4], VALUES[4]);
      put.add(FAMILIES[0], null, ts[2], VALUES[2]);
      put.add(FAMILIES[0], null, ts[3], VALUES[3]);
      ht.put(put);
      
      System.out.println("PASS 5");
      delete = new Delete(ROW);
      delete.deleteColumn(FAMILIES[0], QUALIFIER);
      
      ht.delete(delete);

      System.out.println("PASS 6");
      get = new Get(ROW);
      get.addColumn(FAMILIES[0], QUALIFIER);
      get.setMaxVersions(Integer.MAX_VALUE);
      result = ht.get(get);
      assertNResult(result, ROW, FAMILIES[0], QUALIFIER,
                    new long [] {ts[1], ts[2], ts[3]},
                    new byte[][] {VALUES[1], VALUES[2], VALUES[3]},
                    0, 2);
      System.out.println("PASS 7");

      scan = new Scan(ROW);
      scan.addColumn(FAMILIES[0], QUALIFIER);
      scan.setMaxVersions(Integer.MAX_VALUE);
      result = getSingleScanResult(ht, scan);
      assertNResult(result, ROW, FAMILIES[0], QUALIFIER,
                    new long [] {ts[1], ts[2], ts[3]},                                                              
                    new byte[][] {VALUES[1], VALUES[2], VALUES[3]},
                    0, 2);
      System.out.println("PASS 8");

      // Test for HBASE-1847
      delete = new Delete(ROW);
      delete.deleteColumn(FAMILIES[0], null);
      ht.delete(delete);
      System.out.println("PASS 9");
      
      // Cleanup null qualifier
      delete = new Delete(ROW);
      delete.deleteColumns(FAMILIES[0], null);
      ht.delete(delete);
      System.out.println("PASS 10");
      
      // Expected client behavior might be that you can re-put deleted values
      // But alas, this is not to be.  We can't put them back in either case.
      
      put = new Put(ROW);
      put.add(FAMILIES[0], QUALIFIER, ts[0], VALUES[0]); // 1000
      put.add(FAMILIES[0], QUALIFIER, ts[4], VALUES[4]); // 5000
      ht.put(put);
      System.out.println("PASS 11");

      // It used to be due to the internal implementation of Get, that
      // the Get() call would return ts[4] UNLIKE the Scan below. With
      // the switch to using Scan for Get this is no longer the case.
      get = new Get(ROW);
      get.addFamily(FAMILIES[0]);
      get.setMaxVersions(Integer.MAX_VALUE);
      result = ht.get(get);
      assertNResult(result, ROW, FAMILIES[0], QUALIFIER,
                    new long [] {ts[1], ts[2], ts[3]},
                    new byte[][] {VALUES[1], VALUES[2], VALUES[3]},
                    0, 2);
      System.out.println("PASS 12");
      
      // The Scanner returns the previous values, the expected-naive-unexpected behavior
      
      scan = new Scan(ROW);
      scan.addFamily(FAMILIES[0]);
      scan.setMaxVersions(Integer.MAX_VALUE);
      result = getSingleScanResult(ht, scan);
      assertNResult(result, ROW, FAMILIES[0], QUALIFIER,
                    new long [] {ts[1], ts[2], ts[3]},
                    new byte[][] {VALUES[1], VALUES[2], VALUES[3]},
                    0, 2);
      System.out.println("PASS 13");

      // Test deleting an entire family from one row but not the other various ways
      
      put = new Put(ROWS[0]);
      put.add(FAMILIES[1], QUALIFIER, ts[0], VALUES[0]);
      put.add(FAMILIES[1], QUALIFIER, ts[1], VALUES[1]);
      put.add(FAMILIES[2], QUALIFIER, ts[2], VALUES[2]);
      put.add(FAMILIES[2], QUALIFIER, ts[3], VALUES[3]);
      ht.put(put);
      System.out.println("PASS 14");
      
      put = new Put(ROWS[1]);
      put.add(FAMILIES[1], QUALIFIER, ts[0], VALUES[0]);
      put.add(FAMILIES[1], QUALIFIER, ts[1], VALUES[1]);
      put.add(FAMILIES[2], QUALIFIER, ts[2], VALUES[2]);
      put.add(FAMILIES[2], QUALIFIER, ts[3], VALUES[3]);
      ht.put(put);
      System.out.println("PASS 15");

      put = new Put(ROWS[2]);
      put.add(FAMILIES[1], QUALIFIER, ts[0], VALUES[0]);
      put.add(FAMILIES[1], QUALIFIER, ts[1], VALUES[1]);
      put.add(FAMILIES[2], QUALIFIER, ts[2], VALUES[2]);
      put.add(FAMILIES[2], QUALIFIER, ts[3], VALUES[3]);
      ht.put(put);
      System.out.println("PASS 16");
      
      // Assert that above went in.
      get = new Get(ROWS[2]);
      get.addFamily(FAMILIES[1]);
      get.addFamily(FAMILIES[2]);
      get.setMaxVersions(Integer.MAX_VALUE);
      result = ht.get(get);
      assertTrue("Expected 4 key but received " + result.size() + ": " + result,
          result.size() == 4);
      System.out.println("PASS 17");
 
      delete = new Delete(ROWS[0]);
      delete.deleteFamily(FAMILIES[2]);
      ht.delete(delete);
      System.out.println("PASS 18");
      
      delete = new Delete(ROWS[1]);
      delete.deleteColumns(FAMILIES[1], QUALIFIER);
      ht.delete(delete);
      System.out.println("PASS 19");
      
      delete = new Delete(ROWS[2]);
      delete.deleteColumn(FAMILIES[1], QUALIFIER);
      delete.deleteColumn(FAMILIES[1], QUALIFIER);
      delete.deleteColumn(FAMILIES[2], QUALIFIER);
      ht.delete(delete);
      System.out.println("PASS 20");
      
      get = new Get(ROWS[0]);
      get.addFamily(FAMILIES[1]);
      get.addFamily(FAMILIES[2]);
      get.setMaxVersions(Integer.MAX_VALUE);
      result = ht.get(get);
      assertTrue("Expected 2 keys but received " + result.size(),
          result.size() == 2);
      assertNResult(result, ROWS[0], FAMILIES[1], QUALIFIER,
          new long [] {ts[0], ts[1]},
          new byte[][] {VALUES[0], VALUES[1]},
          0, 1);
      System.out.println("PASS 21");
      
      scan = new Scan(ROWS[0]);
      scan.addFamily(FAMILIES[1]);
      scan.addFamily(FAMILIES[2]);
      scan.setMaxVersions(Integer.MAX_VALUE);
      result = getSingleScanResult(ht, scan);
      assertTrue("Expected 2 keys but received " + result.size(),
          result.size() == 2);
      assertNResult(result, ROWS[0], FAMILIES[1], QUALIFIER,
          new long [] {ts[0], ts[1]},
          new byte[][] {VALUES[0], VALUES[1]},
          0, 1);
      System.out.println("PASS 22");

      get = new Get(ROWS[1]);
      get.addFamily(FAMILIES[1]);
      get.addFamily(FAMILIES[2]);
      get.setMaxVersions(Integer.MAX_VALUE);
      result = ht.get(get);       
      assertTrue("Expected 2 keys but received " + result.size(),
          result.size() == 2);
      System.out.println("PASS 23");

      scan = new Scan(ROWS[1]);
      scan.addFamily(FAMILIES[1]);
      scan.addFamily(FAMILIES[2]);
      scan.setMaxVersions(Integer.MAX_VALUE);
      result = getSingleScanResult(ht, scan);
      assertTrue("Expected 2 keys but received " + result.size(),
          result.size() == 2);
      System.out.println("PASS 24");
      
      get = new Get(ROWS[2]);
      get.addFamily(FAMILIES[1]);
      get.addFamily(FAMILIES[2]);
      get.setMaxVersions(Integer.MAX_VALUE);
      result = ht.get(get);
      assertEquals(1, result.size());
      assertNResult(result, ROWS[2], FAMILIES[2], QUALIFIER,
          new long [] {ts[2]},
          new byte[][] {VALUES[2]},
          0, 0);
      System.out.println("PASS 25");
      
      scan = new Scan(ROWS[2]);
      scan.addFamily(FAMILIES[1]);
      scan.addFamily(FAMILIES[2]);
      scan.setMaxVersions(Integer.MAX_VALUE);
      result = getSingleScanResult(ht, scan);
      assertEquals(1, result.size());
      assertNResult(result, ROWS[2], FAMILIES[2], QUALIFIER,
          new long [] {ts[2]},
          new byte[][] {VALUES[2]},
          0, 0);
      System.out.println("PASS 26");

      // Test if we delete the family first in one row (HBASE-1541)
      delete = new Delete(ROWS[3]);
      delete.deleteFamily(FAMILIES[1]);
      ht.delete(delete);
      
      System.out.println("PASS 27");
      
      put = new Put(ROWS[3]);
      put.add(FAMILIES[2], QUALIFIER, VALUES[0]);
      ht.put(put);
      System.out.println("PASS 28");
      
      put = new Put(ROWS[4]);
      put.add(FAMILIES[1], QUALIFIER, VALUES[1]);
      put.add(FAMILIES[2], QUALIFIER, VALUES[2]);
      ht.put(put);
      System.out.println("PASS 29");
      
      get = new Get(ROWS[3]);
      get.addFamily(FAMILIES[1]);
      get.addFamily(FAMILIES[2]);
      get.setMaxVersions(Integer.MAX_VALUE);
      result = ht.get(get);
      assertTrue("Expected 1 key but received " + result.size(),
          result.size() == 1);
      System.out.println("PASS 30");
      
      get = new Get(ROWS[4]);
      get.addFamily(FAMILIES[1]);
      get.addFamily(FAMILIES[2]);
      get.setMaxVersions(Integer.MAX_VALUE);
      result = ht.get(get);
      assertTrue("Expected 2 keys but received " + result.size(),
          result.size() == 2);
      System.out.println("PASS 31");
      
      scan = new Scan(ROWS[3]);
      scan.addFamily(FAMILIES[1]);     
      scan.addFamily(FAMILIES[2]);
      scan.setMaxVersions(Integer.MAX_VALUE);
      ResultScanner scanner = ht.getScanner(scan);
      result = scanner.next();
      assertTrue("Expected 1 key but received " + result.size(),
          result.size() == 1);
      assertTrue(Bytes.equals(result.raw()[0].getRow(), ROWS[3]));
      assertTrue(Bytes.equals(result.raw()[0].getValue(), VALUES[0]));                                                                                                                                      
      result = scanner.next();
      assertTrue("Expected 2 keys but received " + result.size(),
          result.size() == 2);                                                                                                                                              
      assertTrue(Bytes.equals(result.raw()[0].getRow(), ROWS[4]));      
      assertTrue(Bytes.equals(result.raw()[1].getRow(), ROWS[4]));
      assertTrue(Bytes.equals(result.raw()[0].getValue(), VALUES[1]));
      assertTrue(Bytes.equals(result.raw()[1].getValue(), VALUES[2]));
      scanner.close();
      System.out.println("PASS 32");

      // Add test of bulk deleting.
      for (int i = 0; i < 10; i++) {
        byte [] bytes = Bytes.toBytes(i);
        put = new Put(bytes);
        put.setWriteToWAL(false);
        put.add(FAMILIES[0], QUALIFIER, bytes);
        ht.put(put);
      }
      System.out.println("PASS 33");
      for (int i = 0; i < 10; i++) {
        byte [] bytes = Bytes.toBytes(i);
        get = new Get(bytes);
        get.addFamily(FAMILIES[0]);
        result = ht.get(get);
        assertTrue(result.size() == 1);
      }
      System.out.println("PASS 34");
      ArrayList<Delete> deletes = new ArrayList<Delete>();
      for (int i = 0; i < 10; i++) {
        byte [] bytes = Bytes.toBytes(i);
        delete = new Delete(bytes);
        delete.deleteFamily(FAMILIES[0]);
        deletes.add(delete);
      }
      System.out.println("PASS 35");
      ht.delete(deletes);
      for (int i = 0; i < 10; i++) {
        byte [] bytes = Bytes.toBytes(i);
        get = new Get(bytes);
        get.addFamily(FAMILIES[0]);
        result = ht.get(get);
        assertTrue(result.size() == 0);
      }
      System.out.println("DONE");
    } catch (Exception e) {
      System.err.println("error " + e);
      e.printStackTrace();
      System.exit(1);
    }
  }

  public static void testappend(Configuration config) {
    byte [][] cfamily = new byte[families][];
    try {
      HTable ht = new HTable(config, tablename);
      for (int f =0; f < families; ++f) {
        String fname =  "CF-" + Integer.toString(f);
        cfamily[f] = Bytes.toBytes(fname);
      }
      byte [][] ROWS = new byte [][] {
        Bytes.toBytes("a"), Bytes.toBytes("b"), Bytes.toBytes("c"),
        Bytes.toBytes("d"), Bytes.toBytes("e"), Bytes.toBytes("f"),
        Bytes.toBytes("g"), Bytes.toBytes("h"), Bytes.toBytes("i")
      };

      byte [] FAMILY = cfamily[0];
      byte [] ROW = ROWS[0];
  
      byte[] v1 = Bytes.toBytes("42");
      byte[] v2 = Bytes.toBytes("23");
      byte [][] QUALIFIERS = new byte [][] {
        Bytes.toBytes("a"), Bytes.toBytes("b")
      };

      Append a = new Append(ROW);
      a.add(FAMILY, QUALIFIERS[0], v1);
      a.add(FAMILY, QUALIFIERS[1], v2);
      a.setReturnResults(false);
      if (ht.append(a) != null ) {
        System.err.println("append test pass 1 failed non null return value");
        System.exit(1);
      }
      
      a = new Append(ROW);
      a.add(FAMILY, QUALIFIERS[0], v2);
      a.add(FAMILY, QUALIFIERS[1], v1);
      Result r = ht.append(a);
      if (Bytes.compareTo(Bytes.add(v1,v2), r.getValue(FAMILY, QUALIFIERS[0])) != 0) {
        System.err.println("append test pass 2 failed wrong values expected " + 
                           Bytes.toString(Bytes.add(v1,v2)) + " got " + 
                           Bytes.toString(r.getValue(FAMILY, QUALIFIERS[0]))); 
      }
      if (Bytes.compareTo(Bytes.add(v2,v1), r.getValue(FAMILY, QUALIFIERS[1])) != 0) {
        System.err.println("append test pass 3 failed wrong values expected " + 
                           Bytes.toString(Bytes.add(v2,v1)) + " got " + 
                           Bytes.toString(r.getValue(FAMILY, QUALIFIERS[1]))); 
      }
      System.out.println("DONE");
    } catch (Exception e) {
      System.err.println("error " + e);
      System.exit(1);
    }
  }

  /* from hbase test */
  public static List<Row> constructPutRequests() {
    List<Row> puts = new ArrayList<Row>();
    for (byte[] k : KEYS) {
      Put put = new Put(k);
      put.add(BYTES_FAMILY, QUALIFIER, VALUE);
      puts.add(put);
    }
    return puts;
  }

  public static final byte[] QUALIFIER = Bytes.toBytes("qual");
  public static final byte[] BYTES_FAMILY = Bytes.toBytes("CF-0");
  public static final byte[] VALUE = Bytes.toBytes("value");

  public static final byte[][] GKEYS = {
    HConstants.EMPTY_BYTE_ARRAY, Bytes.toBytes("bbb"),
    Bytes.toBytes("ccc"), Bytes.toBytes("ddd"), Bytes.toBytes("eee"),
    Bytes.toBytes("fff"), Bytes.toBytes("ggg"), Bytes.toBytes("hhh"),
    Bytes.toBytes("iii"), Bytes.toBytes("jjj"), Bytes.toBytes("kkk"),
    Bytes.toBytes("lll"), Bytes.toBytes("mmm"), Bytes.toBytes("nnn"),
    Bytes.toBytes("ooo"), Bytes.toBytes("ppp"), Bytes.toBytes("qqq"),
    Bytes.toBytes("rrr"), Bytes.toBytes("sss"), Bytes.toBytes("ttt"),
    Bytes.toBytes("uuu"), Bytes.toBytes("vvv"), Bytes.toBytes("www"),
    Bytes.toBytes("xxx"), Bytes.toBytes("yyy")
  };

  public static final byte [][] KEYS = makeKeys();
  public static byte[][] makeKeys() {
    byte [][] starterKeys = DBTest.GKEYS;
    // Create a "non-uniform" test set with the following characteristics:
    // a) Unequal number of keys per region
    
    // Don't use integer as a multiple, so that we have a number of keys that is
    // not a multiple of the number of regions
    int numKeys = (int) ((float) starterKeys.length * 10.33F);
    
    List<byte[]> keys = new ArrayList<byte[]>();
    for (int i = 0; i < numKeys; i++) {
      int kIdx = i % starterKeys.length;
      byte[] k = starterKeys[kIdx];
      byte[] cp = new byte[k.length + 1];
      System.arraycopy(k, 0, cp, 0, k.length);
      cp[k.length] = new Integer(i % 256).byteValue();
      keys.add(cp);
    }
    
    // b) Same duplicate keys (showing multiple Gets/Puts to the same row, which
    // should work)
    // c) keys are not in sorted order (within a region), to ensure that the
    // sorting code and index mapping doesn't break the functionality
    for (int i = 0; i < 100; i++) {
      int kIdx = i % starterKeys.length;
      byte[] k = starterKeys[kIdx];
      byte[] cp = new byte[k.length + 1];
      System.arraycopy(k, 0, cp, 0, k.length);
      cp[k.length] = new Integer(i % 256).byteValue();
      keys.add(cp);
    }
    return keys.toArray(new byte [][] {new byte [] {}});
  }

  public static void testbatch(Configuration config) {
    try {
      HTable table = new HTable(config, tablename);
      System.out.println("testbatch: loading data");
      // load test data
      List<Row> puts = constructPutRequests();
      table.batch(puts);
      System.out.println("testbatch: loading data done");
      table.flushCommits();
      
      System.out.println("testbatch: reading data with batch");
      // create a list of gets and run it
      List<Row> gets = new ArrayList<Row>();
      for (byte[] k : KEYS) {
        Get get = new Get(k);
        get.addColumn(BYTES_FAMILY, QUALIFIER);
        gets.add(get);
      }
      Result[] multiRes = new Result[gets.size()];
      table.batch(gets, multiRes);
      System.out.println("testbatch: reading data with batch done");

      System.out.println("testbatch: reading data with single call");
      // Same gets using individual call API
      List<Result> singleRes = new ArrayList<Result>();
      for (Row get : gets) {
        singleRes.add(table.get((Get) get));
      }
      System.out.println("testbatch: reading data with single call done");

      // Compare results
      System.out.println("Check 1");
      assertEquals(singleRes.size(), multiRes.length);
      System.out.println("Check 2 with in the rows " + singleRes.size());
      for (int i = 0; i < singleRes.size(); i++) {
        assertTrue(singleRes.get(i).containsColumn(BYTES_FAMILY, QUALIFIER));
        KeyValue[] singleKvs = singleRes.get(i).raw();
        KeyValue[] multiKvs = multiRes[i].raw();
        for (int j = 0; j < singleKvs.length; j++) {
          if (!singleKvs[j].equals(multiKvs[j])) {
            System.err.println("Error keyvalues are different row " + i +
                               " keyvalue from single call " + singleKvs[j].toString() + 
                               " keyvalue from batch call " + multiRes[j].toString()); 
            System.exit(1);
          }
          assertEquals(0, Bytes.compareTo(singleKvs[j].getValue(), multiKvs[j]
                      .getValue()));
        }
      }
      System.out.println("DONE");
      table.close();
    } catch (Exception e) {
      System.err.println("Error : " + e);
    } finally {
    }
  } 
}
