/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.streams.impl;

import static org.ojai.DocumentConstants.ID_FIELD;
import static org.ojai.store.QueryCondition.Op.GREATER_OR_EQUAL;
import static org.ojai.store.QueryCondition.Op.LESS;

import java.io.IOException;
import java.util.List;
import java.util.ArrayList;

import org.ojai.Value;
import org.ojai.Value.Type;
import org.ojai.store.QueryCondition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.db.MapRDB;
import com.mapr.db.TabletInfo;
import com.mapr.db.TableSplitInternal;
import com.mapr.db.impl.ConditionImpl;
import com.mapr.db.impl.ConditionNode.RowkeyRange;
import com.mapr.db.impl.IdCodec;

public class MarlinSplitterCore {

  static final Logger LOG = LoggerFactory.getLogger(MarlinSplitterCore.class);
  private static final String SEMICOLON = ";";
  private static final String UTF8 = "UTF-8";
  private static final char COLON = ':';

  /* Checks to see if key is a normal message key, and if so, return the
   * topic infinity key.  If not, then returns null. No key checks are done here,
   * since marlin tables do key format checks in put path.
   */
  private static String GetTopicEndKey(String key) {
    if (key == null || key.length() == 0) {
      return null;
    }

    int topicEndIndex = key.indexOf(COLON);
    if (topicEndIndex < 0) {
      return null;
    }

    /* return p<feedid>t<tpoicname>; -- semicolon ; is after : in ascii */
    return (key.substring(0, topicEndIndex) + SEMICOLON);
  }

  public static List<TableSplitInternal> getMarlinSplits(String tablename, TabletInfo[] tablets) throws IOException {

    int i = 0;
    String nextStartKey = null;
    List<TableSplitInternal> tableSplitInts = new ArrayList<TableSplitInternal>();

    while (i < tablets.length) {
      TabletInfo tab = tablets[i];
      ConditionImpl cond = (ConditionImpl)tab.getCondition();
      List<RowkeyRange> rowKeys = cond.getRowkeyRanges();

      /* Get end key of the tablet. */
      if (nextStartKey == null) {
        Value nextTempKey = IdCodec.decode(rowKeys.get(0).getStartRow());
        if (nextTempKey.getType() == Type.STRING) {
          nextStartKey = nextTempKey.getString();
        } else {
          nextStartKey = "";
        }
      }
      Value tempEndKey = IdCodec.decode(rowKeys.get(rowKeys.size() - 1).getStopRow());
      String curEndKey = null;
      if (tempEndKey.getType() == Type.STRING) {
        curEndKey = tempEndKey.getString();
      } else {
        curEndKey = "";
      }
      String topicEndKey = GetTopicEndKey(curEndKey);

      if (topicEndKey == null) {
        /* Topic partition does not span across this tablet, so add the key range */
        QueryCondition splitCond = MapRDB.newCondition().and()
          .is(ID_FIELD, GREATER_OR_EQUAL, nextStartKey)
          .is(ID_FIELD, LESS, curEndKey)
          .close()
          .build();
        TableSplitInternal split = new TableSplitInternal(tablename, splitCond,
                                                          tab.getLocations(),
                                                          tab.getEstimatedSize());
        tableSplitInts.add(split);

        LOG.trace("Adding MarlinSplits with endKey [" + nextStartKey + ", " + curEndKey + ")");

        nextStartKey = null;  /* Pick up the next tablet start key */
      } else {
        /* A topic partition is going across this tablet, so find the tablet where
         * this topic partition ends. */
        int j = i+1;
        while (j < tablets.length) {
          TabletInfo nextTab = tablets[j];
          ConditionImpl nextCond = (ConditionImpl)nextTab.getCondition();
          List<RowkeyRange> nextRowKeys = nextCond.getRowkeyRanges();
          String nextEndKey = null;
          Value tempNextEndKey = IdCodec.decode(nextRowKeys.get(nextRowKeys.size() - 1).getStopRow());
          if (tempNextEndKey.getType() == Type.STRING) {
            nextEndKey = tempNextEndKey.getString();
          } else {
            nextEndKey = "";
          }

          if (topicEndKey.compareTo(nextEndKey) <= 0) {
            QueryCondition splitCond = MapRDB.newCondition().and()
              .is(ID_FIELD, GREATER_OR_EQUAL, nextStartKey)
              .is(ID_FIELD, LESS, topicEndKey)
              .close()
              .build();
            TableSplitInternal split = new TableSplitInternal(tablename, splitCond,
                                                              tab.getLocations(),
                                                              tab.getEstimatedSize());
            tableSplitInts.add(split);
            LOG.trace("Adding MarlinSplits with topicEndKey ["+ nextStartKey + ", " + topicEndKey + ")");

            nextStartKey = topicEndKey;
            i = j - 1;  /* Next tablet we need to check is this current tablet */
            break;
          }
          j++;
        }
        assert(j < tablets.length); /* the last tablet should have empty end key,
                                     * so shouldn't have topicEndKey != null */
      }
      i++;
    }

    return tableSplitInts;
  }
}


