/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.util;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import com.mapr.org.apache.hadoop.hbase.util.Bytes;

/**
 * Contains helper methods related to {@link ByteBuffer}.
 */
public class ByteBufs {

  private static final ByteOrder DB_BYTE_ORDER = ByteOrder.LITTLE_ENDIAN;

  public static ByteBuffer ensurePreferred(ByteBuffer in) {
    ByteBuffer ret = !in.isDirect() ? in : Bytes.copyToByteBuffer(in);
    return order(ret);
  }

  public static ByteBuffer wrap(byte[] array) {
    return order(ByteBuffer.wrap(array));
  }

  public static ByteBuffer wrap(byte[] array, int offset, int length) {
    return order(ByteBuffer.wrap(array, offset, length));
  }

  public static ByteBuffer allocate(int capacity) {
    return order(ByteBuffer.allocate(capacity));
  }

  public static ByteBuffer allocateDirect(int capacity) {
    return order(ByteBuffer.allocateDirect(capacity));
  }

  public static ByteBuffer allocatePreferred(int capacity) {
    return order(ByteBuffer.allocate(capacity));
  }
  
  public static ByteBuffer order(ByteBuffer original) {
    return original.order(DB_BYTE_ORDER);
  }

  public static ByteBuffer allocateReadWrite(ByteBuffer original) {
    if (!original.isReadOnly()) {
      return order(original);
    } else {
      return order(Bytes.copyToByteBuffer(original));
    }
  }
}
