package com.mapr.db.tools;

import com.mapr.db.Admin;
import com.mapr.db.TableDescriptor;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.util.GenericOptionsParser;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.ojai.Document;
import org.ojai.DocumentStream;
import org.ojai.FieldPath;
import org.ojai.json.Json;
import org.ojai.json.Events.Delegate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.db.MapRDB;
import com.mapr.db.Table;
import com.mapr.db.impl.MapRDBTableImplHelper;
import com.mapr.fs.MapRFileSystem;

import java.io.IOException;

public class ImportJson extends Configured implements Tool {
  private static final Logger logger = LoggerFactory.getLogger(ImportJson.class);

  private MapRFileSystem maprfs_;

  private Delegate eventDelegate_;
  private Path jsonFilePath_;
  private Path tablePath_;
  private FieldPath idField_;

  public ImportJson() {}

  public ImportJson(Path tablePath, Path jsonFilePath,
      FieldPath idField, Delegate eventDelegate) {
    tablePath_ = tablePath;
    jsonFilePath_ = jsonFilePath;
    idField_ = idField;
    eventDelegate_ = eventDelegate;
  }

  @Override
  public int run(String[] args) throws Exception {
    String[] otherArgs = new GenericOptionsParser(getConf(), args).getRemainingArgs();
    if (otherArgs == null || otherArgs.length != 2) {
      usage();
      throw new IllegalArgumentException(
          "ImportJson requires table name and json file path as argument."
          + " All other parameters must be specified as properties.");
    }

    tablePath_ = new Path(otherArgs[0]);
    jsonFilePath_ = new Path(otherArgs[1]);

    @SuppressWarnings("unchecked")
    Class<? extends Delegate> delegateClass = (Class<? extends Delegate>)
        getConf().getClass("import.record.reader.delegate", null);
    if (delegateClass != null) {
      eventDelegate_ = delegateClass.newInstance();
    }

    String id = getConf().get("import.id.fieldpath");
    if (id != null) {
      idField_ = FieldPath.parseFrom(id);
    }

    return run0();
  }

  public int run0() throws Exception {
    int code = 0;
    maprfs_ = MapRDBTableImplHelper.getMapRFileSystem(getConf());
    FileSystem importFileFS_ = FileSystem.get(jsonFilePath_.toUri(), getConf());
    int printStatusEveryXRecord = getConf().getInt("print.status.every.x.record", 10000);

    if (maprfs_.exists(tablePath_) &&
        !maprfs_.getTableProperties(tablePath_).getAttr().getJson()) {
      logger.error("Table path should to point to JSON table");
      return 1;
    }

    try (
         Table table = maprfs_.exists(tablePath_)
            ? MapRDB.getTable(tablePath_) : MapRDB.createTable(tablePath_.toUri().getPath());
         FSDataInputStream jsonInStream = importFileFS_.open(jsonFilePath_);
         DocumentStream stream = Json.newDocumentStream(jsonInStream, eventDelegate_);) {
      int recordCount = 0;
      for (Document record : stream) {
        if (idField_ != null) {
          table.insertOrReplace(record.getString(idField_), record);
        } else {
          table.insertOrReplace(record);
        }
        recordCount++;
        if (recordCount % printStatusEveryXRecord == 0) {
          logger.info("Imported {} records.", recordCount);
        }
      }
      logger.info("{} records imported.", recordCount);
      Cleanup();
    } catch (Exception e) {
      logger.error(e.getMessage(), e);
      throw e;
    }
    return code;
  }

  private void Cleanup() throws IOException {
    Admin maprAdmin = MapRDB.newAdmin();
    TableDescriptor dstTableDesc = maprAdmin.getTableDescriptor(tablePath_);
    if (dstTableDesc.isBulkLoad()) {
      dstTableDesc.setBulkLoad(false);
      maprAdmin.alterTable(dstTableDesc);
    }
  }

  private void usage() {
    System.out.println("Usage: ImportJson [options...] <table_name> <json_file>");
  }

  Path getJsonFilePath() {
    return jsonFilePath_;
  }

  void setJsonFilePath(Path jsonFilePath) {
    this.jsonFilePath_ = jsonFilePath;
  }

  FieldPath getIdField() {
    return idField_;
  }

  void setIdField(FieldPath idField) {
    this.idField_ = idField;
  }

  /**
   * Main entry point.
   *
   * @param args the command line parameters
   * @throws Exception when running the job fails
   */
  public static void main(String[] args) throws Exception {
    int ret = ToolRunner.run(new ImportJson(), args);
    System.exit(ret);
  }

}
