package com.mapr.db.tools;

import java.io.IOException;
import java.io.PrintStream;

import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.util.GenericOptionsParser;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;
import org.ojai.Document;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.db.MapRDB;
import com.mapr.db.Table;
import com.mapr.db.Table.TableOption;

/**
 * This tool is experimental and requires careful review before externalizing.
 */
public class ExportJson extends Configured implements Tool {
  private static final Logger logger = LoggerFactory.getLogger(ExportJson.class);

  private Path jsonFilePath_;
  private Path tablePath_;

  public ExportJson(Path tablePath, Path jsonFilePath) {
    tablePath_ = tablePath;
    jsonFilePath_ = jsonFilePath;
  }

  public ExportJson() { }

  @Override
  public int run(String[] args) throws Exception {
    String[] otherArgs = new GenericOptionsParser(getConf(), args).getRemainingArgs();
    if (otherArgs == null || otherArgs.length != 2) {
      throw new IllegalArgumentException("ExportJson requires table name and json file path as argument. All other parameters must be specified as properties.");
    }

    tablePath_ = new Path(otherArgs[0]);
    jsonFilePath_ = new Path(otherArgs[1]);

    return run0();
  }

  public int run0() throws Exception {
    int code = 0;
    int recordCount = 0;
    boolean overwrite = getConf().getBoolean("export.overwrite", false);
    int printStatusEveryXRecord = getConf().getInt("print.status.every.x.record", 10000);
    FileSystem exportFileFS = FileSystem.get(jsonFilePath_.toUri(), getConf());
    if (exportFileFS.exists(jsonFilePath_) && !overwrite) {
      throw new IOException("The output file '" + jsonFilePath_ + "' already exists.");
    }

    try (Table table = MapRDB.getTable(tablePath_);
         PrintStream out = new PrintStream(
             exportFileFS.create(jsonFilePath_, null, overwrite, 8192,
                 exportFileFS.getDefaultReplication(jsonFilePath_),
                 exportFileFS.getDefaultBlockSize(jsonFilePath_), null));) {
      table.setOption(TableOption.EXCLUDEID, false);
      for (Document record : table.find()) {
        out.println(record);
        recordCount++;
        if (recordCount % printStatusEveryXRecord == 0) {
          logger.info("Exported {} records.", recordCount);
        }
      }
    } catch (Exception e) {
      logger.error(e.getMessage(), e);
      throw e;
    }
    logger.info("{} records exported.", recordCount);

    return code;
  }

  /**
   * Main entry point.
   *
   * @param args the command line parameters
   * @throws Exception when running the job fails
   */
  public static void main(String[] args) throws Exception {
    int ret = ToolRunner.run(new ExportJson(), args);
    System.exit(ret);
  }

}
