package com.mapr.db.rowcol;

import java.nio.ByteBuffer;

/**
 * Timedescriptor: contains information about three types
 * 1.CreateTime 2.UpdateTime 3.DeleteTime
 *
 * Each stored time contains two parts - Time in milliseconds and uniqifier.
 * Uniqifier is optional and is stored when the same row gets multiple
 * updates within the same millisecond time. In most cases, uniqifier will be
 * zero. It will not be stored if it's zero.
 *
 * TimeDescriptor is stored as 1 mandatory byte.
 * There are two bits in timedesc for each type.
 * For Create, Update, and Delete time, there are 2 bits for each type that have the
 * following meanings:
 * 0x0 - Field is not valid
 * 0x1 - Field is valid and the time is the same as reftime (both ts + uniq)
 * 0x2 - Field is valid and the external timeinfo is in sizeByte following this
 *
 * SizeByte format is 2 bytes long and stored in little endian format.
 * It stores the information about external time for each of the three
 * timestamps listed above.
 * There are 5 bits for each time type that have the following meanings:
 * bit 0 - bit 2 : Sign and size of the diff
 * 0x0 - time is same as base ts
 * 0x1 - time is +ve diff of base rowtime and diff is 1 byte long
 * 0x2 - time is +ve diff of base rowtime and diff is 2 byte long
 * 0x3 - time is +ve diff of base rowtime and diff is 4 byte long
 * 0x4 - time is absolute and is  6 byte long
 * 0x5 - time is -ve diff of base rowtime and diff is 4 byte long
 * 0x6 - time is -ve diff of base rowtime and diff is 2 byte long
 * 0x7 - time is -ve diff of base rowtime and diff is 1 byte long
 *
 * bit 3-4 : size of uniq
 * 0x0 - uniq is 0
 * 0x1 - uniq is abs and 1 byte long
 * 0x2 - uniq is abs and 2 byte long
 * 0x3 - uniq is abs and 4 byte long
 *
 * There are two modes of storing the timestamp:
 * 1. StoreFullTimestamp = false. This is the default mode
 * when table replication is not enabled. In this mode we
 * don't store the full timestamp for a field. If the time for
 * a field is valid then we store a flag indicating that the time
 * is the same as updatetime for the row. Only update time is stored in
 * full. This is not a problem because we don't expect out of order updates
 * without table replication.
 *
 * 2. StoreFullTimestamp = true. This is true when a table is involved in
 * replication. In this mode we store the full timestamp for each entry with the
 * encoding shown above.
 *
 * At the time of reading the timestamp, if the row was created without
 * storefulltimestamp then it must not have the external ts and any valid
 * timestamp will be converted to the update timestamp of the row.
 * Whether a row is created with storefulltimestamp flag or not, the flag is
 * stored in the rootimedescriptor for the row.
 *
 * When a client sends an op, it will always send the op with storefulltimestamp
 * = false because it has updatetime and all relative times are the same as
 * updatetime.
 */

public class TimeDescriptor {

   enum TimeStampState {
    NOT_VALID,
    VALID_WITH_REFTIME,
    VALID_WITH_EXTERNAL_TIME;
  }
  enum TimeStampType{
    TIMESTAMP_TIME_CREATETIME,
    TIMESTAMP_TIME_UPDATETIME,
    TIMESTAMP_TIME_DELETETIME,
  }

  // Index of different types in the kvstore times array
  public static final int CreateTimeIndex = 0;
  public static final int UpdateTimeIndex = 1;
  public static final int DeleteTimeIndex = 2;


  public static final int CreateTimeShift = 0;
  public static final int CreateTimeMask = 0x3;
  public static final int UpdateTimeShift = 2;
  public static final int UpdateTimeMask = 0x3 << UpdateTimeShift;
  public static final int DeleteTimeShift = 4;
  public static final int DeleteTimeMask = 0x3 << DeleteTimeShift;

  /* used in external size info */
  static final short CreateTimeSizeShift = 0;
  static final short CreateTimeUniqSizeShift = 3;
  static final short UpdateTimeSizeShift = 5;
  static final short UpdateTimeUniqSizeShift = 8;
  static final short DeleteTimeSizeShift = 10;
  static final short DeleteTimeUniqSizeShift = 13;

  static final short CreateTimeSizeMask = 0x7 << CreateTimeSizeShift;
  static final short CreateTimeUniqSizeMask= 3 << CreateTimeUniqSizeShift;
  static final short UpdateTimeSizeMask = 0x7 << UpdateTimeSizeShift;
  static final short UpdateTimeUniqSizeMask= 3 << UpdateTimeUniqSizeShift;
  static final short DeleteTimeSizeMask = 0x7 << DeleteTimeSizeShift;
  static final short DeleteTimeUniqSizeMask= 3 << DeleteTimeUniqSizeShift;

  public static void reset(KeyValue kv) {
    kv.timeDescriptor = 0;
  }
  public static void setCreateTimeValid(KeyValue kv) {
    kv.timeDescriptor |= setCreateTimeValidHelper();
  }

  public static int setCreateTimeValidHelper() {
    return TimeStampState.VALID_WITH_REFTIME.ordinal() << CreateTimeShift;
  }

  public static boolean isCreateTimeValid(byte timeDescriptor) {
    return (((timeDescriptor & CreateTimeMask) >> CreateTimeShift) == 0) ? false : true;
  }
  public static boolean isCreateTimeValid(KeyValue kv) {
    return isCreateTimeValid(kv.timeDescriptor);
  }

  public static void setDeleteTimeValid(KeyValue kv) {
    kv.timeDescriptor |= setDeleteTimeValid();
  }

  public static int setDeleteTimeValid() {
    return (TimeStampState.VALID_WITH_REFTIME.ordinal() << DeleteTimeShift);
  }

  public static boolean isDeleteTimeValid(KeyValue kv) {
    return isDeleteTimeValid(kv.timeDescriptor);
  }
  public static boolean isDeleteTimeValid(byte timeDescriptor) {
    return (((timeDescriptor & DeleteTimeMask) >> DeleteTimeShift) == 0) ? false : true;
  }

  public static void setUpdateTimeValid(KeyValue kv) {
    kv.timeDescriptor |= setUpdateTimeValid();
  }

  public static int setUpdateTimeValid() {
    return (TimeStampState.VALID_WITH_REFTIME.ordinal() << UpdateTimeShift);
  }

  public static boolean isUpdateTimeValid(KeyValue kv) {
    return isUpdateTimeValid(kv.timeDescriptor);
  }

  public static boolean isUpdateTimeValid(byte timeDescriptor) {
    return (((timeDescriptor & UpdateTimeMask) >> UpdateTimeShift) == 0) ? false : true;
  }

  public static byte resetCreateTime(byte timeDescriptor) {
    return (byte)(timeDescriptor & (byte) 0xfc);
  }

  public static void resetCreateTime(KeyValue kv) {
    kv.timeDescriptor = resetCreateTime(kv.timeDescriptor);
  }

  public static void resetUpdateTime(KeyValue kv) {
    kv.timeDescriptor = resetUpdateTime(kv.timeDescriptor);
  }

  public static byte resetUpdateTime(byte timeDescriptor) {
    return (byte) (timeDescriptor & (byte) 0xf3);
  }

  public static byte resetDeleteTime(byte timeDescriptor) {
    return (byte) (timeDescriptor & (byte) 0xcf);
  }

  public static void  serialize(KeyValue kv, ByteWriter w, SerializationContext ctx) {
    /* none of the time should be external from the java client */
    assert(((kv.timeDescriptor & CreateTimeMask) >> CreateTimeShift) <= TimeStampState.VALID_WITH_REFTIME.ordinal());
    assert(((kv.timeDescriptor & UpdateTimeMask) >> UpdateTimeShift) <= TimeStampState.VALID_WITH_REFTIME.ordinal());
    assert(((kv.timeDescriptor & DeleteTimeMask) >> DeleteTimeShift) <= TimeStampState.VALID_WITH_REFTIME.ordinal());

    w.put(kv.timeDescriptor);
  }

  public static void  serialize(KeyValue kv, ByteWriter w) {
    TimeDescriptor.serialize(kv, w, null);
  }

  public static byte getTimeDescriptor(ByteBuffer r) {
    return r.get();
  }

  public static boolean isExternal(TimeStampState t, byte desc) {
    t = TimeStampState.values()[((desc & CreateTimeMask) >> CreateTimeShift)];
    switch (t) {
    case VALID_WITH_EXTERNAL_TIME:
      return true;
    case NOT_VALID:
    case VALID_WITH_REFTIME:
    default:
      return false;

    }
  }

  public static byte deserialize (KeyValue v, ByteBuffer r, SerializationContext ctx) {

    KeyValueWithTS kv = null;

    byte timeDescriptor = getTimeDescriptor(r);
    ctx.setTimeDescriptor(timeDescriptor);
    boolean decodeTimestamp = ctx.getDecodeTimestamp();

    if (decodeTimestamp) {
      kv = (KeyValueWithTS)v;
      kv.times = new TimeAndUniq[3];
      for (int i = 0; i < 3; ++i) {
        kv.times[i] = new TimeAndUniq();
      }
    }


    boolean createTimeExternal = false;
    boolean updateTimeExternal = false;
    boolean deleteTimeExternal = false;


    TimeStampState t =
        TimeStampState.values()[((timeDescriptor & CreateTimeMask) >> CreateTimeShift)];

    switch (t) {
    case NOT_VALID:
      break;
    case VALID_WITH_REFTIME:
    	if (decodeTimestamp) {
    	  kv.times[CreateTimeIndex].setTime(ctx.getBaseTime());
    	}
    	break;
    case VALID_WITH_EXTERNAL_TIME:
      createTimeExternal = true;
    }


    t = TimeStampState.values()[((timeDescriptor & UpdateTimeMask) >> UpdateTimeShift)];

    switch (t) {
    case NOT_VALID:
      break;
    case VALID_WITH_REFTIME:
      if (decodeTimestamp) {
        kv.times[UpdateTimeIndex].setTime(ctx.getBaseTime());
      }
      break;
    case VALID_WITH_EXTERNAL_TIME:
      updateTimeExternal = true;
    }


    t = TimeStampState.values()[((timeDescriptor & DeleteTimeMask) >> DeleteTimeShift)];
    switch (t) {
    case NOT_VALID:
      break;
    case VALID_WITH_REFTIME:
      if (decodeTimestamp) {
        kv.times[DeleteTimeIndex].setTime(ctx.getBaseTime());
      }
      break;
    case VALID_WITH_EXTERNAL_TIME:
      deleteTimeExternal = true;
    }

    // Reset the timedescriptor because client doesn't send
    // the timedescriptor with external timestamp in it
    if (TimeDescriptor.isCreateTimeValid(timeDescriptor)) {
      timeDescriptor = TimeDescriptor.resetCreateTime(timeDescriptor);
      timeDescriptor |= setCreateTimeValidHelper();
    }

    if (TimeDescriptor.isUpdateTimeValid(timeDescriptor)) {
      timeDescriptor = TimeDescriptor.resetUpdateTime(timeDescriptor);
      timeDescriptor |= TimeDescriptor.setUpdateTimeValid();
    }

    /*
     * reset the delete timestamp as we might need to put the read document
     * back into another table. At that time we don't want to carry the delete
     * flag. Delete flag will only be set at the top level document by insertOrReplace
     * or mutation APIs
     * Only ES decoding of the document may require to preserve the flag. That caller
     * must set preserveDeleteTime flag to keep the delete time.
     */
    if (ctx.preserveDeleteTime()) {
      if (TimeDescriptor.isDeleteTimeValid(timeDescriptor)) {
        timeDescriptor = TimeDescriptor.resetDeleteTime(timeDescriptor);
        timeDescriptor |= setDeleteTimeValid();
      }
    } else {
      timeDescriptor = TimeDescriptor.resetDeleteTime(timeDescriptor);
    }

    // In java client we don't need to have the real timestamp
    // so just read and consume it from the stream. No need to
    // make any sense out of it.
    if (createTimeExternal || updateTimeExternal || deleteTimeExternal) {
      short sizeByte;
      sizeByte = r.getShort();
      TimeAndUniq time = null;
      if (createTimeExternal) {
        if (decodeTimestamp) {
          time = kv.times[CreateTimeIndex];
        }
        readCreateTimeAndUniq(r, sizeByte, time, ctx);
      }
      if (updateTimeExternal) {
        if (decodeTimestamp) {
          time = kv.times[UpdateTimeIndex];
        }

        readUpdateTimeAndUniq(r, sizeByte, time, ctx);
      }
      if (deleteTimeExternal) {
        if (decodeTimestamp) {
          time = kv.times[DeleteTimeIndex];
        }

        readDeleteTimeAndUniq(r, sizeByte, time, ctx);
      }
    }

    if (v != null) {
      v.timeDescriptor = timeDescriptor;
    }

    return timeDescriptor;
  }

  static void readCreateTimeAndUniq(ByteBuffer r, short sizeInfo, TimeAndUniq time, SerializationContext ctx) {
    short timeSizeInfo =
      (short) ((sizeInfo & CreateTimeSizeMask) >> CreateTimeSizeShift);
    short uniqSizeInfo =
      (short) ((sizeInfo & CreateTimeUniqSizeMask) >> CreateTimeUniqSizeShift);
    readTimeAndUniq(r, timeSizeInfo, uniqSizeInfo, time, ctx);
  }

  static void readUpdateTimeAndUniq(ByteBuffer r, short sizeInfo, TimeAndUniq time, SerializationContext ctx) {
    short timeSizeInfo =
      (short) ((sizeInfo & UpdateTimeSizeMask) >> UpdateTimeSizeShift);
    short uniqSizeInfo =
      (short) ((sizeInfo & UpdateTimeUniqSizeMask) >> UpdateTimeUniqSizeShift);
    readTimeAndUniq(r, timeSizeInfo, uniqSizeInfo, time, ctx);
  }

  static void readDeleteTimeAndUniq(ByteBuffer r, short sizeInfo, TimeAndUniq time, SerializationContext ctx) {
    short timeSizeInfo =
      (short) ((sizeInfo & DeleteTimeSizeMask) >> DeleteTimeSizeShift);
    short uniqSizeInfo =
      (short) ((sizeInfo & DeleteTimeUniqSizeMask) >> DeleteTimeUniqSizeShift);
    readTimeAndUniq(r, timeSizeInfo, uniqSizeInfo, time, ctx);
  }

  static void readTimeAndUniq(ByteBuffer r, short timeSizeInfo, short
      uniqSizeInfo, TimeAndUniq time, SerializationContext ctx) {

    assert((timeSizeInfo >=0) && (timeSizeInfo <= 7));
    boolean decodeTS = ctx.getDecodeTimestamp();

    assert(time != null || !decodeTS);

    long diff = 0;
    switch (timeSizeInfo) {
    case 0: {
      if (decodeTS) {
        time.setTime(ctx.getBaseTime().time());
      }
      break;
    }

    case 1: {
      diff = r.get();
      if (decodeTS) {
        time.setTime(ctx.getBaseTime().time() + diff);
      }
      break;
    }
    case 2: {
      diff = r.getShort();
      if (decodeTS) {
        time.setTime(ctx.getBaseTime().time() + diff);
      }
      break;
    }
    case 3: {
      diff = r.getInt();
      if (decodeTS) {
        time.setTime(ctx.getBaseTime().time() + diff);
      }
      break;
    }
    case 4: {
      byte[] b = new byte[6];
      r.get(b);
      if (decodeTS) {
        long t = 0;
        for (int i = 0; i < b.length; ++i) {
          t |= ((long) (b[i] & 0xff)) << (8 * i);
        }
        time.setTime(t);
      }
      break;
    }
    case 5: {
      diff = r.getInt();
      if (decodeTS) {
        time.setTime(ctx.getBaseTime().time() - diff);
      }
      break;
    }
    case 6: {
      diff = r.getShort();
      if (decodeTS) {
        time.setTime(ctx.getBaseTime().time() - diff);
      }
      break;
    }
    case 7: {
      diff = r.get();
      if (decodeTS) {
        time.setTime(ctx.getBaseTime().time() - diff);
      }
      break;
    }
    }

    int uniq = 0;
    switch (uniqSizeInfo) {
    case 0: break;
    case 1: uniq = r.get(); break;
    case 2: uniq = r.getShort(); break;
    case 3: uniq = r.getInt(); break;
    }
    if (decodeTS) {
      time.setUniq(uniq);
    }
  }
  public static String toStringWithTimestamp(KeyValue v) {
    KeyValueWithTS kv = (KeyValueWithTS)v;
    StringBuilder b = new StringBuilder();
    b.append("[");
    for (int i = 0; i < 3; ++i) {
      TimeAndUniq t = kv.times[i];
      String s = String.format("%d.%d", t.time(), t.uniq());
      b.append(s);
      if (i != 2) {
        b.append(", ");
      }
    }
    b.append("]");
    return b.toString();
  }
}
