/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.rowcol;

import static com.mapr.db.impl.Constants.DUMMY_FIELDPATH_V;
import static com.mapr.db.impl.Constants.DUMMY_FIELD_V;
import static com.mapr.db.impl.Constants.FIELD_SEPARATOR;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeMap;

import org.ojai.FieldPath;
import org.ojai.FieldSegment;
import org.ojai.Value.Type;

import com.mapr.db.rowcol.ArrayIndexDescriptor.ArrayIndexType;
import com.mapr.db.rowcol.InsertContext.OpType;

/*
 * Class to pass the serialization flags to each level
 * during serialization / de-serialization
 */

public class SerializationContext extends BaseSerializationContext {
  class FamilyInfo implements SerializedFamilyInfo {
    KeyValue rootKV;
    int familyId;
    ByteWriter writer;
    FieldPath jsonPath;
    ByteBuffer cachedBuffer;

    // Optype for this element or any element in the parent of this element
    // This is used in mutation operation to decide the correct serialized
    // encoding for this column family.
    // For example if parent optype is merge - then root of this cf will not
    // have delete flag set. But if the optype is set, setorreplace, replace, inc
    // etc then this element needs to have delete flag valid
    OpType parentOpType;
    public OpType opType;
    public SerializationAction action;
    public KeyValue parentKV;


    public FamilyInfo() {}
    public FamilyInfo(int familyId, FieldPath jsonPath, ByteBuffer cachedBuffer)
    {
      this.familyId = familyId;
      this.jsonPath = jsonPath;
      this.cachedBuffer = cachedBuffer;
      if (this.cachedBuffer == null) {
        action = SerializationAction.NO_ACTION;
      } else {
        action = SerializationAction.SET;
      }
    }
    @Override
    public int getFamilyId() {
      return familyId;
    }

    @Override
    public SerializationAction getAction() {
      return action;
    }

    @Override
    public ByteBuffer getByteBuffer() {
      if (cachedBuffer != null) {
        return cachedBuffer;
      } else if (writer == null) {
        return null;
      }
      return writer.getByteBuffer();
    }
  }

  byte flags;
  DBDocumentImpl rec;
  Map <FieldPath, Integer> jsonPathMap;
  private KeyValue currentKV;
  private FamilyInfo[] familyInfoList;
  private int lastSearchIndex = -1;
  private boolean fullRecordOp = false;
  private boolean hasMutation = false;
  private boolean storeRowTS = false;
  private boolean decodeTimestamp = false;
  private boolean preserveDeleteTime = false;
  private TimeAndUniq baseTime;
  private boolean hasDeletes = false;
  private ArrayIndexType indexType;
  private int arrayIndex;
  private DBValueBuilder builder  = null;
  private FieldPath[] paths = null;
  private FieldPath currentPath = null;
  private ProjectionState currentProjState = null;
  private boolean shouldPrunePaths = false;


  public void setHasDeletes(boolean v) {
    hasDeletes = v;
  }

  public boolean hasDeletes() {
    return hasDeletes;
  }

  public void setPreserveDeleteTime(boolean v) {
    preserveDeleteTime = v;
  }

  public boolean preserveDeleteTime() {
    return preserveDeleteTime;
  }

  public void setDecodeTimestamp(boolean v) {
    decodeTimestamp = v;
  }
  public boolean getDecodeTimestamp() {
    return decodeTimestamp;
  }

  public boolean isAbsoluteIndex() {
    return (indexType == ArrayIndexType.ARRAY_INDEX_TYPE_ABSOLUTE) ? true : false;
  }

  public void setIndexType(ArrayIndexType type) {
    indexType = type;
  }

  public ArrayIndexType getIndexType() {
    return indexType;
  }

  public void setArrayIndex(int i) {
    arrayIndex = i;
  }

  public int getArrayIndex() {
    return arrayIndex;
  }


  void setupFamilyList(DBDocumentImpl r, Map<FieldPath, Integer> jsonPathMap) {
    assert(jsonPathMap.size() > 0);
    this.jsonPathMap = jsonPathMap;
    rec = r;
    familyInfoList = new FamilyInfo[jsonPathMap.size()];

    int i = 0;
    for (Entry<FieldPath, Integer> kv : jsonPathMap.entrySet()) {
      familyInfoList[i++] = setupFamilyInfo(kv.getKey(), kv.getValue());
    }
  }

  private FamilyInfo setupFamilyInfo(FieldPath path, int id) {
    FamilyInfo info = new FamilyInfo();
    KeyValue kv = null;
    KeyValue lastParentKV = rec;
    OpType opType = OpType.NONE;

    if (path.equals(FieldPath.EMPTY)) {
      kv = rec;
    } else {
      /* for non default family set the optype and rootkv */
      Iterator <FieldSegment> iter = path.iterator();
      FieldSegment field;
      DBDocumentImpl curRec = rec;
      while ((field = iter.next()) != null) {
        String key = field.getNameSegment().getName();

        KeyValue tmpkv = curRec.map.get(key);
        if (tmpkv == null) {
          break;
        }

        if (tmpkv.opType != OpType.NONE.ordinal()) {
          /* only one element in the tree can have non none optype */
          assert(opType == OpType.NONE);
          opType = OpType.valueOf(tmpkv.opType);
        }

        // if this is the last path then return the value at this key in map
        if (field.isLastPath()) {
          kv = tmpkv;
          break;
        }

        lastParentKV = tmpkv;
        if (tmpkv.getType() != Type.MAP) {
          kv = null;
          break;
        }

        curRec = (DBDocumentImpl) tmpkv;
      }
    }
    /*
     * search the record for the given path
     * and mark the kv to be part of column family
     */
    info.rootKV = kv;
    info.parentKV = lastParentKV;
    info.familyId = id;
    info.jsonPath = path;
    info.opType = opType;
    if (kv != null) {
      info.writer = new ByteWriter();
      kv.setRootOfColumnFamily(true);
      kv.setCFRootId(id);
    }
    return info;
  }

  public SerializedFamilyInfo[] getSerializedBuffers(Map<FieldPath, Integer> jsonPathMap,
                                                     Map<Integer, ByteBuffer> cachedBufferMap) {
    assert(jsonPathMap.size() > 0);
    assert(cachedBufferMap.size() > 0);
    TreeMap<Integer, FamilyInfo> famInfoMap = new TreeMap<Integer, FamilyInfo>();  //Ordered on famId

    for (Entry<FieldPath, Integer> kv : jsonPathMap.entrySet()) {
      famInfoMap.put(kv.getValue(), new FamilyInfo(kv.getValue(), kv.getKey(),
                          cachedBufferMap.get(kv.getValue())));
    }

    FamilyInfo[] famInfos = new FamilyInfo[famInfoMap.size()];
    int i = 0;
    for (Map.Entry<Integer, FamilyInfo> fi : famInfoMap.entrySet()) {
      famInfos[i++] = fi.getValue();
    }
    return famInfos;
  }

  public void serializeFamilies(DBDocumentImpl r, Map<FieldPath, Integer> jsonPathMap) {
    setupFamilyList(r, jsonPathMap);

    /*
     * Set the root level flags at the time of serialization
     * This is needed for the case where someone is just
     * putting the empty record. If we don't set it here
     * then the record would remains without any flags and is
     * like a no-op on the server
     */
    if (isFullRecordOp()) {
      TimeDescriptor.setCreateTimeValid(r);
      TimeDescriptor.setUpdateTimeValid(r);
      TimeDescriptor.setDeleteTimeValid(r);
    }

    for (int i = 0; i < familyInfoList.length; ++i) {
      FamilyInfo info = familyInfoList[i];

      if (info.rootKV == null) {
        /*
         * this column family is not there in the document
         * so set the correct action based on the optype
         */
        if (isFullRecordOp()) {
          info.action = SerializationAction.DELETE_FAMILY;
        } else {
          /*
           * if the last parent is not a map then we need to
           * delete this column family root.
           * For example there is operation - a.b = 10
           * and the column family is on a.b.c
           * In this case we need to delete root of CF
           */
          if (info.parentKV.getType() != Type.MAP) {
            info.action = SerializationAction.DELETE_FAMILY;
          } else if (info.opType == OpType.NONE) {
            /* this cf is not affected at all */
            info.action = SerializationAction.NO_ACTION;
          } else if (info.opType != OpType.MERGE) {
            info.action = SerializationAction.DELETE_FAMILY;
          } else {
            info.action = SerializationAction.NO_ACTION;
          }
        }
        continue;
      }

      // For full record operation the action for cf which is present
      // in the document is always set. For mutation set the action as
      // SET here. In case where a particular CF doesn't contain any
      // mutation operation -- it will be changed to NO_OP later.
      info.action = SerializationAction.SET;

      /*
       * For non default CF we need to add a dummy root
       */
      if (info.jsonPath.equals(FieldPath.EMPTY)) {
        rec.serializeToRowCol(info.writer, this);
      } else {
        DBDocumentImpl dummyRoot = new DBDocumentImpl();
        /*
         * based on the parent op -- set the flags on the dummy root node
         * For full record operation root of each column family needs to
         * have the create, update and delete time valid
         */
        if (isFullRecordOp()) {
          TimeDescriptor.setCreateTimeValid(dummyRoot);
          TimeDescriptor.setUpdateTimeValid(dummyRoot);
          TimeDescriptor.setDeleteTimeValid(dummyRoot);
        } else {
          /*
           * if the optype is NONE then above the CF there is no
           * mutation so just copy the flag from the rootKV into
           * the dummy root.
           * If the optype is merge then set the create and update flag
           * If the optype is anything else then set the create, update
           * and delete flags.
           */
          switch (info.opType) {
          case NONE:
            if (TimeDescriptor.isCreateTimeValid(info.rootKV)) {
              TimeDescriptor.setCreateTimeValid(dummyRoot);
            }
            if (TimeDescriptor.isUpdateTimeValid(info.rootKV)) {
              TimeDescriptor.setUpdateTimeValid(dummyRoot);
            }
            if (TimeDescriptor.isDeleteTimeValid(info.rootKV)) {
              TimeDescriptor.setDeleteTimeValid(dummyRoot);
            }
            break;
          case DELETE:
            assert(TimeDescriptor.isDeleteTimeValid(info.rootKV));
            assert(TimeDescriptor.isCreateTimeValid((info.rootKV)) == false);
            assert(TimeDescriptor.isUpdateTimeValid(info.rootKV));
            TimeDescriptor.setUpdateTimeValid(dummyRoot);
            TimeDescriptor.setDeleteTimeValid(dummyRoot);
            break;
          case MERGE:
          case APPEND:
          case INCREMENT:
            assert(TimeDescriptor.isDeleteTimeValid(info.rootKV) == false);
            assert(TimeDescriptor.isCreateTimeValid((info.rootKV)));
            assert(TimeDescriptor.isUpdateTimeValid(info.rootKV));
            TimeDescriptor.setCreateTimeValid(dummyRoot);
            TimeDescriptor.setUpdateTimeValid(dummyRoot);
            break;
          case SET:
          case SET_OR_REPLACE:
            assert(TimeDescriptor.isCreateTimeValid((info.rootKV)));
            assert(TimeDescriptor.isUpdateTimeValid(info.rootKV));
            TimeDescriptor.setCreateTimeValid(dummyRoot);
            TimeDescriptor.setUpdateTimeValid(dummyRoot);
            TimeDescriptor.setDeleteTimeValid(dummyRoot);
          }
        }

        // Now insert the rootKV at path v in the dummy root
        // we already have the timedescriptor flags for the dummy root
        // so just put in the map.
        currentKV = info.rootKV.shallowCopy();
        currentKV.timeDescriptor = info.rootKV.timeDescriptor;
        currentKV.opType = info.rootKV.opType;
        currentKV.setRootOfColumnFamily(false);

        dummyRoot.map.put(DUMMY_FIELD_V, currentKV);
        hasMutation = false;
        dummyRoot.serializeToRowCol(info.writer, this);
      }

      /*
       * if this is mutation encoding and this CF doesn't have
       * any mutation in it then don't need to send anything for this
       * CF.
       * For example, mutation operation is inc a.b.c.d by 1.
       * suppose a.b is a column family then we just need to
       * send c.d in that cf no operation is needed for root CF.
       * Parent optype should also not have any operation for us to
       * make no action for the cf.
       */
      if (!isFullRecordOp() && !hasMutation &&
          (info.opType == OpType.NONE)) {
        info.action = SerializationAction.NO_ACTION;
        info.writer = null;
      }
    }
  }


  /*
   * Returns the serialized byte buffers for each column family
   * which are passed in the decode function.
   */
  public SerializedFamilyInfo[] getSerializedBuffers() {
    return familyInfoList;
  }

  /* returns true if the given kv is root of a cf in the current context */
  boolean isFamilyRoot(KeyValue kv) {
    if ((familyInfoList == null) ||
        (currentKV == kv)) {
      return false;
    }
    // search in the list of familyInfo to see if this kv is present in list
    for (int i = 1; i <= familyInfoList.length; ++i) {
      int index = (i + lastSearchIndex ) % familyInfoList.length;
      if (familyInfoList[index].rootKV == kv) {
        lastSearchIndex = i;
        return true;
      }
    }
    return false;
  }


  // TODO : If we add more flags then start using bits for flags
  public void setHasArrayProjection() {
    flags = 1;
  }

  public boolean hasArrayProjection() {
    return flags == 0 ? false : true;
  }

  public void setFullRecordOp(boolean v) {
    fullRecordOp = v;
  }

  public boolean isFullRecordOp() {
    return fullRecordOp;
  }

  public void setHasMutation(boolean v) {
    hasMutation = v;
  }

  public boolean hasMutation() {
    return hasMutation;
  }

  // Helper API to check if current path is part of given CF path or is owned
  // in a parent/encompassing family path. For if cf1=a.b and cf2=a.b.c, then
  // a.b is owned by cf1 and not cf2.
  // TODO BB - check if there is existing api to do this or move it to util api.
  private boolean pathIsOwner(String pathSoFar, FieldPath cfPath,
                              Map <FieldPath, Integer> jsonPathMap) {
    FieldPath curPath = FieldPath.parseFrom(pathSoFar);

    Integer idx = jsonPathMap.get(curPath);

    if ((idx != null) && (!cfPath.equals(curPath))) {
      return false;
    }

    return true;
  }

  /*
   * Function to convert a set of bytebuffers into a single logical view of record
   * The expectation is that the map should contain the families with default family
   * first, if it exists and then followed by the elements in the ascending order of
   * family. Such that the family containing another family comes first in the iterator
   * For example CF1 - a.b and CF2 is a.b.c.d then CF1 should come before CF2.
   */

  DBDocumentImpl decode(Map <Integer, ByteBuffer> map,
                        Map <FieldPath, Integer> jsonPathMap,
                        Map <Integer, String> idToCFNameMap,
                        boolean restoreOrder,
                        DBDocumentImpl rec) {

    if (decodeTimestamp) {
      builder = DBValueWithTSBuilder.keyValueWithTSBuilder;
    } else {
      builder = DBValueBuilderImpl.KeyValueBuilder;
    }

    Iterator<Entry<FieldPath, Integer>> iter = jsonPathMap.entrySet().iterator();
    ProjectionState rootProjState = null;

    if (paths != null && paths.length > 0) {
      rootProjState = new ProjectionState(0, paths.length - 1, ProjectionState.State.ANCESTOR);
    } else {
      rootProjState = DESCENDENT;
    }

    for (int i = 0; i < jsonPathMap.size(); ++i) {
      Entry<FieldPath, Integer> e = iter.next();
      FieldPath path = e.getKey();
      int cfId = e.getValue();
      ByteBuffer buf = map.get(cfId);
      /* for default cf deserialize the record if the buffer is not null */
      if (i == 0) {
        if (buf != null) {
          if (shouldPrunePaths()) {
            setCurrentPath(FieldPath.EMPTY);
            setCurrentProjectionState(rootProjState);
          }
          rec.deserializeFromRowCol(buf, this);
        }
      } else {
        ProjectionState cfProjState = getProjectionState(rootProjState, path);
        if (buf != null && !cfProjState.isUnrelated()) {
          DBDocumentImpl r = new DBDocumentImpl();
          if (shouldPrunePaths()) {
            setCurrentPath(path);
            setCurrentProjectionState(cfProjState);
          }
          r.deserializeFromRowCol(buf, this);
          KeyValue kv = r.getKeyValue(DUMMY_FIELDPATH_V);

          if (kv == null) {
            continue;
          }

          kv.setRecursiveNonDefaultColumnFamily(true, cfId);

          rec.set(path, kv);

          // Add info that it is part of a non-default col fam for each segment
          // of the path.
          Iterator<FieldSegment> fpiter = path.iterator();
          String pathSoFar = null;
          while (fpiter.hasNext()) {
            FieldSegment field = fpiter.next();
            String thisKey = field.getNameSegment().getName();
            if (pathSoFar == null) {
              pathSoFar = thisKey;
            } else {
              pathSoFar = pathSoFar + FIELD_SEPARATOR + thisKey;
            }
            boolean ret = pathIsOwner(pathSoFar, path, jsonPathMap);
            if (ret) {
              KeyValue rkv = rec.getKeyValue(pathSoFar);
              rkv.setPartOfNonDefaultColumnFamily(true);
              rkv.setCFRootId(cfId);
            }
          }
        }
      }
    }

    if (restoreOrder)
      return rec.restoreOrder(idToCFNameMap);
    else
      return rec;
  }

  public void setStoreRowTS(boolean v) {
    storeRowTS = v;
  }

  public boolean storeRowTS() {
    return storeRowTS;
  }

  public void setBaseTime(long time, int uniq) {
    baseTime = new TimeAndUniq(time, uniq);
  }

  public void setBaseTime(TimeAndUniq t) {
    baseTime = t;
  }

  public TimeAndUniq getBaseTime() {
    return baseTime;
  }

  public KeyValue newKeyValue() {
    return builder.newDBValue();
  }

  public boolean shouldPrunePaths() {
    return shouldPrunePaths;
  }

  public void setProjectedPaths(String[] ps) {

    if (ps == null || (ps.length == 0)) {
      paths = null;
      shouldPrunePaths = false;
      return;
    }

    shouldPrunePaths = true;
    Set<String> set = new HashSet<String>();

    // Remove all duplicate paths
    for (int i = 0; i < ps.length; ++i) {
      set.add(ps[i]);
    }

    String[] p = new String[set.size()];
    set.toArray(p);

    // sort lexicographically
    Arrays.sort(p);

    // convert them into FieldPath
    paths = new FieldPath[p.length];
    for (int i = 0 ; i < p.length; ++i) {
      paths[i] = FieldPath.parseFrom(p[i]);
    }
  }

  /*
   * ProjectionState class has been created in order to skip the parts of
   * the Document that were only added due to condition paths, not
   * projection paths.
   * Since we perform a DFS walk of all the Document ByteBuffers, each
   * key will have its ProjectionState.
   * UNRELATED state implies that current path is not present in projected
   * paths, hence all its children should be skipped without any checks.
   * DESCENDENT state implies that current path or its parents are present
   * in the projected list of paths, hence all its children should be added
   * without any checks.
   * ANCESTOR state implies that current path is the ancestor of one or
   * more of the projected paths, hence we need to perform the
   * getProjectionState() check for its children. The lowIndex, highIndex
   * fields of this class will be used to look for relevant projected
   * paths in the paths array. This will avoid unnecessary and expensive
   * checks.
   */
  public static class ProjectionState {

    public enum State {
      UNRELATED,
      ANCESTOR,
      DESCENDENT
    };

    public int lowIndex;
    public int highIndex;
    State      state;

    public ProjectionState(State st) {
      state = st;
      // ancestor requires lowIndex,highIndex to be set
      assert(state != State.ANCESTOR);
    }

    public ProjectionState(int low, int high, State st) {
      lowIndex = low;
      highIndex = high;
      state = st;
    }

    public boolean isAncestor() {
      return state == State.ANCESTOR;
    }

    public boolean isDescendent() {
      return state == State.DESCENDENT;
    }

    public boolean isUnrelated() {
      return state == State.UNRELATED;
    }

    public String toString() {
      return "Low Index: " + lowIndex + ", High Index: " + highIndex + " State: " + state;
    }
  }

  public static ProjectionState UNRELATED = new ProjectionState(ProjectionState.State.UNRELATED);
  public static ProjectionState DESCENDENT = new ProjectionState(ProjectionState.State.DESCENDENT);

  public ProjectionState getProjectionState(ProjectionState parentProjState, FieldPath path) {

    if (parentProjState.isUnrelated() || parentProjState.isDescendent()) {
      return parentProjState;
    }

    int low = parentProjState.lowIndex;
    int high = parentProjState.highIndex;

    int i = low;

    while ((i <= high) && !path.isAtOrAbove(paths[i])) {
      if (path.isAtOrBelow(paths[i])) {
        return DESCENDENT;
      }
      i ++;
    }

    if (i == high + 1) {
      return UNRELATED;
    }

    low = i;
    i ++;
    while ((i <= high) && (path.isAtOrAbove(paths[i]))) {
      i ++;
    }

    high = i - 1;

    if (low < high) {
      return new ProjectionState(low, high, ProjectionState.State.ANCESTOR);
    }

    if (path.compareTo(paths[low]) == 0) {
      return DESCENDENT;
    }

    return new ProjectionState(low, high, ProjectionState.State.ANCESTOR);
  }

  public void setCurrentPath(FieldPath p) {
    currentPath = p;
  }

  public FieldPath currentPath() {
    return currentPath;
  }

  public void setCurrentProjectionState(ProjectionState s) {
    currentProjState = s;
  }

  public ProjectionState currentProjectionState() {
    return currentProjState;
  }
}
