package com.mapr.db.rowcol;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.ojai.Document;
import org.ojai.FieldPath;

import com.google.common.collect.BiMap;
import com.mapr.db.impl.IdCodec;
import com.mapr.fs.proto.Dbserver.MCFRowColOutputFormat;
import com.mapr.fs.proto.Dbserver.MCFRowColOutputFormat.CFmetadata;

public class SequenceFileRowColCodec {
  public static int SEQUENCEFILE_CODEC_VERSION = 1;

  public static ByteBuffer encode(Document document) {
    if (document instanceof DBDocumentImpl) {
      return getMCFEncoded((DBDocumentImpl)document);
    } else {
      return RowcolCodec.encode(document);
    }
  }

  /*
   * This function should create a serialized bytebuffer from MCF data. It can
   * be used to dump the MCF data as a single buffer into sequence file. Format
   * of the buffer : byte indicating if there is an MCF If mCF then put cfname
   * and buffer length in sequence. Next copy the buffer content.
   */
  private static ByteBuffer getMCFEncoded(DBDocumentImpl document) {
    if (!document.getNeedDOMStruct()) {
      //if no DOM needed, we need to create bytebuffer for whole document
      //and set the buffers in document object
      ByteBuffer buf =  RowcolCodec.encode(document);
      Map<FieldPath, Integer> pathMap = new HashMap<FieldPath, Integer>();
      Map<Integer, ByteBuffer> buffersToCache = new HashMap<Integer, ByteBuffer>();
      Map<Integer, String> cfIdMap = new HashMap<Integer, String>();
      pathMap.put(FieldPath.parseFrom(""), 1);
      buffersToCache.put(1, buf);
      cfIdMap.put(1, "default");

      document.setSerializedJson(buffersToCache, pathMap, cfIdMap, document.getId(),
            false /* excludeId */,
            false  /* insertionOrder */,
            false /*decode timestamp */, false /*preserveDeleteTs*/, null);
    }

    Map<FieldPath, Integer> jsonPathMap = document.getJsonPathMap();
    Map<Integer, ByteBuffer> cachedBuffer = document.getCachedBuffers();
    MCFRowColOutputFormat.Builder header = MCFRowColOutputFormat.newBuilder();

    header.setVersion(SEQUENCEFILE_CODEC_VERSION);
    ByteBuffer encodedId = IdCodec.encode(document.getId());
    int rowKeyLength = encodedId.remaining();
    header.setRowkeysize(rowKeyLength);


    for (Entry<FieldPath, Integer> kv : jsonPathMap.entrySet()) {
      Integer kvInt = kv.getValue();
      ByteBuffer bBuf = cachedBuffer.get(kvInt);
      if (bBuf == null) {
        continue;
      }
      CFmetadata.Builder cfMeta = CFmetadata.newBuilder();
      cfMeta.setJsonPath(kv.getKey().asPathString());
      cfMeta.setCfId(kvInt);

      cfMeta.setBufferLength(bBuf.remaining());
      cfMeta.build();
      header.addCfDescriptor(cfMeta);
    }

    MCFRowColOutputFormat mcfHeader = header.build();

    int mcfEncodingSize = 0;
    for (Entry<Integer, ByteBuffer> kv : cachedBuffer.entrySet()) {
      mcfEncodingSize += kv.getValue().remaining();
    }

    byte[] mcfHeaderArray = mcfHeader.toByteArray();
    mcfEncodingSize += (Integer.SIZE / Byte.SIZE) + mcfHeaderArray.length;
    mcfEncodingSize += rowKeyLength;

    /* write header size first followed by the protobuf as byte array */
    ByteBuffer encodedBuffer = ByteBuffer.allocate(mcfEncodingSize);
    encodedBuffer.order(ByteOrder.LITTLE_ENDIAN);
    encodedBuffer.putInt(mcfHeaderArray.length);
    encodedBuffer.put(mcfHeaderArray);
    encodedBuffer.put(encodedId);
    for (Entry<FieldPath, Integer> kv : jsonPathMap.entrySet()) {
      ByteBuffer buff = cachedBuffer.get(kv.getValue());
      /* if buffer for the current MCF is null, nothing to write */
      if (buff == null) {
        continue;
      }
      ByteBuffer newBuffer = buff.duplicate();
      newBuffer.position(buff.position());
      newBuffer.limit(buff.limit());
      encodedBuffer.put(newBuffer);
    }

    encodedBuffer.rewind();
    return encodedBuffer;

  }

  public static Document decode(ByteBuffer buffer) throws IOException {
    return decode(buffer, null);
  }

  public static Document decode(ByteBuffer buffer, BiMap<FieldPath, Integer> destMap) throws IOException {

    DBDocumentImpl document = new DBDocumentImpl();
    Map<Integer, ByteBuffer> cachedBuffer = new HashMap<Integer, ByteBuffer>();
    Map<FieldPath, Integer> jsonPathMap = new HashMap<FieldPath, Integer>();
    Map<Integer, String> cfIdToName = new HashMap<Integer, String>();


    /* length of header */
    int headerSize = buffer.getInt();

    byte[] header = new byte[headerSize];
    buffer.get(header);

    MCFRowColOutputFormat mcfMetadata = MCFRowColOutputFormat.parseFrom(header);

    /* no of cfs */
    int numCFs = mcfMetadata.getCfDescriptorCount();

    /* rowkey length */
    int bufferPosition = buffer.position();

    int rowkeyLength = mcfMetadata.getRowkeysize();
    /* get rowkey */

    ByteBuffer rowkey = buffer.duplicate();
    rowkey.position(bufferPosition);
    bufferPosition += rowkeyLength;
    rowkey.limit(bufferPosition);


    int i = 0;

    while (i < numCFs) {
      CFmetadata metaData = mcfMetadata.getCfDescriptor(i);
      /* cfid */
      String jsonPath = metaData.getJsonPath();
      FieldPath fPath = FieldPath.parseFrom(jsonPath);
      Integer cfId ;
      if (destMap != null) {
        if (destMap.containsKey(fPath)) {
          cfId = destMap.get(fPath);
        } else {
          throw new IOException("CF Mapping not found for path "+fPath.toString()+" in destination table");
        }
      } else {
        cfId = metaData.getCfId();
      }


      int bufferLength = metaData.getBufferLength();
      String cfName;
      if (jsonPath.length() == 0) {
        cfName = "default";
      } else {
        cfName = "CF" + i;
      }

      jsonPathMap.put(fPath, cfId);
      cfIdToName.put(cfId, cfName);

      ByteBuffer thisCFData = buffer.duplicate();
      thisCFData.order(ByteOrder.LITTLE_ENDIAN);
      thisCFData.position(bufferPosition);
      bufferPosition += bufferLength;
      thisCFData.limit(bufferPosition);
      cachedBuffer.put(cfId, thisCFData);
      i++;
    }

    document.setSerializedJson(cachedBuffer, jsonPathMap, cfIdToName,
          IdCodec.decode(rowkey),
          false /* excludeId */,
          false /*insertionOrder */,
          false /* decode timestamp */, false /*preserveDeleteFlags*/,
          null);

    return document;
  }

}
